/**
 * GraphApplet.java
 *
 * Created on Feb 3, 2007
 * 
 * Copyright (c) 2007 Eureka Team
 */
package components;

import java.awt.Color;
import java.awt.image.BufferedImage;
import java.io.IOException;
import java.net.MalformedURLException;
import java.net.URL;

import javax.imageio.ImageIO;
import javax.swing.JApplet;
import javax.swing.UIManager;

/**
 * Cette applet represente un graphe / image / ensemble dans <b>R^2</b>.
 * Gere aussi la recuperation des paramtres et le renvoi des resultats.
 *
 * @author Nicolae
 */
public abstract class GraphApplet extends JApplet {

	private GraphPanel graph = null;
	
	/**
	 * Constructeur standard.
	 */
	public GraphApplet() {
		super();
	}

	/**
	 * This method initializes this
	 */
	public void init() {
		try {
			UIManager.setLookAndFeel(UIManager.getSystemLookAndFeelClassName());
		} catch (Exception e) {
			e.printStackTrace();
		}		

        graphInit();        
        this.setContentPane(getGraph());  // Generated
	}

	/**
	 * This method initializes graphPanel	
	 * 	
	 * @return components.GraphPanel	
	 */
	protected GraphPanel getGraph() {
		if (graph == null) {
			graph = new GraphPanel();
		}
		return graph;
	}

	/**
	 * Initialisation de <code>graph</code>.
	 */
	protected abstract void graphInit();

	/**
	 * Renvoie un entier donne en parametre a l'applet.
	 * @param name le nom du parametre.
	 *
	 * @return un entier donne en parametre a l'applet, 
	 *         <code>null</code> sinon.
	 */
	protected Integer getIntParam(String name) {
		String par = getParameter(name);
		if(par == null)
			return null;

		Integer i;
		try {
			i = new Integer(par);
		} catch (NumberFormatException e) {
			System.out.println("Parameter \"" + name + 
					"\" malformed. Integer expected, found : " + par);
			return null;
		}
		
		return i;
	}

	/**
	 * Renvoie un reel donne en parametre a l'applet.
	 * @param name le nom du parametre.
	 *
	 * @return un reel donne en parametre a l'applet, 
	 *         <code>null</code> sinon.
	 */
	protected Double getRealParam(String name) {
		String par = getParameter(name);
		if(par == null)
			return null;

		Double d;
		try {
			d = new Double(par);
		} catch (NumberFormatException e) {
			System.out.println("Parameter \"" + name + 
					"\" malformed. Real expected, found : " + par);
			return null;
		}
		
		return d;
	}

	/**
	 * Renvoie une couleur donnee en parametre a l'applet.
	 * @param name le nom du parametre.
	 *
	 * @return une couleur donnee en parametre a l'applet, 
	 *         <code>null</code> sinon.
	 */
	protected Color getColorParam(String name) {
		String par = getParameter(name);
		if(par == null)
			return null;

		Color c;
		try {
			par = par.trim().toUpperCase();
			int pc = Integer.valueOf(par, 16).intValue();
			c = new Color((pc >> 16) & 0xFF, (pc >> 8) & 0xFF, pc & 0xFF);
		} catch (NumberFormatException e) {
			System.out.println("Parameter \"" + name + 
					"\" malformed. Hex integer expected, found : " + par);
			return null;
		}
		
		return c;
	}

	/**
	 * Renvoie un boolean donne en parametre a l'applet.
	 * @param name le nom du parametre.
	 *
	 * @return un boolean donne en parametre a l'applet, 
	 *         <code>null</code> sinon.
	 */
	protected Boolean getBoolParam(String name) {
		String par = getParameter(name);
		if(par == null)
			return null;

		Boolean b;
		try {
			b = new Boolean(par);
		} catch (NumberFormatException e) {
			System.out.println("Parameter \"" + name + 
					"\" malformed. \"true\" / \"false\" expected, found : " + par);
			return null;
		}
		
		return b;
	}

	/**
	 * Renvoie une couleur donnee en parametre a l'applet.
	 * @param name le nom du parametre.
	 *
	 * @return une couleur donnee en parametre a l'applet, 
	 *         <code>null</code> sinon.
	 */
	protected BufferedImage getImageParam(String name) {
		String par = getParameter(name);
		if(par == null)
			return null;

		BufferedImage img = null;
		try {
				URL url = new URL(par);
				img = ImageIO.read(url);
		} catch (MalformedURLException e) {
			System.out.println("Parameter \"" + name + 
					"\" malformed. URL expected, found : " + par);
			return null;
		} catch (IOException e) {
			System.out.println("Image \"" + par + 
					"\" not founded or IO error !");
			return null;
		}
		
		return img;
	}
}
