/*
 * Print bit values
 */
#include <limits.h>
/** \cond LLONG_MAX and LLONG_LIN might not be defined. */
#ifndef LLONG_MAX
#  define LLONG_MAX    9223372036854775807LL
#endif
#ifndef LLONG_MIN
#  define LLONG_MIN    (-LLONG_MAX - 1LL)
# ifndef ULLONG_MAX
#  define ULLONG_MAX   (LLONG_MAX * 2ULL + 1)
#endif
#endif

#include <unistd.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <err.h>
#include <errno.h>
#include <stdint.h>
#include <stdbool.h>

/*
 * String conversion with strtoll(..., 0) semantics
 */
static bool atou64(const char *str, uint64_t *val)
{
	char *endptr;

	errno = 0;				/* strtol() manpage */
	*val  = strtoull(str, &endptr, 0);
	if ((errno == ERANGE && *val == ULLONG_MAX) ||
	    (errno != 0 && *val == 0)	||	/* other error */
	    endptr == str		||	/* no digits */
	    *endptr != '\0')			/* junk at end */
		return false;
	return true;
}

/* Base-2 logarithm of val > 0. */
static uint8_t lb(uint64_t val)
{
	uint8_t r = 0;

	while (val >>= 1)
		r++;
	return r;
}

/* Return minimum number of bits needed to represent @val */
static uint8_t num_bits(uint64_t val)
{
	return lb(val) + 1;
}

/* Analogously */
static uint8_t num_nybbles(uint64_t val)
{
	uint8_t nybbles = 1;

	while (val >>= 4)
		nybbles++;

	return nybbles;
}

/* Return bitmask of up to 64 bits */
static uint64_t get_mask(uint64_t val)
{
	return val == 0 ? 0 : ~0ULL >> (64 - num_bits(val));
}

void print_decimal(uint64_t val)
{
	uint8_t n = num_nybbles(val);

	printf("%*llu\n", n * 4 + n - 1, (unsigned long long)val);
}

void print_hexadecimal(uint64_t val)
{
	uint8_t tmp, i, n = num_nybbles(val);

	for (i = n; i--; ) {
		tmp = (val >> (i * 4)) & 0xf;
		printf("%4X%s", tmp, i ? " " : "");
	}
	printf("\n");

}

void print_binary(uint64_t val)
{
	uint8_t tmp, i, j, n = num_nybbles(val);

	for (i = n; i--; ) {
		for (j = 4, tmp = val >> (i * j); j--; )
			printf("%u", (tmp >> j) & 0x1);
		if (n)
			printf(" ");
	}
	printf("\n");
}

int main(int ac, char **av)
{
	uint64_t val, mask;

	while (--ac) {
		if (!atou64(*++av, &val)) {
			warnx("can not convert '%s' to u64", *av);
			continue;
		}
		printf("val  = %-10llu = ", (unsigned long long)val);
		print_binary(val);

		mask = get_mask(val);
		printf("mask = %-10llu = ", (unsigned long long)mask);
		print_binary(mask);
		printf("\n");
	}

	return EXIT_SUCCESS;
}
