/*
 * Copyright 2013 Canonical Ltd.
 *
 * This file is part of powerd.
 *
 * powerd is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; version 3.
 *
 * powerd is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#include <stdio.h>
#include <string.h>
#include <glib.h>
#include <glib-object.h>
#include <gio/gio.h>

#include "powerd-cli.h"

#define ARRAY_SIZE(a) (sizeof(a) / sizeof((a)[0]))

static struct client_test_data {
    GDBusProxy *proxy;
    int state;
    int ack_delay;
    guint ack_timeout_id;
    powerd_cookie_t cookie;
} test_data;

enum ack_status {
    NOT_ACKED,
    ACK_SUCCESS,
    ACK_FAILURE,
    ACK_TIMEOUT,
} ack_status;;

static GMainLoop *main_loop;

static gboolean run_next_test_handler(gpointer unused);

static void reset_test_state(void)
{
    ack_status = NOT_ACKED;
}

static void run_next_test(void)
{
    g_timeout_add(0, run_next_test_handler, NULL);
}

static gboolean register_client(GDBusProxy *proxy)
{
    GVariant *ret;
    GError *error = NULL;

    ret = g_dbus_proxy_call_sync(proxy, "registerClient",
                                 g_variant_new("(s)", "powerd-cli"),
                                 G_DBUS_CALL_FLAGS_NONE, -1, NULL, &error);
    if (!ret) {
        cli_warn("registerClient failed: %s", error->message);
        g_error_free(error);
        return FALSE;
    }

    g_variant_unref(ret);
    return TRUE;
}

static gboolean unregister_client(GDBusProxy *proxy)
{
    GVariant *ret;
    GError *error = NULL;

    ret = g_dbus_proxy_call_sync(proxy, "unregisterClient",
                                 g_variant_new("(s)", "powerd-cli"),
                                 G_DBUS_CALL_FLAGS_NONE, -1, NULL, &error);
    if (!ret) {
        cli_warn("unregisterClient failed: %s", error->message);
        g_error_free(error);
        return FALSE;
    }

    g_variant_unref(ret);
    return TRUE;
}

static gboolean ack_state_change(GDBusProxy *proxy, int state)
{
    GVariant *ret;
    GError *error = NULL;

    ret = g_dbus_proxy_call_sync(proxy, "ackStateChange",
                                 g_variant_new("(i)", state),
                                 G_DBUS_CALL_FLAGS_NONE, -1, NULL, &error);
    if (!ret) {
        cli_warn("ackStateChange failed: %s", error->message);
        g_error_free(error);
        ack_status = ACK_FAILURE;
        return FALSE;
    }

    g_variant_unref(ret);
    ack_status = ACK_SUCCESS;
    return TRUE;
}

static gboolean send_ack_handler(gpointer unused)
{
    if (ack_state_change(test_data.proxy, test_data.state))
        run_next_test();
    return FALSE;
}

static gboolean ack_timeout_handler(gpointer unused)
{
    test_data.ack_timeout_id = 0;
    ack_status = ACK_TIMEOUT;
    run_next_test();
    return FALSE;
}

static void powerd_signal_handler(GDBusProxy *proxy, gchar *sender,
                                  gchar *signal, GVariant *params,
                                  gpointer unused)
{
    cli_debug("Received signal %s\n", signal);
    if (strcmp(signal, "SysPowerStateChange"))
        return;

    /* Check if we're trying to time out */
    if (test_data.ack_delay < 0)
        return;

    if (test_data.ack_timeout_id > 0) {
        g_source_remove(test_data.ack_timeout_id);
        test_data.ack_timeout_id = 0;
    }

    if (test_data.ack_delay == 0) {
        /* immediate ack */
        ack_state_change(proxy, test_data.state);
        run_next_test();
    } else {
        /* delayed ack */
        g_timeout_add(test_data.ack_delay, send_ack_handler, NULL);
    }
}

static gboolean register_test(gpointer unused)
{
    gboolean result;

    result = do_test(register_client(test_data.proxy) == TRUE);
    if (!result)
        g_main_loop_quit(main_loop);
    else {
        run_next_test();
    }
    return FALSE;
}

static gboolean ack_test_part1(gpointer unused)
{
    printf("Testing ack on active state transition\n");

    test_data.state = POWERD_SYS_STATE_ACTIVE;
    test_data.ack_delay = 0;
    reset_test_state();
    test_data.ack_timeout_id = g_timeout_add(5000, ack_timeout_handler, NULL);
    requestSysState("ack-test", test_data.state, &test_data.cookie);
    return FALSE;
}

static gboolean ack_test_part2(gpointer unused)
{
    /* Check result of part 1 */
    do_test(ack_status == ACK_SUCCESS);

    reset_test_state();
    printf("Testing ack on suspend state transition\n");
    test_data.state = POWERD_SYS_STATE_SUSPEND;
    test_data.ack_delay = 0;
    test_data.ack_timeout_id = g_timeout_add(5000, ack_timeout_handler, NULL);
    clearSysState(test_data.cookie);
    return FALSE;
}

static gboolean ack_test_check_result(gpointer unused)
{
    do_test(ack_status == ACK_SUCCESS);
    run_next_test();
    return FALSE;
}

static gboolean no_ack_test_part1(gpointer unused)
{
    printf("Testing active state transition with no ack\n");

    reset_test_state();
    test_data.state = POWERD_SYS_STATE_ACTIVE;
    test_data.ack_delay = -1;
    test_data.ack_timeout_id = g_timeout_add(5000, ack_timeout_handler, NULL);
    requestSysState("no-ack-test", test_data.state, &test_data.cookie);
    return FALSE;
}

static gboolean no_ack_test_part2(gpointer unused)
{
    printf("Testing suspend state transition with no ack\n");

    reset_test_state();
    test_data.state = POWERD_SYS_STATE_SUSPEND;
    test_data.ack_delay = -1;
    test_data.ack_timeout_id = g_timeout_add(5000, ack_timeout_handler, NULL);
    clearSysState(test_data.cookie);
    return FALSE;
}

static gboolean bad_ack_test(gpointer unused)
{
    printf("Testing acknowledge of wrong state\n");

    reset_test_state();
    test_data.state = POWERD_SYS_STATE_SUSPEND;
    test_data.ack_delay = 0;
    test_data.ack_timeout_id = g_timeout_add(5000, ack_timeout_handler, NULL);
    requestSysState("bad-ack-test", POWERD_SYS_STATE_ACTIVE, &test_data.cookie);
    return FALSE;
}

static gboolean bad_ack_test_result(gpointer unused)
{
    do_test(ack_status == ACK_FAILURE);
    run_next_test();
    return FALSE;
}

static gboolean unregister_test(gpointer unused)
{
    gboolean result;

    result = do_test(unregister_client(test_data.proxy) == TRUE);
    if (!result)
        g_main_loop_quit(main_loop);
    else
        run_next_test();
    return FALSE;
}

static gboolean no_unregister_test(gpointer unused)
{
    printf("Testing exit without unregistering\n");
    run_next_test();
    return FALSE;
}

static GSourceFunc client_tests[] = {
    register_test,
    ack_test_part1,
    ack_test_part2,
    ack_test_check_result,
    no_ack_test_part1,
    no_ack_test_part2,
    bad_ack_test,
    bad_ack_test_result,
    unregister_test,
    register_test,
    no_unregister_test,
};

static gboolean run_next_test_handler(gpointer unused)
{
    static int test_num = 0;

    if (test_data.ack_timeout_id > 0) {
        g_source_remove(test_data.ack_timeout_id);
        test_data.ack_timeout_id = 0;
    }

    if (test_num >= ARRAY_SIZE(client_tests)) {
        test_num = 0;
        g_main_loop_quit(main_loop);
        return FALSE;
    }

    g_timeout_add(0, client_tests[test_num], NULL);
    test_num++;
    return FALSE;
}

void powerd_cli_client_test(GDBusProxy *proxy)
{
    printf("Running client tests, cannot verify all results\n");

    silent_errors = TRUE;
    test_data.proxy = proxy;
    main_loop = g_main_loop_new(NULL, FALSE);
    g_signal_connect(proxy, "g-signal", G_CALLBACK(powerd_signal_handler), NULL);
    g_timeout_add(0, run_next_test_handler, NULL);
    g_main_loop_run(main_loop);
    g_main_loop_unref(main_loop);
    silent_errors = FALSE;
}
