/*
 * Picviz - Parallel coordinates ploter
 * Copyright (C) 2008 Sebastien Tricaud <toady@gscore.org>
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation version 3.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 * $Id: py-picviz.c 307 2008-11-08 10:49:13Z toady $
 */

#include <Python.h>

#include <stdio.h>

#include <picviz.h>
#include <linuxlist.h>

#include "py-picviz.h"


static int debug = 0;

PyObject *pypicviz_version_get(PyObject *self, PyObject *args)
{
	PyObject *py_result = Py_BuildValue("s", PICVIZ_PYTHON_VERSION);
	return py_result;
}

PyObject *pypicviz_set_debug(PyObject *self, PyObject *args)
{
	PyObject *py_result = Py_BuildValue("d", 1);

	fprintf(stderr,"[PICVIZ_PYTHON_BINDINGS]: Debug mode activated\n");
	debug = 1;

	return py_result;
}

int pypicviz_dict_keyval_append_str(PyObject *dict, char *key, char *val)
{
        PyObject *value;
        int ret;

        value = PyString_FromString(val);

        ret = PyDict_SetItemString(dict, key, value);
        if (ret < 0) {
                fprintf(stderr, "[PYTHON_BINDINGS]: Can not append value '%s' to '%s'", val, key);
        }

        return ret;
}

int pypicviz_dict_keyval_append_long(PyObject *dict, char *key, long val)
{
        PyObject *value;
        int ret;

        value = PyInt_FromLong(val);

        ret = PyDict_SetItemString(dict, key, value);
        if (ret < 0) {
                fprintf(stderr, "[PYTHON_BINDINGS]: Can not append value '%li' to '%s'", val, key);
        }

        return ret;
}

int pypicviz_dict_keyval_append_float(PyObject *dict, char *key, char *val)
{
        PyObject *value;
        int ret;

        value = PyFloat_FromString(PyString_FromString(val), NULL); /* second arg. ignored, only for backward compat. */

        ret = PyDict_SetItemString(dict, key, value);
        if (ret < 0) {
                fprintf(stderr, "[PYTHON_BINDINGS]: Can not append value '%s' to '%s'", val, key);
        }

        return ret;
}

PyObject *pypicviz_lines_list_build_from_file(PyObject *self, PyObject *args)
{
	PicvizImage *image;
	PicvizAxis *axis;
	PicvizLine *line;
	PicvizAxisPlot *axisplot;

	unsigned int ui = 0;
	int ret;
	char *filename;
	char *filter;

        PyObject *main_dict = PyDict_New();
        PyObject *axeslist = PyList_New(0);
        PyObject *lineslist = PyList_New(0);
        //PyObject *linedata = PyDict_New();

	picviz_init();
	if ( ! PyArg_ParseTuple(args, "sz", &filename, &filter) ) {
                PyErr_Print();
                PyErr_Clear();
                Py_RETURN_FALSE;
        }

	if (debug) {
		fprintf(stderr, "ParseImage name=[%s], filter=[%s]\n", filename, filter);
	}


	image = (PicvizImage *)pcv_parse(filename, filter);
	if ( ! image ) {
		fprintf(stderr, "Cannot create image");
		Py_RETURN_NONE;
	}

        /* Set Image properties */
        ret = pypicviz_dict_keyval_append_long(main_dict, "height", image->height);
        if (ret < 0)
                Py_RETURN_NONE;

        /* Set axes */
	llist_for_each_entry(axis, &image->axes, list) {
		PyObject *axisdata = PyDict_New();
		ret = pypicviz_dict_keyval_append_long(axisdata, "id", axis->id);
                if (ret < 0) {
                        Py_RETURN_NONE;
                }
		ret = pypicviz_dict_keyval_append_str(axisdata, "label", picviz_properties_get(axis->props, "label"));
                if (ret < 0) {
                        Py_RETURN_NONE;
                }
                ui++;
		PyList_Append(axeslist, axisdata);
        }
        ret = pypicviz_dict_keyval_append_long(main_dict, "axes_number", ui);
        if (ret < 0)
                Py_RETURN_NONE;

        ret = PyDict_SetItemString(main_dict, "axes", axeslist);

        /* Set Lines plots and properties */
	llist_for_each_entry(line, &image->lines, list) {
                PyObject *axisplotlist = PyList_New(0);
                //ret = pypicviz_dict_keyval_append_long(linedata, "id", line->id);
                //if (ret < 0) {
                //        Py_RETURN_NONE;
                //}
                //ret = pypicviz_dict_keyval_append_str(linedata, "color", line->props->color);
                //if (ret < 0) {
                //        Py_RETURN_NONE;
                //}
                /* Set Line plots */
		llist_for_each_entry(axisplot, &line->axisplot, list) {
			PyObject *plotsdata = PyDict_New();
			/* We add the color information along with plots, to ease the line creation with QT */
			ret = pypicviz_dict_keyval_append_long(plotsdata, "hidden", line->hidden);
			if (ret < 0) {
				Py_RETURN_NONE;
			}
			ret = pypicviz_dict_keyval_append_str(plotsdata, "color", picviz_properties_get(line->props, "color"));
			if (ret < 0) {
				Py_RETURN_NONE;
			}
			ret = pypicviz_dict_keyval_append_float(plotsdata, "penwidth", picviz_properties_get(line->props, "penwidth"));
			if (ret < 0) {
				Py_RETURN_NONE;
			}
			ret = pypicviz_dict_keyval_append_str(plotsdata, "strval", axisplot->strval);
			if (ret < 0) {
				Py_RETURN_NONE;
			}
			ret = pypicviz_dict_keyval_append_long(plotsdata, "y", axisplot->y);
			if (ret < 0) {
				Py_RETURN_NONE;
			}
			ret = pypicviz_dict_keyval_append_long(plotsdata, "axis_id", axisplot->axis_id);
			if (ret < 0) {
				Py_RETURN_NONE;
			}
			PyList_Append(axisplotlist, plotsdata);

		}
		PyList_Append(lineslist, axisplotlist);
	}
        ret = PyDict_SetItemString(main_dict, "lines", lineslist);

	picviz_image_destroy(image);

	return main_dict;
}


/****************
 * Method table *
 ****************/
static PyMethodDef picvizMethods[] =
{
    { "Version", pypicviz_version_get, METH_VARARGS,
      "Returns the Picviz version"},
    { "Debug", pypicviz_set_debug, METH_VARARGS,
      "Set Picviz Python bindings in debug mode"},
    { "ParseImage", pypicviz_lines_list_build_from_file, METH_VARARGS,
      "Parse lines from the file"},

    {NULL,NULL}
};


#ifdef WIN32
__declspec(dllexport)
#endif
void initpicviz(void)
{
 	PyObject *m, *d;


	m = Py_InitModule4("picviz",
			picvizMethods,
			"Parse, Render and Manipulate data from Picviz",
			0,
			PYTHON_API_VERSION);

	d = PyModule_GetDict(m);

}

