/*
 *  Copyright (c) 2008 Cyrille Berger <cberger@cberger.net>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation;
 * either version 2, or (at your option) any later version of the License.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library; see the file COPYING.  If not, write to
 * the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
 * Boston, MA 02110-1301, USA.
 */

#include "ConvertExpression.h"

#include "../LLVMBackend/CodeGenerator_p.h"
#include "../LLVMBackend/ExpressionResult_p.h"
#include "../LLVMBackend/ExpressionGenerationContext_p.h"
#include "GenerationVisitor.h"

using namespace GTLCore;
using namespace GTLCore::AST;

ConvertExpression::ConvertExpression( AST::Expression* _value ) : m_value(_value)
{
}

ConvertExpression::~ConvertExpression()
{
  delete m_value;
}

void ConvertExpression::markAsReturnExpression()
{
  m_value->markAsReturnExpression();
}

bool ConvertExpression::isConstant() const
{
  return m_value->isConstant();
}

const AST::Expression* ConvertExpression::value() const
{
  return m_value;
}

DefaultConvertExpression::DefaultConvertExpression( AST::Expression* _value,  const GTLCore::Type* _dstType ) : ConvertExpression(_value), m_dstType(_dstType)
{
}

DefaultConvertExpression::~DefaultConvertExpression()
{
}

const GTLCore::Type* DefaultConvertExpression::type() const
{
  return m_dstType;
}

LLVMBackend::ExpressionResult DefaultConvertExpression::generateValue( LLVMBackend::GenerationContext& _gc, LLVMBackend::ExpressionGenerationContext& _egc ) const
{
  LLVMBackend::ExpressionResult result = value()->generateValue( _gc, _egc );
  if( result.isConstant() )
  {
    return LLVMBackend::ExpressionResult( LLVMBackend::CodeGenerator::convertConstantTo( result.constant(), result.type(), m_dstType ), m_dstType);
  } else {
    return LLVMBackend::ExpressionResult( LLVMBackend::CodeGenerator::convertValueTo( _egc.currentBasicBlock(),  result.value(), result.type(), m_dstType ), m_dstType);
  }
}

ExpressionResultSP DefaultConvertExpression::generateValue( GenerationVisitor* _generationVisitor) const
{
  return _generationVisitor->convertExpressionTo( value()->generateValue(_generationVisitor), m_dstType, annotation());
}

ArrayToVectorConvertExpression::ArrayToVectorConvertExpression( AST::Expression* _value, const GTLCore::Type* _dstType ) : ConvertExpression(_value), m_dstType(_dstType)
{
}

ArrayToVectorConvertExpression::~ArrayToVectorConvertExpression()
{
}

const GTLCore::Type* ArrayToVectorConvertExpression::type() const
{
  return m_dstType;
}

LLVMBackend::ExpressionResult ArrayToVectorConvertExpression::generateValue( LLVMBackend::GenerationContext& _gc, LLVMBackend::ExpressionGenerationContext& _egc ) const
{
  LLVMBackend::ExpressionResult result = value()->generateValue( _gc, _egc );
  if( result.isConstant() )
  {
    return LLVMBackend::ExpressionResult( LLVMBackend::CodeGenerator::convertConstantArrayToVector( result.constant(), result.type(), m_dstType ), m_dstType);
  } else {
    return LLVMBackend::ExpressionResult( LLVMBackend::CodeGenerator::convertArrayToVector( _gc, _egc,  result.value(), result.type(), m_dstType ), m_dstType);
  }
}

ExpressionResultSP ArrayToVectorConvertExpression::generateValue( GenerationVisitor* _generationVisitor) const
{
  return _generationVisitor->convertArrayToVector( value()->generateValue(_generationVisitor), m_dstType, annotation() );
}

