/*
 * This file is part of the Ubuntu TV Media Scanner
 * Copyright (C) 2012-2013 Canonical Ltd.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License version 3 as
 * published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Contact: Jim Hodapp <jim.hodapp@canonical.com>
 * Authored by: Mathias Hasselmann <mathias@openismus.com>
 */
#include "mediascanner/mediaroot.h"

// POSIX Library
#include <sys/types.h>
#include <sys/stat.h>

// Boost C++
#include <boost/locale/format.hpp>

// C++ Standard Library
#include <map>
#include <set>
#include <string>
#include <vector>

// Media Scanner Library
#include "mediascanner/glibutils.h"
#include "mediascanner/logging.h"
#include "mediascanner/utilities.h"

namespace mediascanner {

// Boost C++
using boost::locale::format;

// Context specific logging domains
static const logging::Domain kWarning("warning/roots", logging::warning());

////////////////////////////////////////////////////////////////////////////////

class MediaRoot::Private {
public:
    Private(const Wrapper<GFile> &base,
            const Wrapper<GFile> &root)
        : base_(base)
        , root_(root) {
    }

    explicit Private(const std::string &error_message)
        : error_message_(error_message) {
    }

    const Wrapper<GFile> base_;
    const Wrapper<GFile> root_;

    const std::string error_message_;
};

////////////////////////////////////////////////////////////////////////////////

class MediaRootManager::Private {
public:
    Private()
        : idle_id_(0)
        , enabled_(true) {
    }

    ~Private() {
        teardown();
    }

    MediaRoot make_root(const MediaRoot &parent,
                        const std::string &path);
    MediaRoot make_root(const std::string &path);

    void AddRoot(const MediaRoot &root);
    void RemoveRoot(const MediaRoot &root);

    void setup();
    void teardown();

    bool enabled() const {
        return enabled_;
    }

    void initialize() {
        if (enabled_ && not volume_monitor_)
            on_init();
    }

    Wrapper<GUdevClient> udev_client() {
        if (not udev_client_) {
            static const char* const subsystems[] = { "block", nullptr };
            udev_client_ = take(g_udev_client_new(subsystems));
        }

        return udev_client_;
    }

private:
    void on_init();

    static void OnMountChanged(Private *d, GMount *mount);
    static void OnMountRemoved(Private *d, GMount *mount);

public:
    std::vector<MediaRoot> media_roots_;
    std::vector<std::string> manual_roots_;
    std::vector<Listener *> listeners_;

    typedef std::map<std::string, std::set<std::string> > RelativeRootMap;
    RelativeRootMap relative_roots_;

private:
    Wrapper<GVolumeMonitor> volume_monitor_;
    Wrapper<GUdevClient> udev_client_;
    unsigned idle_id_;
    bool enabled_;
};

////////////////////////////////////////////////////////////////////////////////

static Wrapper<GFile> resolve_path(const Wrapper<GFile> &parent,
                                   const std::string &path) {
    if (g_path_is_absolute(path.c_str()))
        return take(g_file_new_for_path(path.c_str()));

    if (parent)
        return take(g_file_resolve_relative_path(parent.get(), path.c_str()));

    return Wrapper<GFile>();
}

static std::string get_path(const Wrapper<GFile> &file) {
    if (not file)
        return std::string();

    return take_string(g_file_get_path(file.get()));
}

static std::string get_relative_path(const Wrapper<GFile> &parent,
                                     const Wrapper<GFile> &descendant) {
    return take_string(g_file_get_relative_path(parent.get(),
                                                descendant.get()));
}

////////////////////////////////////////////////////////////////////////////////

MediaRoot::MediaRoot(Private *d)
    : d(d) {
}

MediaRoot::~MediaRoot() {
}

bool MediaRoot::is_valid() const {
    return d->error_message_.empty()
            && d->root_;
}

std::string MediaRoot::error_message() const {
    return d->error_message_;
}

Wrapper<GFile> MediaRoot::file() const {
    return d->root_;
}

bool MediaRoot::operator==(const MediaRoot &other) const {
    return g_file_equal(file().get(), other.file().get());
}

bool MediaRoot::operator<(const MediaRoot &other) const {
    return path() < other.path();
}

std::string MediaRoot::path() const {
    return get_path(d->root_);
}

std::string MediaRoot::base_path() const {
    return get_path(d->base_);
}

std::string MediaRoot::relative_path() const {
    return get_relative_path(d->base_, d->root_);
}

std::string MediaRoot::group_id() const {
    std::string id;

    if (is_valid()) {
        const std::string path = relative_path();
        id += "media:"  + path;
    }

    return id;
}

////////////////////////////////////////////////////////////////////////////////

MediaRoot MediaRootManager::Private::make_root(const MediaRoot &parent,
                                               const std::string &path) {
    BOOST_ASSERT_MSG(not g_path_is_absolute(path.c_str()), path.c_str());

    return MediaRoot(new MediaRoot::Private(parent.d->base_,
                                        resolve_path(parent.d->base_, path)));
}

MediaRoot MediaRootManager::Private::make_root(const std::string &path) {
    BOOST_ASSERT_MSG(g_path_is_absolute(path.c_str()), path.c_str());

    std::string error_message;
    const Wrapper<GFile> root = take(g_file_new_for_path(path.c_str()));

    const Wrapper<GFile> base = take(g_file_new_for_path("/"));

    return MediaRoot(new MediaRoot::Private(base, root));
}

void MediaRootManager::Private::AddRoot(const MediaRoot &root) {
    for (const MediaRoot &r: media_roots_) {
        if (r.path() == root.path())
            return;
    }

    media_roots_.push_back(root);

    for (Listener *const l: listeners_) {
        l->OnMediaRootAdded(root);
    }
}

void MediaRootManager::Private::RemoveRoot(const MediaRoot &root) {
    std::vector<MediaRoot>::iterator it = media_roots_.begin();

    while (it != media_roots_.end()) {
        if (it->path() == root.path()) {
            // The media root passed to this function might not have
            // proper UUID and base path because the media got removed
            // already. Therefore use the stored variant for notification.
            const MediaRoot removed = *it;
            media_roots_.erase(it);

            for (Listener *const l: listeners_) {
                l->OnMediaRootRemoved(removed);
            }

            return;
        }

        ++it;
    }
}

void MediaRootManager::Private::setup() {
    enabled_ = true;

    if (not idle_id_ && not volume_monitor_)
        idle_id_ = Idle::AddOnce(std::bind(&Private::on_init, this));
}

void MediaRootManager::Private::teardown() {
    enabled_ = false;

    if (volume_monitor_)
        g_signal_handlers_disconnect_by_data(volume_monitor_.get(), this);

    if (idle_id_) {
        Idle::Remove(idle_id_);
        idle_id_ = 0;
    }
}

void MediaRootManager::Private::on_init() {
    // Initialize in idle handler to ensure all stakeholders are in place.
    volume_monitor_ = take(g_volume_monitor_get());
    idle_id_ = 0;

    g_signal_connect_swapped(volume_monitor_.get(), "mount-added",
                             G_CALLBACK(&Private::OnMountChanged),
                             this);
    g_signal_connect_swapped(volume_monitor_.get(), "mount-changed",
                             G_CALLBACK(&Private::OnMountChanged),
                             this);
    g_signal_connect_swapped(volume_monitor_.get(), "mount-pre-unmount",
                             G_CALLBACK(&Private::OnMountRemoved),
                             this);
    g_signal_connect_swapped(volume_monitor_.get(), "mount-removed",
                             G_CALLBACK(&Private::OnMountRemoved),
                             this);

    const ListWrapper<GMount> mounts
            (g_volume_monitor_get_mounts(volume_monitor_.get()));

    for (GList *l = mounts.get(); l; l = l->next) {
        GMount *const mount = static_cast<GMount *>(l->data);
        OnMountChanged(this, mount);
    }
}

void MediaRootManager::Private::OnMountChanged(Private *d, GMount *mount) {
    // Ignore shadow mounts, they are of no interest to us.
    if (g_mount_is_shadowed(mount))
        return;

    const Wrapper<GFile> root = take(g_mount_get_root(mount));
    d->AddRoot(d->make_root(get_path(root)));
}

void MediaRootManager::Private::OnMountRemoved(Private *d, GMount *mount) {
    // Ignore shadow mounts, they are of no interest to us.
    if (g_mount_is_shadowed(mount))
        return;

    const Wrapper<GFile> root = take(g_mount_get_root(mount));
    d->RemoveRoot(d->make_root(get_path(root)));
}

////////////////////////////////////////////////////////////////////////////////

MediaRootManager::MediaRootManager()
    : d(new Private) {
    d->setup();
}

MediaRootManager::~MediaRootManager() {
    delete d;
}

void MediaRootManager::initialize() {
    d->initialize();
}

void MediaRootManager::add_listener(Listener *listener) {
    d->listeners_.push_back(listener);
}

void MediaRootManager::remove_listener(Listener *listener) {
    std::vector<Listener *>::iterator it = d->listeners_.begin();

    while (it != d->listeners_.end()) {
        if (*it == listener) {
            it = d->listeners_.erase(it);
            continue;
        }

        ++it;
    }
}

std::vector<MediaRoot> MediaRootManager::media_roots() const {
    return d->media_roots_;
}

MediaRoot MediaRootManager::AddRelativeRoot(const std::string &relative_path) {
    // Check if this relative root's volume is already know,
    // and report this relative media root if that's the case.
    for (const MediaRoot &known: d->media_roots_) {
            if (known.relative_path().empty()) {
                // Reuse media root as parent
                const MediaRoot relative = d->make_root(known, relative_path);
                d->AddRoot(relative);
                return relative;

            if (known.relative_path() == relative_path) {
                // Reuse manual media root
                return known;
            }
        }
    }

    const MediaRoot parent = d->make_root(std::string("/"));

    if (relative_path.empty() || not parent.is_valid())
        return parent;

    return d->make_root(parent, relative_path);
}

void MediaRootManager::AddManualRoot(const std::string &path) {
    d->manual_roots_.push_back(path);
    d->AddRoot(d->make_root(path));
}

std::vector<std::string> MediaRootManager::manual_roots() const {
    return d->manual_roots_;
}

void MediaRootManager::set_enabled(bool enabled) {
    if (enabled) {
        d->setup();
    } else {
        d->teardown();
    }
}

bool MediaRootManager::enabled() const {
    return d->enabled();
}

MediaRoot MediaRootManager::make_root(const std::string &path) const {
    return d->make_root(path);
}

} // namespace mediascanner
