from twisted.internet.defer import inlineCallbacks

from juju.lib.testing import TestCase
from juju.lib.mocker import MATCH, KWARGS
from juju.lib.service import TwistedDaemonService
from juju.lib.lxc.tests.test_lxc import uses_sudo

import os


class TwistedDaemonServiceTest(TestCase):

    @inlineCallbacks
    def setUp(self):
        yield super(TwistedDaemonServiceTest, self).setUp()
        self.setup_service()

    def setup_service(self):
        service = TwistedDaemonService("juju-machine-agent",
                                       "/tmp/.juju-test.pid",
                                       use_sudo=False)
        service.set_description("Juju machine agent")
        service.set_environ({"JUJU_MACHINE_ID": 0})
        service.set_command(["/bin/true", ])
        self.service = service

        if os.path.exists("/tmp/.juju-test.pid"):
            os.remove("/tmp/.juju-test.pid")

        return service

    def setup_mock(self):
        self.check_call = self.mocker.replace("subprocess.check_call")

    def mock_call(self, args):
        def exe_match(cmd):
            cmd = " ".join(str(s) for s in cmd)
            return cmd.startswith(args)

        self.check_call(MATCH(exe_match), KWARGS)
        self.mocker.result(0)

    @inlineCallbacks
    def test_simple_service_start(self):
        self.setup_mock()
        self.mock_call("/bin/true")
        self.mocker.replay()

        yield self.service.start()

    def test_set_output_path(self):
        # defaults work
        self.assertEqual(self.service.output_path,
                         "/tmp/juju-machine-agent.output")
        # override works
        self.service.output_path = "/tmp/valid.log"
        self.assertEqual(self.service.output_path, "/tmp/valid.log")

    @inlineCallbacks
    def test_simple_service_start_destroy(self):
        self.setup_mock()
        mock_service = self.mocker.patch(self.service)
        get_pid = mock_service.get_pid
        get_pid()
        self.mocker.result(1337)

        is_running = mock_service.is_running
        is_running()
        self.mocker.result(False)

        is_running()
        self.mocker.result(True)

        self.mock_call(("/bin/true", ))
        self.mock_call(("kill", "1337"))
        self.mocker.replay()

        yield self.service.start()
        yield self.service.destroy()

    @inlineCallbacks
    def test_webservice_start(self):
        # test using a real twisted service (with --pidfile)
        # arg ordering matters here so we set pidfile manually
        self.service.set_command([
            "env", "twistd",
            "--pidfile", "/tmp/.juju-test.pid",
            "--logfile", "/tmp/.juju-test.log",
            "web",
            "--port", "9871",
            "--path", "/lib",
            ])

        yield self.service.start()
        yield self.sleep(0.5)
        self.assertTrue(self.service.get_pid())
        self.assertTrue(self.service.is_running())
        self.assertTrue(os.path.exists("/tmp/.juju-test.pid"))
        yield self.service.destroy()
        yield self.sleep(0.1)
        self.assertFalse(os.path.exists("/tmp/.juju-test.pid"))

    @uses_sudo
    @inlineCallbacks
    def test_sudo_env_vars(self):
        self.service.set_environ(
            {"JUJU_MACHINE_ID": 0, "PYTHONPATH": "foo2"})
        self.service.set_daemon(False)
        self.service.set_command(["/usr/bin/env"])
        self.service.output_path = self.makeFile()
        yield self.service.start()

        with open(self.service.output_path) as fh:
            contents = fh.read()
            self.assertIn("PYTHONPATH=foo2", contents)
            self.assertIn("JUJU_MACHINE_ID=0", contents)
