package gollectd

import (
	"encoding/hex"
	"testing"

	"github.com/stretchr/testify/assert"
	"github.com/stretchr/testify/require"
)

type FormatTests struct {
	Description string
	Expected    string
	Packet      Packet
}

var formatTests = []FormatTests{
	{
		Description: "leeloo/cpu-0/cpu-idle",
		Expected:    "leeloo/cpu-0/cpu-idle",
		Packet: Packet{
			Hostname:       "leeloo",
			Plugin:         "cpu",
			PluginInstance: "0",
			Type:           "cpu",
			TypeInstance:   "idle",
		},
	},
	{
		Description: "alyja/memory/memory-used",
		Expected:    "alyja/memory/memory-used",
		Packet: Packet{
			Hostname:     "alyja",
			Plugin:       "memory",
			Type:         "memory",
			TypeInstance: "used",
		},
	},
	{
		Description: "wanda/disk-hdc1/disk_octets",
		Expected:    "wanda/disk-hdc1/disk_octets",
		Packet: Packet{
			Hostname:       "wanda",
			Plugin:         "disk",
			PluginInstance: "hdc1",
			Type:           "disk_octets",
		},
	},
	{
		Description: "leeloo/load/load",
		Expected:    "leeloo/load/load",
		Packet: Packet{
			Hostname: "leeloo",
			Plugin:   "load",
			Type:     "load",
		},
	},
}

func TestFormat(t *testing.T) {
	assert := assert.New(t)

	for _, test := range formatTests {
		got := test.Packet.FormatName()
		assert.Equal(test.Expected, got, test.Description)
	}
}

func TestTypesDB(t *testing.T) {
	typedb, err := TypesDB(TypesDBData)
	if err != nil {
		t.Error(err)
	}
	dataSources, ok := typedb["compression"]
	if !ok {
		t.Error(`"compression" not found`)
	}

	found := len(dataSources)
	expected := 2
	if found != expected {
		t.Errorf("found %d datasources; expected %d", found, expected)
		return
	}

	ds := dataSources[0]

	if ds.Name != "uncompressed" {
		t.Errorf("ds.Name = %s; expected \"uncompressed\"", ds.Name)
	}

	if ds.Type != TypeDerive {
		dsType, ok := ValueTypeValues[ds.Type]
		if !ok {
			t.Error("unrecognized type")
		} else {
			t.Errorf("ds.Type = %s; expected TypeDerive", dsType)
		}
	}

	if ds.Min != "0" {
		t.Errorf("ds.Min = %s; expected 0", ds.Min)
	}

	if ds.Max != "U" {
		t.Errorf("ds.Max = %s; expected 0", ds.Max)
	}

	ds = dataSources[1]

	if ds.Name != "compressed" {
		t.Errorf("ds.Name = %s; expected \"uncompressed\"", ds.Name)
	}

	if ds.Type != TypeDerive {
		dsType, ok := ValueTypeValues[ds.Type]
		if !ok {
			t.Error("unrecognized type")
		} else {
			t.Errorf("ds.Type = %s; expected TypeDerive", dsType)
		}
	}

	if ds.Min != "0" {
		t.Errorf("ds.Min = %s; expected 0", ds.Min)
	}

	if ds.Max != "U" {
		t.Errorf("ds.Max = %s; expected 0", ds.Max)
	}
}

func TestTypesDB2(t *testing.T) {
	tdb := []byte(`
vs_memory		value:GAUGE:0:9223372036854775807
vs_processes		value:GAUGE:0:65535
vs_threads		value:GAUGE:0:65535
# The following line has intentional whitespace
    
`)

	typesdb, err := TypesDB(tdb)
	if err != nil {
		t.Error(err)
	}

	assert.Equal(t, 3, len(typesdb), "whitespace")
}

var typesDB = func() Types {
	t, err := TypesDB(TypesDBData)
	if err != nil {
		panic(err)
	}
	return t
}()

func TestPackets(t *testing.T) {
	for i, raw := range rawPacketData {
		_, err := Packets(raw, typesDB)
		if err != nil {
			t.Errorf("i = %d: %s", i, err.Error())
		}
	}
}

func TestPacketsShort(t *testing.T) {
	var err error

	require.NotPanics(t, func() {
		short := []byte("ab")
		_, err = Packets(short, typesDB)
	})

	assert.NotNil(t, err)
}

var result *[]Packet

func BenchmarkPackets(b *testing.B) {
	var r *[]Packet
	for n := 0; n < b.N; n++ {
		r, _ = Packets(rawPacketData[0], typesDB)
	}
	result = r
}

// Collectd packets captured using Wireshark
var rawPacketData = func() [][]byte {
	strs := []string{
		"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",
		"0000000e6c6f63616c686f7374000008000c1513676d4380551f0009000c0000000280000000000200086370750000030006310000040008637075000005000975736572000006000f0001020000000000007cad0008000c1513676d43805dbe000500096e696365000006000f00010200000000000001de0008000c1513676d4380697d0005000b73797374656d000006000f0001020000000000001ce80008000c1513676d438072bd0005000969646c65000006000f000102000000000005931c0008000c1513676d43807c430005000977616974000006000f000102000000000000094b0008000c1513676d43808cee0005000c736f6674697271000006000f00010200000000000000120008000c1513676d4380843a0005000e696e74657272757074000006000f00010200000000000000000008000c1513676d438096230005000a737465616c000006000f00010200000000000000000008000c1513676d4380aa9c0003000632000005000975736572000006000f00010200000000000089580008000c1513676d4380b29f000500096e696365000006000f00010200000000000003610008000c1513676d4380c44c0005000969646c65000006000f000102000000000005873d0008000c1513676d4380bc0f0005000b73797374656d000006000f000102000000000000201d0008000c1513676d4380cea40005000977616974000006000f00010200000000000005810008000c1513676d4380d7370005000e696e74657272757074000006000f00010200000000000000000008000c1513676d4380ea830005000a737465616c000006000f00010200000000000000000008000c1513676d437eef62000300063000000500096e696365000006000f00010200000000000003920008000c1513676d4380e0260003000632000005000c736f6674697271000006000f00010200000000000000160008000c1513676d438101410003000633000005000975736572000006000f0001020000000000007d8a0008000c1513676d438109f5000500096e696365000006000f00010200000000000004350008000c1513676d4380244b0003000630000005000e696e74657272757074000006000f00010200000000000000000008000c1513676d438122070003000633000005000969646c65000006000f0001020000000000058eb60008000c1513676d43812e830005000977616974000006000f0001020000000000000ca80008000c1513676d438141480005000c736f6674697271000006000f000102000000000000001e0008000c1513676d43814a5d0005000a737465616c000006000f00010200000000000000000008000c1513676d4381149e0005000b73797374656d000006000f0001020000000000001b9a0008000c1513676d437ea8600003000630000005000975736572000006000f00010200000000000089a80008000c1513676d438138190003000633000005000e696e74657272757074000006000f00010200000000000000000008000c1513676d438a9ca00002000e696e74657266616365000003000965746830000004000e69665f6f63746574730000050005000006001800020202000000000000000000000000000000000008000c1513676d438aea760004000f69665f7061636b657473000006001800020202000000000000000000000000000000000008000c1513676d438b214d0004000e69665f6572726f727300000600180002020200000000000000000000000000000000",
		"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",
		"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",
		"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",
	}
	var rawData [][]byte
	for _, str := range strs {
		buf, err := hex.DecodeString(str)
		if err != nil {
			panic(err)
		}
		rawData = append(rawData, buf)
	}
	return rawData
}()

// Taken from /usr/share/collectd/types.db on a Ubuntu system
var TypesDBData = []byte(`absolute		value:ABSOLUTE:0:U
apache_bytes		value:DERIVE:0:U
apache_connections	value:GAUGE:0:65535
apache_idle_workers	value:GAUGE:0:65535
apache_requests		value:DERIVE:0:U
apache_scoreboard	value:GAUGE:0:65535
ath_nodes		value:GAUGE:0:65535
ath_stat		value:DERIVE:0:U
backends		value:GAUGE:0:65535
bitrate			value:GAUGE:0:4294967295
bytes			value:GAUGE:0:U
cache_eviction		value:DERIVE:0:U
cache_operation		value:DERIVE:0:U
cache_ratio		value:GAUGE:0:100
cache_result		value:DERIVE:0:U
cache_size		value:GAUGE:0:4294967295
charge			value:GAUGE:0:U
compression_ratio	value:GAUGE:0:2
compression		uncompressed:DERIVE:0:U, compressed:DERIVE:0:U
connections		value:DERIVE:0:U
conntrack		value:GAUGE:0:4294967295
contextswitch		value:DERIVE:0:U
counter			value:COUNTER:U:U
cpufreq			value:GAUGE:0:U
cpu			value:DERIVE:0:U
current_connections	value:GAUGE:0:U
current_sessions	value:GAUGE:0:U
current			value:GAUGE:U:U
delay			value:GAUGE:-1000000:1000000
derive			value:DERIVE:0:U
df_complex		value:GAUGE:0:U
df_inodes		value:GAUGE:0:U
df			used:GAUGE:0:1125899906842623, free:GAUGE:0:1125899906842623
disk_latency		read:GAUGE:0:U, write:GAUGE:0:U
disk_merged		read:DERIVE:0:U, write:DERIVE:0:U
disk_octets		read:DERIVE:0:U, write:DERIVE:0:U
disk_ops_complex	value:DERIVE:0:U
disk_ops		read:DERIVE:0:U, write:DERIVE:0:U
disk_time		read:DERIVE:0:U, write:DERIVE:0:U
dns_answer		value:DERIVE:0:U
dns_notify		value:DERIVE:0:U
dns_octets		queries:DERIVE:0:U, responses:DERIVE:0:U
dns_opcode		value:DERIVE:0:U
dns_qtype_cached	value:GAUGE:0:4294967295
dns_qtype		value:DERIVE:0:U
dns_query		value:DERIVE:0:U
dns_question		value:DERIVE:0:U
dns_rcode		value:DERIVE:0:U
dns_reject		value:DERIVE:0:U
dns_request		value:DERIVE:0:U
dns_resolver		value:DERIVE:0:U
dns_response		value:DERIVE:0:U
dns_transfer		value:DERIVE:0:U
dns_update		value:DERIVE:0:U
dns_zops		value:DERIVE:0:U
duration		seconds:GAUGE:0:U
email_check		value:GAUGE:0:U
email_count		value:GAUGE:0:U
email_size		value:GAUGE:0:U
entropy			value:GAUGE:0:4294967295
fanspeed		value:GAUGE:0:U
file_size		value:GAUGE:0:U
files			value:GAUGE:0:U
fork_rate		value:DERIVE:0:U
frequency_offset	value:GAUGE:-1000000:1000000
frequency		value:GAUGE:0:U
fscache_stat		value:DERIVE:0:U
gauge			value:GAUGE:U:U
hash_collisions		value:DERIVE:0:U
http_request_methods	value:DERIVE:0:U
http_requests		value:DERIVE:0:U
http_response_codes	value:DERIVE:0:U
humidity		value:GAUGE:0:100
if_collisions		value:DERIVE:0:U
if_dropped		rx:DERIVE:0:U, tx:DERIVE:0:U
if_errors		rx:DERIVE:0:U, tx:DERIVE:0:U
if_multicast		value:DERIVE:0:U
if_octets		rx:DERIVE:0:U, tx:DERIVE:0:U
if_packets		rx:DERIVE:0:U, tx:DERIVE:0:U
if_rx_errors		value:DERIVE:0:U
if_rx_octets		value:DERIVE:0:U
if_tx_errors		value:DERIVE:0:U
if_tx_octets		value:DERIVE:0:U
invocations		value:DERIVE:0:U
io_octets		rx:DERIVE:0:U, tx:DERIVE:0:U
io_packets		rx:DERIVE:0:U, tx:DERIVE:0:U
ipt_bytes		value:DERIVE:0:U
ipt_packets		value:DERIVE:0:U
irq			value:DERIVE:0:U
latency			value:GAUGE:0:U
links			value:GAUGE:0:U
load			shortterm:GAUGE:0:5000, midterm:GAUGE:0:5000, longterm:GAUGE:0:5000
md_disks		value:GAUGE:0:U
memcached_command	value:DERIVE:0:U
memcached_connections	value:GAUGE:0:U
memcached_items		value:GAUGE:0:U
memcached_octets	rx:DERIVE:0:U, tx:DERIVE:0:U
memcached_ops		value:DERIVE:0:U
memory			value:GAUGE:0:281474976710656
multimeter		value:GAUGE:U:U
mutex_operations	value:DERIVE:0:U
mysql_commands		value:DERIVE:0:U
mysql_handler		value:DERIVE:0:U
mysql_locks		value:DERIVE:0:U
mysql_log_position	value:DERIVE:0:U
mysql_octets		rx:DERIVE:0:U, tx:DERIVE:0:U
nfs_procedure		value:DERIVE:0:U
nginx_connections	value:GAUGE:0:U
nginx_requests		value:DERIVE:0:U
node_octets		rx:DERIVE:0:U, tx:DERIVE:0:U
node_rssi		value:GAUGE:0:255
node_stat		value:DERIVE:0:U
node_tx_rate		value:GAUGE:0:127
objects			value:GAUGE:0:U
operations		value:DERIVE:0:U
percent			value:GAUGE:0:100.1
percent_bytes		value:GAUGE:0:100.1
percent_inodes		value:GAUGE:0:100.1
pf_counters		value:DERIVE:0:U
pf_limits		value:DERIVE:0:U
pf_source		value:DERIVE:0:U
pf_states		value:GAUGE:0:U
pf_state		value:DERIVE:0:U
pg_blks			value:DERIVE:0:U
pg_db_size		value:GAUGE:0:U
pg_n_tup_c		value:DERIVE:0:U
pg_n_tup_g		value:GAUGE:0:U
pg_numbackends		value:GAUGE:0:U
pg_scan			value:DERIVE:0:U
pg_xact			value:DERIVE:0:U
ping_droprate		value:GAUGE:0:100
ping_stddev		value:GAUGE:0:65535
ping			value:GAUGE:0:65535
players			value:GAUGE:0:1000000
power			value:GAUGE:0:U
protocol_counter	value:DERIVE:0:U
ps_code			value:GAUGE:0:9223372036854775807
ps_count		processes:GAUGE:0:1000000, threads:GAUGE:0:1000000
ps_cputime		user:DERIVE:0:U, syst:DERIVE:0:U
ps_data			value:GAUGE:0:9223372036854775807
ps_disk_octets		read:DERIVE:0:U, write:DERIVE:0:U
ps_disk_ops		read:DERIVE:0:U, write:DERIVE:0:U
ps_pagefaults		minflt:DERIVE:0:U, majflt:DERIVE:0:U
ps_rss			value:GAUGE:0:9223372036854775807
ps_stacksize		value:GAUGE:0:9223372036854775807
ps_state		value:GAUGE:0:65535
ps_vm			value:GAUGE:0:9223372036854775807
queue_length		value:GAUGE:0:U
records			value:GAUGE:0:U
requests		value:GAUGE:0:U
response_time		value:GAUGE:0:U
response_code		value:GAUGE:0:U
route_etx		value:GAUGE:0:U
route_metric		value:GAUGE:0:U
routes			value:GAUGE:0:U
serial_octets		rx:DERIVE:0:U, tx:DERIVE:0:U
signal_noise		value:GAUGE:U:0
signal_power		value:GAUGE:U:0
signal_quality		value:GAUGE:0:U
snr			value:GAUGE:0:U
spam_check		value:GAUGE:0:U
spam_score		value:GAUGE:U:U
spl			value:GAUGE:U:U
swap_io			value:DERIVE:0:U
swap			value:GAUGE:0:1099511627776
tcp_connections		value:GAUGE:0:4294967295
temperature		value:GAUGE:U:U
threads			value:GAUGE:0:U
time_dispersion		value:GAUGE:-1000000:1000000
timeleft		value:GAUGE:0:U
time_offset		value:GAUGE:-1000000:1000000
total_bytes		value:DERIVE:0:U
total_connections	value:DERIVE:0:U
total_objects		value:DERIVE:0:U
total_operations	value:DERIVE:0:U
total_requests		value:DERIVE:0:U
total_sessions		value:DERIVE:0:U
total_threads		value:DERIVE:0:U
total_time_in_ms	value:DERIVE:0:U
total_values		value:DERIVE:0:U
uptime			value:GAUGE:0:4294967295
users			value:GAUGE:0:65535
vcl			value:GAUGE:0:65535
vcpu			value:GAUGE:0:U
virt_cpu_total		value:DERIVE:0:U
virt_vcpu		value:DERIVE:0:U
vmpage_action		value:DERIVE:0:U
vmpage_faults		minflt:DERIVE:0:U, majflt:DERIVE:0:U
vmpage_io		in:DERIVE:0:U, out:DERIVE:0:U
vmpage_number		value:GAUGE:0:4294967295
volatile_changes	value:GAUGE:0:U
voltage_threshold	value:GAUGE:U:U, threshold:GAUGE:U:U
voltage			value:GAUGE:U:U
vs_memory		value:GAUGE:0:9223372036854775807
vs_processes		value:GAUGE:0:65535
vs_threads		value:GAUGE:0:65535

#
# Legacy types
# (required for the v5 upgrade target)
#
arc_counts		demand_data:COUNTER:0:U, demand_metadata:COUNTER:0:U, prefetch_data:COUNTER:0:U, prefetch_metadata:COUNTER:0:U
arc_l2_bytes		read:COUNTER:0:U, write:COUNTER:0:U
arc_l2_size		value:GAUGE:0:U
arc_ratio		value:GAUGE:0:U
arc_size		current:GAUGE:0:U, target:GAUGE:0:U, minlimit:GAUGE:0:U, maxlimit:GAUGE:0:U
mysql_qcache		hits:COUNTER:0:U, inserts:COUNTER:0:U, not_cached:COUNTER:0:U, lowmem_prunes:COUNTER:0:U, queries_in_cache:GAUGE:0:U
mysql_threads		running:GAUGE:0:U, connected:GAUGE:0:U, cached:GAUGE:0:U, created:COUNTER:0:U
`)
