// ---------------------------------------------------------------------------
// - TlsParams.cpp                                                           -
// - afnix:tls service - tls parameters class implementation                 -
// ---------------------------------------------------------------------------
// - This program is free software;  you can redistribute it  and/or  modify -
// - it provided that this copyright notice is kept intact.                  -
// -                                                                         -
// - This program  is  distributed in  the hope  that it will be useful, but -
// - without  any  warranty;  without  even   the   implied    warranty   of -
// - merchantability or fitness for a particular purpose.  In no event shall -
// - the copyright holder be liable for any  direct, indirect, incidental or -
// - special damages arising in any way out of the use of this software.     -
// ---------------------------------------------------------------------------
// - copyright (c) 1999-2015 amaury darsch                                   -
// ---------------------------------------------------------------------------

#include "Byte.hpp"
#include "Vector.hpp"
#include "TlsTypes.hxx"
#include "TlsParams.hpp"
#include "QuarkZone.hpp"
#include "Exception.hpp"

namespace afnix {

  // -------------------------------------------------------------------------
  // - private section                                                       -
  // -------------------------------------------------------------------------

  // the plist parameters name and info
  static const String PN_PRM_VMAJ = "TLS-PRM-VMAJ";
  static const String PI_PRM_VMAJ = "TLS PARAMETER MAJOR VERSION";
  static const String PN_PRM_VMIN = "TLS-PRM-VMIN";
  static const String PI_PRM_VMIN = "TLS PARAMETER MINOR VERSION";

  // -------------------------------------------------------------------------
  // - class section                                                         -
  // -------------------------------------------------------------------------

  // create a default tls parameters

  TlsParams::TlsParams (void) {
    reset ();
  }

  // copy construct this tls parameters

  TlsParams::TlsParams (const TlsParams& that) {
    that.rdlock ();
    try {
      d_vmaj = that.d_vmaj;
      d_vmin = that.d_vmin;
      that.unlock ();
    } catch (...) {
      that.unlock ();
      throw;
    }
  }

  // assign a tls parameters to this one

  TlsParams& TlsParams::operator = (const TlsParams& that) {
    // check for self assignation
    if (this == &that) return *this;
    // lock and assign
    wrlock ();
    that.rdlock ();
    try {
      d_vmaj = that.d_vmaj;
      d_vmin = that.d_vmin;
      unlock ();
      that.unlock ();
      return *this;
    } catch (...) {
      unlock ();
      that.unlock ();
      throw;
    }
  }

  // return the class name
  
  String TlsParams::repr (void) const {
    return "TlsParams";
  }


  // clone this object

  Object* TlsParams::clone (void) const {
    return new TlsParams (*this);
  }


  // reset the tls parameters

  void TlsParams::reset (void) {
    wrlock ();
    try {
      d_vmaj = TLS_VMAJ_DEF;
      d_vmin = TLS_VMIN_DEF;
      unlock ();
    } catch (...) {
      unlock ();
      throw;
    }
  }

  // get the tls parameters as a plist

  Plist TlsParams::getinfo (void) const {
    rdlock ();
    try {
      // create a result plist
      Plist plst;
      // set the result plist
      plst.add (PN_PRM_VMAJ, PI_PRM_VMAJ, (t_long) d_vmaj);
      plst.add (PN_PRM_VMIN, PI_PRM_VMIN, (t_long) d_vmin);
      // here it is
      unlock ();
      return plst;
    } catch (...) {
      unlock ();
      throw;
    }
  }

  // get the tls major version

  t_byte TlsParams::getvmaj (void) const {
    rdlock ();
    try {
      t_byte result = d_vmaj;
      unlock ();
      return result;
    } catch (...) {
      unlock ();
      throw;
    }
  }

  // get the tls minor version

  t_byte TlsParams::getvmin (void) const {
    rdlock ();
    try {
      t_byte result = d_vmin;
      unlock ();
      return result;
    } catch (...) {
      unlock ();
      throw;
    }
  }

  // set the tls version

  void TlsParams::setvers (const t_byte vmaj, const t_byte vmin) {
    wrlock ();
    try {
      // check version
      if (tls_vers_isvalid (vmaj, vmin) == false) {
	String vers = tls_vers_tostring (vmaj, vmin);
	throw Exception ("tls-error", "invalid tls version", vers);
      }
      d_vmaj = vmaj;
      d_vmin = vmin;
      unlock ();
    } catch (...) {
      unlock ();
      throw;
    }
  }

  // -------------------------------------------------------------------------
  // - object section                                                        -
  // -------------------------------------------------------------------------

  // the quark zone
  static const long QUARK_ZONE_LENGTH = 3;
  static QuarkZone  zone (QUARK_ZONE_LENGTH);

  // the object supported quarks
  static const long QUARK_SETVERS = zone.intern ("set-version");
  static const long QUARK_GETVMAJ = zone.intern ("get-major-version");
  static const long QUARK_GETVMIN = zone.intern ("get-minor-version");

  // create a new object in a generic way

  Object* TlsParams::mknew (Vector* argv) {
    // get the number of arguments
    long argc = (argv == nilp) ? 0 : argv->length ();

    // check for 0 argument
    if (argc == 0) return new TlsParams;
    // too many arguments
    throw Exception ("argument-error", 
		     "too many argument with tls parameters");
  }

  // return true if the given quark is defined

  bool TlsParams::isquark (const long quark, const bool hflg) const {
    rdlock ();
    if (zone.exists (quark) == true) {
      unlock ();
      return true;
    }
    bool result = hflg ? TlsInfos::isquark (quark, hflg) : false;
    unlock ();
    return result;
  }

  // apply this object with a set of arguments and a quark

  Object* TlsParams::apply (Runnable* robj, Nameset* nset, const long quark,
			    Vector* argv) {
    // get the number of arguments
    long argc = (argv == nilp) ? 0 : argv->length ();
    
    // check for 0 argument
    if (argc == 0) {
      if (quark == QUARK_GETVMAJ) return new Byte (getvmaj ());
      if (quark == QUARK_GETVMIN) return new Byte (getvmin ());
    }
    // check for 2 arguments
    if (argc == 2) {
      if (quark == QUARK_SETVERS) {
	t_byte vmaj = argv->getbyte (0);
	t_byte vmin = argv->getbyte (1);
	setvers (vmaj, vmin);
	return nilp;
      }
    }
    // call the tls info method
    return TlsInfos::apply (robj, nset, quark, argv);
  }
}
