/*	mainwindow.c
	Copyright (C) 2004-2007 Mark Tyler and Dmitry Groshev

	This file is part of rgbPaint.

	rgbPaint is free software; you can redistribute it and/or modify
	it under the terms of the GNU General Public License as published by
	the Free Software Foundation; either version 2 of the License, or
	(at your option) any later version.

	rgbPaint is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
	GNU General Public License for more details.

	You should have received a copy of the GNU General Public License
	along with rgbPaint in the file COPYING.
*/

#include <stdlib.h>
#include <gtk/gtk.h>
#include <gdk/gdkkeysyms.h>
#include <string.h>
#include <math.h>

#include "global.h"

#include "memory.h"
#include "png.h"
#include "mainwindow.h"
#include "viewer.h"
#include "mygtk.h"
#include "otherwindow.h"
#include "canvas.h"
#include "toolbar.h"


#include "graphics/icon.xpm"



GtkWidget
	*main_window=NULL, *drawing_canvas, *vbox_right, *scrolledwindow_canvas,

	*menu_undo[5], *menu_redo[5], *menu_crop[5],
	*menu_need_marquee[10], *menu_need_selection[20], *menu_need_clipboard[30]
	;

gboolean drag_index = FALSE;
int files_passed, file_arg_start = -1, drag_index_vals[2], cursor_corner, stamp_start = -1,
	global_argc;
char **global_argv;



static int perim_status, perim_x, perim_y, perim_s;	// Tool perimeter

static void clear_perim_real( int ox, int oy )
{
	int x0, y0, x1, y1, zoom = 1, scale = 1;


	/* !!! This uses the fact that zoom factor is either N or 1/N !!! */
	if (can_zoom < 1.0) zoom = rint(1.0 / can_zoom);
	else scale = rint(can_zoom);

	x0 = margin_main_x + ((perim_x + ox) * scale) / zoom;
	y0 = margin_main_y + ((perim_y + oy) * scale) / zoom;
	x1 = margin_main_x + ((perim_x + ox + perim_s - 1) * scale) / zoom + scale - 1;
	y1 = margin_main_y + ((perim_y + oy + perim_s - 1) * scale) / zoom + scale - 1;

	repaint_canvas(x0, y0, 1, y1 - y0 + 1);
	repaint_canvas(x1, y0, 1, y1 - y0 + 1);
	repaint_canvas(x0 + 1, y0, x1 - x0 - 1, 1);
	repaint_canvas(x0 + 1, y1, x1 - x0 - 1, 1);
}

void men_item_state( GtkWidget *menu_items[], gboolean state )
{	// Enable or disable menu items
	int i = 0;
	while ( menu_items[i] != NULL )
	{
		gtk_widget_set_sensitive( menu_items[i], state );
		i++;
	}
}

void pop_men_dis( GtkItemFactory *item_factory, char *items[], GtkWidget *menu_items[] )
{	// Populate disable menu item array
	int i = 0;
	while ( items[i] != NULL )
	{
		menu_items[i] = gtk_item_factory_get_item(item_factory, items[i]);
		i++;
	}
	menu_items[i] = NULL;
}

void men_dis_add( GtkWidget *widget, GtkWidget *menu_items[] )		// Add widget to disable list
{
	int i = 0;

	while ( menu_items[i] != NULL ) i++;
	menu_items[i] = widget;
	menu_items[i+1] = NULL;
}

void pressed_crop( GtkMenuItem *menu_item, gpointer user_data )
{
	int res, x1, y1, x2, y2;

	mtMIN( x1, marq_x1, marq_x2 )
	mtMIN( y1, marq_y1, marq_y2 )
	mtMAX( x2, marq_x1, marq_x2 )
	mtMAX( y2, marq_y1, marq_y2 )

	if ( marq_status != MARQUEE_DONE ) return;
	if ( x1==0 && x2>=(mem_width-1) && y1==0 && y2>=(mem_height-1) ) return;

	res = mem_image_resize(x2 - x1 + 1, y2 - y1 + 1, -x1, -y1, 0);

	if ( res == 0 )
	{
		pressed_select_none(NULL, NULL);
		canvas_undo_chores();
	}
	else memory_errors(3-res);
}

void pressed_select_none( GtkMenuItem *menu_item, gpointer user_data )
{
	if ( marq_status != MARQUEE_NONE )
	{
		paint_marquee(0, marq_x1, marq_y1);
		marq_status = MARQUEE_NONE;
		marq_x1 = marq_y1 = marq_x2 = marq_y2 = -1;
		update_menus();
		gtk_widget_queue_draw( drawing_canvas );
		set_cursor();
	}
}


static void pressed_open_file( GtkMenuItem *menu_item, gpointer user_data )
{	file_selector( FS_PNG_LOAD ); }

static void pressed_save_file_as( GtkMenuItem *menu_item, gpointer user_data )
{	file_selector( FS_PNG_SAVE ); }


static void pressed_save_file( GtkMenuItem *menu_item, gpointer user_data )
{
	ls_settings settings;

	while (strcmp(mem_filename, _("Untitled")))
	{
		init_ls_settings(&settings, NULL);
		settings.ftype = file_type_by_ext(mem_filename, FF_IMAGE);
		if (settings.ftype == FT_NONE) break;
		settings.mode = FS_PNG_SAVE;
		if (gui_save(mem_filename, &settings) < 0) break;
		if ( close_after_save ) delete_event( NULL, NULL, NULL);
		return;
	}
	file_selector(FS_PNG_SAVE);
}


int gui_save(char *filename, ls_settings *settings)
{
	int res;
	char mess[512];

	/* Prepare to save image */
	memcpy(settings->img, mem_img, sizeof(chanlist));
	settings->pal = mem_pal;
	settings->width = mem_width;
	settings->height = mem_height;
	settings->bpp = mem_img_bpp;
	settings->colors = mem_cols;

	res = save_image(filename, settings);
	if ( res < 0 )
	{
		snprintf(mess, 500, _("Unable to save file: %s"), filename);
		alert_box_stock( _("Error"), mess, GTK_STOCK_OK, NULL, NULL );
	}
	else
	{
		notify_unchanged();
		register_file( filename );
	}

	return res;
}


void zoom_in()
{
	if (can_zoom >= 1) align_size(can_zoom + 1);
	else align_size(1.0 / (rint(1.0 / can_zoom) - 1));
}

void zoom_out()
{
	if (can_zoom > 1) align_size(can_zoom - 1);
	else align_size(1.0 / (rint(1.0 / can_zoom) + 1));
}

void quit_all( GtkMenuItem *menu_item, gpointer user_data )
{
	delete_event( NULL, NULL, NULL );
}

int move_arrows( int *c1, int *c2, int value )
{
	int ox1 = marq_x1, oy1 = marq_y1, ox2 = marq_x2, oy2 = marq_y2;
	int nx1, ny1, nx2, ny2;

	*c1 = *c1 + value;
	*c2 = *c2 + value;

	nx1 = marq_x1;
	ny1 = marq_y1;
	nx2 = marq_x2;
	ny2 = marq_y2;

	marq_x1 = ox1;
	marq_y1 = oy1;
	marq_x2 = ox2;
	marq_y2 = oy2;

	paint_marquee(0, nx1, ny1);
	*c1 = *c1 + value;
	*c2 = *c2 + value;
	paint_marquee(1, ox1, oy1);

	return 1;
}

static void resize_marquee( int dx, int dy )
{
	paint_marquee(0, marq_x1, marq_y1);

	marq_x2 += dx;
	marq_y2 += dy;

	paint_marquee(1, marq_x1, marq_y1);
}

/* Forward declaration */
static void mouse_event(int event, int x0, int y0, guint state, guint button,
	gdouble pressure, int mflag);

/* For "dual" mouse control */
static int unreal_move, lastdx, lastdy;

static void move_mouse(int dx, int dy, int button)
{
	static GdkModifierType bmasks[4] =
		{0, GDK_BUTTON1_MASK, GDK_BUTTON2_MASK, GDK_BUTTON3_MASK};
	GdkModifierType state;
	int x, y, nx, ny, zoom = 1, scale = 1;

	if (!unreal_move) lastdx = lastdy = 0;
	if (!mem_img[CHN_IMAGE]) return;

	/* !!! This uses the fact that zoom factor is either N or 1/N !!! */
	if (can_zoom < 1.0) zoom = rint(1.0 / can_zoom);
	else scale = rint(can_zoom);

	gdk_window_get_pointer(drawing_canvas->window, &x, &y, &state);

	nx = ((x - margin_main_x) * zoom) / scale + lastdx + dx;
	ny = ((y - margin_main_y) * zoom) / scale + lastdy + dy;

	if (button) /* Clicks simulated without extra movements */
	{
		state |= bmasks[button];
		mouse_event(GDK_BUTTON_PRESS, nx, ny, state, button, 1.0, 1);
		state ^= bmasks[button];
		mouse_event(GDK_BUTTON_RELEASE, nx, ny, state, button, 1.0, 1);
		return;
	}

	if ((state & (GDK_BUTTON1_MASK | GDK_BUTTON3_MASK)) ==
		(GDK_BUTTON1_MASK | GDK_BUTTON3_MASK)) button = 13;
	else if (state & GDK_BUTTON1_MASK) button = 1;
	else if (state & GDK_BUTTON3_MASK) button = 3;
	else if (state & GDK_BUTTON2_MASK) button = 2;

	if (zoom > 1) /* Fine control required */
	{
		lastdx += dx; lastdy += dy;
		mouse_event(GDK_MOTION_NOTIFY, nx, ny, state, button, 1.0, 1);

		/* Nudge cursor when needed */
		if ((abs(lastdx) >= zoom) || (abs(lastdy) >= zoom))
		{
			dx = lastdx * can_zoom;
			dy = lastdy * can_zoom;
			lastdx -= dx * zoom;
			lastdy -= dy * zoom;
			unreal_move = 3;
			/* Event can be delayed or lost */
			move_mouse_relative(dx, dy);
		}
		else unreal_move = 2;
	}
	else /* Real mouse is precise enough */
	{
		unreal_move = 1;

		/* Simulate movement if failed to actually move mouse */
		if (!move_mouse_relative(dx * scale, dy * scale))
		{
			lastdx += dx; lastdy += dy;
			mouse_event(GDK_MOTION_NOTIFY, nx, ny, state, button, 1.0, 1);
		}
	}
}

int check_arrows(int action)
{
	int mv = mem_nudge;

	if ( marq_status == MARQUEE_DONE )
	{		// User is selecting so allow CTRL+arrow keys to resize the marquee
		switch (action)
		{
			case ACT_LR_LEFT: mv = 1;
			case ACT_LR_2LEFT:
				resize_marquee(-mv, 0);
				return 1;
			case ACT_LR_RIGHT: mv = 1;
			case ACT_LR_2RIGHT:
				resize_marquee(mv, 0);
				return 1;
			case ACT_LR_DOWN: mv = 1;
			case ACT_LR_2DOWN:
				resize_marquee(0, mv);
				return 1;
			case ACT_LR_UP: mv = 1;
			case ACT_LR_2UP:
				resize_marquee(0, -mv);
				return 1;
		}
	}

	switch (action)
	{
	case ACT_SEL_LEFT: mv = 1;
	case ACT_SEL_2LEFT:
		return (move_arrows(&marq_x1, &marq_x2, -mv));
	case ACT_SEL_RIGHT: mv = 1;
	case ACT_SEL_2RIGHT:
		return (move_arrows(&marq_x1, &marq_x2, mv));
	case ACT_SEL_DOWN: mv = 1;
	case ACT_SEL_2DOWN:
		return (move_arrows(&marq_y1, &marq_y2, mv));
	case ACT_SEL_UP: mv = 1;
	case ACT_SEL_2UP:
		return (move_arrows(&marq_y1, &marq_y2, -mv));
	}
	return (0);
}

gint check_zoom_keys_real(int action)
{
	static double zals[9] = { 0.1, 0.25, 0.5, 1, 4, 8, 12, 16, 20 };

	switch (action)
	{
	case ACT_ZOOM_IN:
		zoom_in(); break;
	case ACT_ZOOM_OUT:
		zoom_out(); break;
	case ACT_ZOOM_01:
	case ACT_ZOOM_025:
	case ACT_ZOOM_05:
	case ACT_ZOOM_1:
	case ACT_ZOOM_4:
	case ACT_ZOOM_8:
	case ACT_ZOOM_12:
	case ACT_ZOOM_16:
	case ACT_ZOOM_20:
		align_size(zals[action - ACT_ZOOM_01]);
		break;
	default: return FALSE;
	}
	return TRUE;
}

gint check_zoom_keys(int action)
{
	if (action == ACT_QUIT)
	{
		quit_all( NULL, NULL );
	}

	if (check_zoom_keys_real(action)) return TRUE;

	switch (action)
	{
	case ACT_BRCOSA:
		pressed_brcosa(NULL, NULL); return TRUE;
	case ACT_PAN:
		pressed_pan(NULL, NULL); return TRUE;
	case ACT_CROP:
		pressed_crop(NULL, NULL); return TRUE;
	}
	return FALSE;
}

#define _C (GDK_CONTROL_MASK)
#define _S (GDK_SHIFT_MASK)
#define _A (GDK_MOD1_MASK)
#define _CS (GDK_CONTROL_MASK | GDK_SHIFT_MASK)
#define _CSA (GDK_CONTROL_MASK | GDK_SHIFT_MASK | GDK_MOD1_MASK)

typedef struct {
	char *actname;
	int action, key, kmask, kflags;
} key_action;

static key_action main_keys[] = {
	{"QUIT",	ACT_QUIT, GDK_q, 0, 0},
	{"",		ACT_QUIT, GDK_Q, 0, 0},
	{"ZOOM_IN",	ACT_ZOOM_IN, GDK_plus, _CS, 0},
	{"",		ACT_ZOOM_IN, GDK_KP_Add, _CS, 0},
//	{"",		ACT_ZOOM_IN, GDK_equal, _CS, 0},
	{"ZOOM_OUT",	ACT_ZOOM_OUT, GDK_minus, _CS, 0},
	{"",		ACT_ZOOM_OUT, GDK_KP_Subtract, _CS, 0},
	{"ZOOM_01",	ACT_ZOOM_01, GDK_KP_1, _CS, 0},
	{"",		ACT_ZOOM_01, GDK_1, _CS, 0},
	{"ZOOM_025",	ACT_ZOOM_025, GDK_KP_2, _CS, 0},
	{"",		ACT_ZOOM_025, GDK_2, _CS, 0},
	{"ZOOM_05",	ACT_ZOOM_05, GDK_KP_3, _CS, 0},
	{"",		ACT_ZOOM_05, GDK_3, _CS, 0},
	{"ZOOM_1",	ACT_ZOOM_1, GDK_KP_4, _CS, 0},
	{"",		ACT_ZOOM_1, GDK_4, _CS, 0},
	{"ZOOM_4",	ACT_ZOOM_4, GDK_KP_5, _CS, 0},
	{"",		ACT_ZOOM_4, GDK_5, _CS, 0},
	{"ZOOM_8",	ACT_ZOOM_8, GDK_KP_6, _CS, 0},
	{"",		ACT_ZOOM_8, GDK_6, _CS, 0},
	{"ZOOM_12",	ACT_ZOOM_12, GDK_KP_7, _CS, 0},
	{"",		ACT_ZOOM_12, GDK_7, _CS, 0},
	{"ZOOM_16",	ACT_ZOOM_16, GDK_KP_8, _CS, 0},
	{"",		ACT_ZOOM_16, GDK_8, _CS, 0},
	{"ZOOM_20",	ACT_ZOOM_20, GDK_KP_9, _CS, 0},
	{"",		ACT_ZOOM_20, GDK_9, _CS, 0},
	{"BRCOSA",	ACT_BRCOSA, GDK_Insert, _CS, 0},
	{"PAN",		ACT_PAN, GDK_End, _CS, 0},
	{"CROP",	ACT_CROP, GDK_Delete, _CS, 0},
	{"PAINT",	ACT_PAINT, GDK_F4, _CSA, 0},
	{"SELECT",	ACT_SELECT, GDK_F9, _CSA, 0},
	{"SEL_2LEFT",	ACT_SEL_2LEFT, GDK_Left, _CS, _S},
	{"",		ACT_SEL_2LEFT, GDK_KP_Left, _CS, _S},
	{"SEL_2RIGHT",	ACT_SEL_2RIGHT, GDK_Right, _CS, _S},
	{"",		ACT_SEL_2RIGHT, GDK_KP_Right, _CS, _S},
	{"SEL_2DOWN",	ACT_SEL_2DOWN, GDK_Down, _CS, _S},
	{"",		ACT_SEL_2DOWN, GDK_KP_Down, _CS, _S},
	{"SEL_2UP",	ACT_SEL_2UP, GDK_Up, _CS, _S},
	{"",		ACT_SEL_2UP, GDK_KP_Up, _CS, _S},
	{"SEL_LEFT",	ACT_SEL_LEFT, GDK_Left, _CS, 0},
	{"",		ACT_SEL_LEFT, GDK_KP_Left, _CS, 0},
	{"SEL_RIGHT",	ACT_SEL_RIGHT, GDK_Right, _CS, 0},
	{"",		ACT_SEL_RIGHT, GDK_KP_Right, _CS, 0},
	{"SEL_DOWN",	ACT_SEL_DOWN, GDK_Down, _CS, 0},
	{"",		ACT_SEL_DOWN, GDK_KP_Down, _CS, 0},
	{"SEL_UP",	ACT_SEL_UP, GDK_Up, _CS, 0},
	{"",		ACT_SEL_UP, GDK_KP_Up, _CS, 0},
	{"LR_2LEFT",	ACT_LR_2LEFT, GDK_Left, _CS, _CS},
	{"",		ACT_LR_2LEFT, GDK_KP_Left, _CS, _CS},
	{"LR_2RIGHT",	ACT_LR_2RIGHT, GDK_Right, _CS, _CS},
	{"",		ACT_LR_2RIGHT, GDK_KP_Right, _CS, _CS},
	{"LR_2DOWN",	ACT_LR_2DOWN, GDK_Down, _CS, _CS},
	{"",		ACT_LR_2DOWN, GDK_KP_Down, _CS, _CS},
	{"LR_2UP",	ACT_LR_2UP, GDK_Up, _CS, _CS},
	{"",		ACT_LR_2UP, GDK_KP_Up, _CS, _CS},
	{"LR_LEFT",	ACT_LR_LEFT, GDK_Left, _CS, _C},
	{"",		ACT_LR_LEFT, GDK_KP_Left, _CS, _C},
	{"LR_RIGHT",	ACT_LR_RIGHT, GDK_Right, _CS, _C},
	{"",		ACT_LR_RIGHT, GDK_KP_Right, _CS, _C},
	{"LR_DOWN",	ACT_LR_DOWN, GDK_Down, _CS, _C},
	{"",		ACT_LR_DOWN, GDK_KP_Down, _CS, _C},
	{"LR_UP",	ACT_LR_UP, GDK_Up, _CS, _C},
	{"",		ACT_LR_UP, GDK_KP_Up, _CS, _C},
	{"ESC",		ACT_ESC, GDK_Escape, _CS, 0},
	{"SCALE",	ACT_SCALE, GDK_Page_Up, _CS, 0},
	{"SIZE",	ACT_SIZE, GDK_Page_Down, _CS, 0},
	{"COMMIT",	ACT_COMMIT, GDK_Return, 0, 0},
	{"",		ACT_COMMIT, GDK_KP_Enter, 0, 0},
	{"RCLICK",	ACT_RCLICK, GDK_BackSpace, 0, 0},
	{"ARROW",	ACT_ARROW, GDK_a, _C, 0},
//	{"",		ACT_ARROW, GDK_A, _C, 0},
	{"ARROW3",	ACT_ARROW3, GDK_s, _C, 0},
	{NULL,		0, 0, 0, 0}
};

static guint main_keycodes[sizeof(main_keys) / sizeof(key_action)];

static void fill_keycodes()
{
	int i;

	for (i = 0; main_keys[i].action; i++)
	{
		main_keycodes[i] = keyval_key(main_keys[i].key);
	}
}

int wtf_pressed(GdkEventKey *event)
{
	int i, cmatch = 0;
	guint realkey = real_key(event);
	guint lowkey = gdk_keyval_to_lower(event->keyval);

	for (i = 0; main_keys[i].action; i++)
	{
		/* Relevant modifiers should match first */
		if ((event->state & main_keys[i].kmask) != main_keys[i].kflags)
			continue;
		/* Let keyval have priority; this is also a workaround for
		 * GTK2 bug #136280 */
		if (lowkey == main_keys[i].key) return (main_keys[i].action);
		/* Let keycodes match when keyvals don't */
		if (realkey == main_keycodes[i]) cmatch = main_keys[i].action;
	}
	/* Return keycode match, if any */
	return (cmatch);
}

gint handle_keypress( GtkWidget *widget, GdkEventKey *event )
{
	int change, action;

	action = wtf_pressed(event);
	if (!action) return (FALSE);

	if (check_zoom_keys(action)) return TRUE;		// Check HOME/zoom keys

	if (marq_status > MARQUEE_NONE)
	{
		if (check_arrows(action) == 1)
		{
			update_menus();
			return TRUE;
		}
	}
	change = mem_nudge;

	switch (action)
	{
	case ACT_SEL_LEFT: change = 1;
	case ACT_SEL_2LEFT:
		move_mouse(-change, 0, 0);
		return (TRUE);
	case ACT_SEL_RIGHT: change = 1;
	case ACT_SEL_2RIGHT:
		move_mouse(change, 0, 0);
		return (TRUE);
	case ACT_SEL_DOWN: change = 1;
	case ACT_SEL_2DOWN:
		move_mouse(0, change, 0);
		return (TRUE);
	case ACT_SEL_UP: change = 1;
	case ACT_SEL_2UP:
		move_mouse(0, -change, 0);
		return (TRUE);
	case ACT_ESC:
		if ((tool_type == TOOL_SELECT) || (tool_type == TOOL_POLYGON))
			pressed_select_none(NULL, NULL);
		return TRUE;
	case ACT_COMMIT:
		if (marq_status >= MARQUEE_PASTE)
		{
			commit_paste(TRUE);
			pen_down = 0;	// Ensure each press of enter is a new undo level
		}
		else move_mouse(0, 0, 1);
		return TRUE;
	case ACT_RCLICK:
		if (marq_status < MARQUEE_PASTE) move_mouse(0, 0, 3);
		return TRUE;
	default:
		return (FALSE);
	}
	/* Finalize colour-change */
	if (mem_channel == CHN_IMAGE)
	{
		mem_col_A24 = mem_pal[mem_col_A];
		mem_col_B24 = mem_pal[mem_col_B];
		init_pal();
	}
	return TRUE;
}

gint destroy_signal( GtkWidget *widget, GdkEvent *event, gpointer data )
{
	quit_all( NULL, NULL );

	return FALSE;
}

int check_for_changes(gchar *type)	// return: 1=SAVE, 2=CANCEL, 10=ESCAPE, -10=NOT CHECKED
{
	int i = -10;
	char *warning = _("This image contains changes that have not been saved.  Do you want to save these changes?");

	if ( mem_changed == 1 )
	{
		i = alert_box_stock( _("Warning"), warning, GTK_STOCK_SAVE, type, NULL );
		if ( i==1 )
		{
			pressed_save_file( NULL, NULL );
		}
	}

	return i;
}

gint delete_event( GtkWidget *widget, GdkEvent *event, gpointer data )
{
	int j;

	close_after_save = TRUE;
	j = check_for_changes(GTK_STOCK_QUIT);

	if ( j==2 || j==-10 )
	{
		gtk_main_quit ();
		return FALSE;
	}
	else
	{
		close_after_save = FALSE;
		return TRUE;
	}
}

/* Mouse event from button/motion on the canvas */
static void mouse_event(int event, int x0, int y0, guint state, guint button,
	gdouble pressure, int mflag)
{
	static int tool_fixx = -1, tool_fixy = -1;	// Fixate on axis
	GdkCursor *temp_cursor = NULL;
	GdkCursorType pointers[] = {GDK_TOP_LEFT_CORNER, GDK_TOP_RIGHT_CORNER,
		GDK_BOTTOM_LEFT_CORNER, GDK_BOTTOM_RIGHT_CORNER};
	int new_cursor;
	int i, pixel, x, y, ox, oy;//, tox = tool_ox, toy = tool_oy;


	ox = x = x0 < 0 ? 0 : x0 >= mem_width ? mem_width - 1 : x0;
	oy = y = y0 < 0 ? 0 : y0 >= mem_height ? mem_height - 1 : y0;

	/* ****** Release-event-specific code ****** */
	if (event == GDK_BUTTON_RELEASE)
	{
		pen_down = 0;

		if (((tool_type == TOOL_SELECT) /*|| (tool_type == TOOL_POLYGON)*/) &&
			(button == 1))
		{
			if (marq_status == MARQUEE_SELECTING)
				marq_status = MARQUEE_DONE;
			if (marq_status == MARQUEE_PASTE_DRAG)
				marq_status = MARQUEE_PASTE;
			cursor_corner = -1;
		}

		update_menus();

		return;
	}

	/* ****** Common click/motion handling code ****** */

	if (!mflag) /* Coordinate fixation */
	{
		if ((tool_fixy < 0) && ((state & _CS) == _CS))
		{
			tool_fixx = -1;
			tool_fixy = y;
		}
		if ((tool_fixx < 0) && ((state & _CS) == _S))
			tool_fixx = x;
		if (!(state & _S)) tool_fixx = tool_fixy = -1;
		if (!(state & _C)) tool_fixy = -1;
	}
	/* No use when moving cursor by keyboard */
	else if (event == GDK_MOTION_NOTIFY) tool_fixx = tool_fixy = -1;

	if (tool_fixx > 0) x = x0 = tool_fixx;
	if (tool_fixy > 0) y = y0 = tool_fixy;

	while ((state & _CS) == _C)	// Set colour A/B
	{
		if ((button != 1) && (button != 3)) break;
		pixel = get_pixel(ox, oy);

		if (button == 1 || button == 3)
		{
			if (PNG_2_INT(mem_col_A24) == pixel) break;
			mem_col_A24.red = INT_2_R(pixel);
			mem_col_A24.green = INT_2_G(pixel);
			mem_col_A24.blue = INT_2_B(pixel);
//		}
//		else /* button == 3 */
//		{
//			if (PNG_2_INT(mem_col_B24) == pixel) break;
			mem_col_B24.red = INT_2_R(pixel);
			mem_col_B24.green = INT_2_G(pixel);
			mem_col_B24.blue = INT_2_B(pixel);
		}
		update_cols();
		toolbar_palette_refresh();	// Update swatch
		break;
	}

	if ((state & _CS) == _C); /* Done above */

	else if ((button == 2) || ((button == 3) && (state & _S)))
		set_zoom_centre(ox, oy);

	/* Pure moves are handled elsewhere */
	else if (button) tool_action(event, x, y, button, pressure);

	/* ****** Now to mouse-move-specific part ****** */

	if (event != GDK_MOTION_NOTIFY) return;

	if ( tool_type == TOOL_SELECT /*|| tool_type == TOOL_POLYGON*/ )
	{
		if ( marq_status == MARQUEE_DONE )
		{
			i = close_to(x, y);
			if ( i!=cursor_corner ) // Stops excessive CPU/flickering
			{
				cursor_corner = i;
				temp_cursor = gdk_cursor_new(pointers[i]);
				gdk_window_set_cursor(drawing_canvas->window, temp_cursor);
				gdk_cursor_destroy(temp_cursor);
			}
		}
		if ( marq_status >= MARQUEE_PASTE )
		{
			new_cursor = 0;		// Cursor = normal
			if ( x>=marq_x1 && x<=marq_x2 && y>=marq_y1 && y<=marq_y2 )
				new_cursor = 1;		// Cursor = 4 way arrow

			if ( new_cursor != cursor_corner ) // Stops flickering on slow hardware
			{
				if ( new_cursor == 0 )
					set_cursor();
				else
					gdk_window_set_cursor( drawing_canvas->window, move_cursor );
				cursor_corner = new_cursor;
			}
		}
	}

///	TOOL PERIMETER BOX UPDATES

	if (perim_status > 0) clear_perim();	// Remove old perimeter box

	if (tool_size * can_zoom > 4)
	{
		perim_x = x - (tool_size >> 1);
		perim_y = y - (tool_size >> 1);
		perim_s = tool_size;
		repaint_perim();			// Repaint 4 sides
	}
}

static gint canvas_button( GtkWidget *widget, GdkEventButton *event )
{
	int x, y, zoom = 1, scale = 1, pflag = event->type != GDK_BUTTON_RELEASE;
	gdouble pressure = 1.0;

	if (pflag) /* For button press events only */
	{
		if (!mem_img[CHN_IMAGE]) return (TRUE);

	}

	/* !!! This uses the fact that zoom factor is either N or 1/N !!! */
	if (can_zoom < 1.0) zoom = rint(1.0 / can_zoom);
	else scale = rint(can_zoom);

	x = ((int)(event->x - margin_main_x) * zoom) / scale;
	y = ((int)(event->y - margin_main_y) * zoom) / scale;
	mouse_event(event->type, x, y, event->state, event->button,
		pressure, unreal_move & 1);

	return (pflag);
}


static gint canvas_left(GtkWidget *widget, GdkEventCrossing *event, gpointer user_data)
{
	/* Skip grab/ungrab related events */
	if (event->mode != GDK_CROSSING_NORMAL) return (FALSE);

	/* Only do this if we have an image */
	if (!mem_img[CHN_IMAGE]) return (FALSE);
	if (perim_status > 0) clear_perim();

	return (FALSE);
}



#define GREY_W 153
#define GREY_B 102
void render_background(unsigned char *rgb, int x0, int y0, int wid, int hgt, int fwid)
{
	int i, j, k, scale, dx, dy, step, ii, jj, ii0, px, py;
	int xwid = 0, xhgt = 0, wid3 = wid * 3;
	static unsigned char greyz[2] = {GREY_W, GREY_B};

	/* !!! This uses the fact that zoom factor is either N or 1/N !!! */
	if (can_zoom < 1.0) step = 6;
	else
	{
		scale = rint(can_zoom);
		step = scale < 4 ? 6 : scale == 4 ? 8 : scale;
	}
	dx = x0 % step;
	dy = y0 % step;
	
	py = (x0 / step + y0 / step) & 1;
	if (hgt + dy > step)
	{
		jj = step - dy;
		xhgt = (hgt + dy) % step;
		if (!xhgt) xhgt = step;
		hgt -= xhgt;
		xhgt -= step;
	}
	else jj = hgt--;
	if (wid + dx > step)
	{
		ii0 = step - dx;
		xwid = (wid + dx) % step;
		if (!xwid) xwid = step;
		wid -= xwid;
		xwid -= step;
	}
	else ii0 = wid--;
	fwid *= 3;

	for (j = 0; ; jj += step)
	{
		if (j >= hgt)
		{
			if (j > hgt) break;
			jj += xhgt;
		}
		px = py;
		ii = ii0;
		for (i = 0; ; ii += step)
		{
			if (i >= wid)
			{
				if (i > wid) break;
				ii += xwid;
			}
			k = (ii - i) * 3;
			memset(rgb, greyz[px], k);
			rgb += k;
			px ^= 1;
			i = ii;
		}
		rgb += fwid - wid3;
		for(j++; j < jj; j++)
		{
			memcpy(rgb, rgb - fwid, wid3);
			rgb += fwid;
		}
		py ^= 1;
	}
}

/* This is for a faster way to pass parameters into render_row() */
typedef struct {
	int dx;
	int width;
	int xwid;
	int zoom;
	int scale;
	int mw;
	int opac;
	int xpm;
	int bpp;
	png_color *pal;
} renderstate;

static renderstate rr;

void setup_row(int x0, int width, double czoom, int mw, int xpm, int opac,
	int bpp, png_color *pal)
{
	/* Horizontal zoom */
	/* !!! This uses the fact that zoom factor is either N or 1/N !!! */
	if (czoom <= 1.0)
	{
		rr.zoom = rint(1.0 / czoom);
		rr.scale = 1;
		x0 = 0;
	}
	else
	{
		rr.zoom = 1;
		rr.scale = rint(czoom);
		x0 %= rr.scale;
	}
	if (width + x0 > rr.scale)
	{
		rr.dx = rr.scale - x0;
		x0 = (width + x0) % rr.scale;
		if (!x0) x0 = rr.scale;
		width -= x0;
		rr.xwid = x0 - rr.scale;
	}
	else
	{
		rr.dx = width--;
		rr.xwid = 0;
	}
	rr.width = width;
	rr.mw = mw;

	if ((xpm > -1) && (bpp == 3)) xpm = PNG_2_INT(pal[xpm]);
	rr.xpm = xpm;
	rr.opac = opac;

	rr.bpp = bpp;
	rr.pal = pal;
}

void render_row(unsigned char *rgb, chanlist base_img, int x, int y,
	chanlist xtra_img)
{
	unsigned char *src = NULL, *dest;
	int i, ii, ds = rr.zoom * 3;

	if (xtra_img)
	{
		src = xtra_img[CHN_IMAGE];
	}
	if (!src) src = base_img[CHN_IMAGE] + (rr.mw * y + x) * rr.bpp;
	dest = rgb;
	ii = rr.dx;

	for (i = 0; ; ii += rr.scale , src += ds)
	{
		if (i >= rr.width)
		{
			if (i > rr.width) break;
			ii += rr.xwid;
		}
		for(; i < ii; i++)
		{
			dest[0] = src[0];
			dest[1] = src[1];
			dest[2] = src[2];
			dest += 3;
		}
	}
}

void overlay_row(unsigned char *rgb, chanlist base_img, int x, int y,
	chanlist xtra_img)
{
	unsigned char *alpha, *sel, *mask, *dest;
	int i, j, k, ii, dw, opA, opS, opM, t0, t1, t2, t3;

	if (xtra_img)
	{
		alpha = xtra_img[CHN_ALPHA];
		sel = xtra_img[CHN_SEL];
		mask = xtra_img[CHN_MASK];
	}
	else alpha = sel = mask = NULL;
	j = rr.mw * y + x;
	if (!alpha && base_img[CHN_ALPHA]) alpha = base_img[CHN_ALPHA] + j;
	if (!sel && base_img[CHN_SEL]) sel = base_img[CHN_SEL] + j;
	if (!mask && base_img[CHN_MASK]) mask = base_img[CHN_MASK] + j;

	k=256;
	opA = 0;
	opS = 0;
	opM = 0;

	/* Nothing to do - don't waste time then */
	j = opA + opS + opM;
	if (!k || !j) return;

	opA = (k * opA) / j;
	opS = (k * opS) / j;
	opM = (k * opM) / j;
	if (!(opA + opS + opM)) return;

	dest = rgb;
	ii = rr.dx;
	for (i = dw = 0; ; ii += rr.scale , dw += rr.zoom)
	{
		if (i >= rr.width)
		{
			if (i > rr.width) break;
			ii += rr.xwid;
		}
		t0 = t1 = t2 = t3 = 0;
		j = (256 * 255) - t0;

		k = t1 + j * dest[0];
		dest[0] = (k + (k >> 8) + 0x100) >> 16;
		k = t2 + j * dest[1];
		dest[1] = (k + (k >> 8) + 0x100) >> 16;
		k = t3 + j * dest[2];
		dest[2] = (k + (k >> 8) + 0x100) >> 16;
or_s:
		dest += 3;
		if (++i >= ii) continue;
		dest[0] = *(dest - 3);
		dest[1] = *(dest - 2);
		dest[2] = *(dest - 1);
		goto or_s;
	}
}

/* Specialized renderer for irregular overlays */
void overlay_preview(unsigned char *rgb, unsigned char *map, int col, int opacity)
{
	unsigned char *dest, crgb[3] = {INT_2_R(col), INT_2_G(col), INT_2_B(col)};
	int i, j, k, ii, dw;

	dest = rgb;
	ii = rr.dx;
	for (i = dw = 0; ; ii += rr.scale , dw += rr.zoom)
	{
		if (i >= rr.width)
		{
			if (i > rr.width) break;
			ii += rr.xwid;
		}
		k = opacity * map[dw];
		k = (k + (k >> 8) + 1) >> 8;
//op_as:
		j = 255 * dest[0] + k * (crgb[0] - dest[0]);
		dest[0] = (j + (j >> 8) + 1) >> 8;
		j = 255 * dest[1] + k * (crgb[1] - dest[1]);
		dest[1] = (j + (j >> 8) + 1) >> 8;
		j = 255 * dest[2] + k * (crgb[2] - dest[2]);
		dest[2] = (j + (j >> 8) + 1) >> 8;
op_s:
		dest += 3;
		if (++i >= ii) continue;
		dest[0] = *(dest - 3);
		dest[1] = *(dest - 2);
		dest[2] = *(dest - 1);
		goto op_s;
	}
}

void repaint_paste( int px1, int py1, int px2, int py2 )
{
	chanlist tlist;
	unsigned char *rgb, *tmp, *pix, *mask, *alpha, *mask0;
	unsigned char *clip_alpha, *clip_image, *t_alpha = NULL;
	int i, j, l, pw, ph, pw3, lop = 255, /*lx = 0, ly = 0,*/ bpp = 3;//MEM_BPP;
	int zoom, scale, pww, j0, jj, dx, di, dc, /*xpm = mem_xpm_trans,*/ opac=255;


	if ((px1 > px2) || (py1 > py2)) return;

	/* !!! This uses the fact that zoom factor is either N or 1/N !!! */
	zoom = scale = 1;
	if (can_zoom < 1.0) zoom = rint(1.0 / can_zoom);
	else scale = rint(can_zoom);

	/* Check bounds */
	i = (marq_x1 * scale + zoom - 1) / zoom;
	j = (marq_y1 * scale + zoom - 1) / zoom;
	if (px1 < i) px1 = i;
	if (py1 < j) py1 = j;
	i = (marq_x2 * scale) / zoom + scale - 1;
	j = (marq_y2 * scale) / zoom + scale - 1;
	if (px2 > i) px2 = i;
	if (py2 > j) py2 = j;

	pw = px2 - px1 + 1; ph = py2 - py1 + 1;
	if ((pw <= 0) || (ph <= 0)) return;

	rgb = malloc(i = pw * ph * 3);
	if (!rgb) return;
	memset(rgb, mem_background, i);

	/* Horizontal zoom */
	if (scale == 1)
	{
		dx = px1 * zoom;
		l = (pw - 1) * zoom + 1;
		pww = pw;
	}
	else
	{
		dx = px1 / scale;
		pww = l = px2 / scale - dx + 1;
	}

	i = l * (bpp + 2);
	pix = malloc(i);
	if (!pix)
	{
		free(rgb);
		return;
	}
	alpha = pix + l * bpp;
	mask = alpha + l;

	memset(tlist, 0, sizeof(chanlist));
	tlist[mem_channel] = pix;
	clip_image = mem_clipboard;
	clip_alpha = NULL;
	if ((mem_channel == CHN_IMAGE) && mem_img[CHN_ALPHA] )//&& !channel_dis[CHN_ALPHA])
	{
		clip_alpha = mem_clip_alpha;
		if (!clip_alpha)
		{
			t_alpha = malloc(l);
			if (!t_alpha)
			{
				free(pix);
				free(rgb);
				return;
			}
		}
	}
	if (clip_alpha || t_alpha) tlist[CHN_ALPHA] = alpha;

	/* Setup opacity mode & mask */
	mask0 = NULL;
	if ((mem_channel <= CHN_ALPHA) && mem_img[CHN_MASK])
		mask0 = mem_img[CHN_MASK];
	{
		render_background(rgb, px1, py1, pw, ph, pw);
	}

	setup_row(px1, pw, can_zoom, mem_width, -1/*xpm*/, lop, mem_img_bpp, mem_pal);
	j0 = -1; tmp = rgb; pw3 = pw * 3;
	for (jj = 0; jj < ph; jj++ , tmp += pw3)
	{
		j = ((py1 + jj) * zoom) / scale;
		if (j != j0)
		{
			j0 = j;
			di = mem_width * j + dx;
			dc = mem_clip_w * (j - marq_y1) + dx - marq_x1;
			if (tlist[CHN_ALPHA])
				memcpy(alpha, mem_img[CHN_ALPHA] + di, l);
			prep_mask(0, zoom, pww, mask, mask0 ? mask0 + di : NULL,
				mem_img[CHN_IMAGE] + di * mem_img_bpp);
			process_mask(0, zoom, pww, mask, alpha, mem_img[CHN_ALPHA] + di,
				clip_alpha ? clip_alpha + dc : t_alpha,
				mem_clip_mask ? mem_clip_mask + dc : NULL, opac, 0);
			if (clip_image)
			{
				if (mem_img[mem_channel])
					memcpy(pix, mem_img[mem_channel] +
						di * bpp, l * bpp);
				else memset(pix, 0, l * bpp);
				process_img(0, zoom, pww, mask, pix,
					mem_img[mem_channel] + di * bpp,
					mem_clipboard + dc * mem_clip_bpp,
					opac, mem_clip_bpp);
			}
		}
		render_row(tmp, mem_img, dx, j, tlist);
		overlay_row(tmp, mem_img, dx, j, tlist);
	}

	gdk_draw_rgb_image(drawing_canvas->window, drawing_canvas->style->black_gc,
			margin_main_x + px1, margin_main_y + py1,
			pw, ph, GDK_RGB_DITHER_NONE, rgb, pw3);
	free(pix);
	free(rgb);
	free(t_alpha);
}



void main_render_rgb(unsigned char *rgb, int px, int py, int pw, int ph)
{
	chanlist tlist;
	unsigned char *mask0 = NULL, *pvi = NULL, *pvm = NULL, *pvx = NULL;
	int pw2, ph2, px2 = px - margin_main_x, py2 = py - margin_main_y;
	int j, jj, j0, l, lx, dx, pww, zoom = 1, scale = 1, nix = 0, niy = 0;
	int lop = 255;//, xpm = mem_xpm_trans;
	unsigned char *gtemp = NULL;

	if (can_zoom < 1.0) zoom = rint(1.0 / can_zoom);
	else scale = rint(can_zoom);

	pw2 = pw + px2;
	ph2 = ph + py2;

	if (px2 < 0) nix = -px2;
	if (py2 < 0) niy = -py2;
	rgb += (pw * niy + nix) * 3;

	// Update image + blank space outside
	j = (mem_width * scale + zoom - 1) / zoom;
	jj = (mem_height * scale + zoom - 1) / zoom;
	if (pw2 > j) pw2 = j;
	if (ph2 > jj) ph2 = jj;
	px2 += nix; py2 += niy;
	pw2 -= px2; ph2 -= py2;

	if ((pw2 < 1) || (ph2 < 1)) return;

	dx = (px2 * zoom) / scale;
	memset(tlist, 0, sizeof(chanlist));
//	if (!channel_dis[CHN_MASK])
		mask0 = mem_img[CHN_MASK];
	pww = pw2;
	if (scale > 1) lx = pww = (px2 + pw2 - 1) / scale - dx + 1;
	else lx = (pw2 - 1) * zoom + 1;

	/* Color transform preview */
	if (mem_preview && (mem_img_bpp == 3))
	{
		pvm = malloc(lx * 4);
		if (pvm)
		{
			pvi = pvm + lx;
			tlist[CHN_IMAGE] = pvi;
		}
	}

	setup_row(px2, pw2, can_zoom, mem_width, -1,/*xpm,*/ lop,
//		gtemp && grstate.rgb ? 3 : mem_img_bpp, mem_pal);
		mem_img_bpp, mem_pal);
 	j0 = -1; pw *= 3; pw2 *= 3;
	for (jj = 0; jj < ph2; jj++ , rgb += pw)
	{
		j = ((py2 + jj) * zoom) / scale;
		if (j != j0)
		{
			j0 = j;
			l = mem_width * j + dx;

			/* Color transform preview */
			if (pvm)
			{
				prep_mask(0, zoom, pww, pvm, mask0 ? mask0 + l : NULL,
					mem_img[CHN_IMAGE] + l * 3);
				do_transform(0, zoom, pww, pvm, pvi,
					mem_img[CHN_IMAGE] + l * 3);
			}
		}
		render_row(rgb, mem_img, dx, j, tlist);
		if (!pvx) overlay_row(rgb, mem_img, dx, j, tlist);
//		else overlay_preview(rgb, pvx, csel_preview, csel_preview_a);
	}
	free(pvm);
	free(pvx);
	free(gtemp);
}

/* Draw grid on rgb memory */
void draw_grid(unsigned char *rgb, int x, int y, int w, int h)
{
	int i, j, k, dx, dy, step, step3;
	unsigned char *tmp;


	if (!mem_show_grid || (can_zoom < mem_grid_min)) return;
	step = can_zoom;

	dx = (x - margin_main_x) % step;
	if (dx < 0) dx += step;
	dy = (y - margin_main_y) % step;
	if (dy < 0) dy += step;
	if (dx) dx = (step - dx) * 3;
	w *= 3;

	for (k = dy , i = 0; i < h; i++)
	{
		tmp = rgb + i * w;
		if (!k) /* Filled line */
		{
			j = 0; step3 = 3;
		}
		else /* Spaced dots */
		{
			j = dx; step3 = step * 3;
		}
		k = (k + 1) % step;
		for (; j < w; j += step3)
		{
			tmp[j + 0] = mem_grid_rgb[0];
			tmp[j + 1] = mem_grid_rgb[1];
			tmp[j + 2] = mem_grid_rgb[2];
		}
	}
}

void repaint_canvas( int px, int py, int pw, int ph )
{
	unsigned char *rgb;
	int pw2, ph2, /*lx = 0, ly = 0,*/ rx1, ry1, rx2, ry2, rpx, rpy;
	int i, j, zoom = 1, scale = 1;

	if (zoom_flag == 1) return;		// Stops excess jerking in GTK+1 when zooming

	if (px < 0)
	{
		pw += px; px = 0;
	}
	if (py < 0)
	{
		ph += py; py = 0;
	}

	if ((pw <= 0) || (ph <= 0)) return;
	rgb = malloc(i = pw * ph * 3);
	if (!rgb) return;
	memset(rgb, mem_background, i);

	if (can_zoom < 1.0) zoom = rint(1.0 / can_zoom);
	else scale = rint(can_zoom);

	rpx = px - margin_main_x;
	rpy = py - margin_main_y;
	pw2 = rpx + pw - 1;
	ph2 = rpy + ph - 1;
	main_render_rgb(rgb, px, py, pw, ph);

	draw_grid(rgb, px, py, pw, ph);

	gdk_draw_rgb_image(drawing_canvas->window, drawing_canvas->style->black_gc,
		px, py, pw, ph, GDK_RGB_DITHER_NONE, rgb, pw * 3);

	free(rgb);

	/* Add clipboard image to redraw if needed */
	if (marq_status >= MARQUEE_PASTE)
	{
		/* Enforce image bounds */
		if (rpx < 0) rpx = 0;
		if (rpy < 0) rpy = 0;
		i = ((mem_width + zoom - 1) * scale) / zoom;
		j = ((mem_height + zoom - 1) * scale) / zoom;
		if (pw2 >= i) pw2 = i - 1;
		if (ph2 >= j) ph2 = j - 1;

		/* Check paste bounds for intersection, but leave actually
		 * enforcing them to repaint_paste() */
		rx1 = (marq_x1 * scale + zoom - 1) / zoom;
		ry1 = (marq_y1 * scale + zoom - 1) / zoom;
		rx2 = (marq_x2 * scale) / zoom + scale - 1;
		ry2 = (marq_y2 * scale) / zoom + scale - 1;
		if ((rx1 <= pw2) && (rx2 >= rpx) && (ry1 <= ph2) && (ry2 >= rpy))
			repaint_paste(rpx, rpy, pw2, ph2);
	}

	if (marq_status != MARQUEE_NONE) paint_marquee(11, marq_x1, marq_y1);
	if (perim_status > 0) repaint_perim();
}

/* Update x,y,w,h area of current image */
void main_update_area(int x, int y, int w, int h)
{
	int zoom, scale;

	if (can_zoom < 1.0)
	{
		zoom = rint(1.0 / can_zoom);
		w += x;
		h += y;
		x = x < 0 ? -(-x / zoom) : (x + zoom - 1) / zoom;
		y = y < 0 ? -(-y / zoom) : (y + zoom - 1) / zoom;
		w = (w - x * zoom + zoom - 1) / zoom;
		h = (h - y * zoom + zoom - 1) / zoom;
		if ((w <= 0) || (h <= 0)) return;
	}
	else
	{
		scale = rint(can_zoom);
		x *= scale;
		y *= scale;
		w *= scale;
		h *= scale;
	}

	gtk_widget_queue_draw_area(drawing_canvas,
		x + margin_main_x, y + margin_main_y, w, h);
}

/* Get zoomed canvas size */
void canvas_size(int *w, int *h)
{
	int i;

	/* !!! This uses the fact that zoom factor is either N or 1/N !!! */
	if (can_zoom < 1.0)
	{
		i = rint(1.0 / can_zoom);
		*w = (mem_width + i - 1) / i;
		*h = (mem_height + i - 1) / i;
	}
	else
	{
		i = rint(can_zoom);
		*w = mem_width * i;
		*h = mem_height * i;
	}
}

void clear_perim()
{
	perim_status = 0; /* Cleared */
	/* Don't bother if tool has no perimeter */
	if (NO_PERIM(tool_type)) return;
	clear_perim_real(0, 0);
}

void repaint_perim_real( int r, int g, int b, int ox, int oy )
{
	int i, j, w, h, x0, y0, x1, y1, zoom = 1, scale = 1;
	unsigned char *rgb;


	/* !!! This uses the fact that zoom factor is either N or 1/N !!! */
	if (can_zoom < 1.0) zoom = rint(1.0 / can_zoom);
	else scale = rint(can_zoom);

	x0 = margin_main_x + ((perim_x + ox) * scale) / zoom;
	y0 = margin_main_y + ((perim_y + oy) * scale) / zoom;
	x1 = margin_main_x + ((perim_x + ox + perim_s - 1) * scale) / zoom + scale - 1;
	y1 = margin_main_y + ((perim_y + oy + perim_s - 1) * scale) / zoom + scale - 1;

	w = x1 - x0 + 1;
	h = y1 - y0 + 1;
	j = (w > h ? w : h) * 3;
	rgb = calloc(j + 2 * 3, 1); /* 2 extra pixels reserved for loop */
	if (!rgb) return;
	for (i = 0; i < j; i += 6 * 3)
	{
		rgb[i + 0] = rgb[i + 3] = rgb[i + 6] = r;
		rgb[i + 1] = rgb[i + 4] = rgb[i + 7] = g;
		rgb[i + 2] = rgb[i + 5] = rgb[i + 8] = b;
	}

	gdk_draw_rgb_image(drawing_canvas->window, drawing_canvas->style->black_gc,
		x0, y0, 1, h, GDK_RGB_DITHER_NONE, rgb, 3);
	gdk_draw_rgb_image(drawing_canvas->window, drawing_canvas->style->black_gc,
		x1, y0, 1, h, GDK_RGB_DITHER_NONE, rgb, 3);

	gdk_draw_rgb_image(drawing_canvas->window, drawing_canvas->style->black_gc,
		x0 + 1, y0, w - 2, 1, GDK_RGB_DITHER_NONE, rgb, 0);
	gdk_draw_rgb_image(drawing_canvas->window, drawing_canvas->style->black_gc,
		x0 + 1, y1, w - 2, 1, GDK_RGB_DITHER_NONE, rgb, 0);
	free(rgb);
}

void repaint_perim()
{
	/* Don't bother if tool has no perimeter */
	if (NO_PERIM(tool_type)) return;
	repaint_perim_real( 255, 255, 255, 0, 0 );
	perim_status = 1; /* Drawn */
}

static gint canvas_motion( GtkWidget *widget, GdkEventMotion *event )
{
	int x, y, rm, zoom = 1, scale = 1, button = 0;
	GdkModifierType state;
	gdouble pressure = 1.0;

	/* Skip synthetic mouse moves */
	if (unreal_move == 3)
	{
		unreal_move = 2;
		return TRUE;
	}
	rm = unreal_move;
	unreal_move = 0;

	/* Do nothing if no image */
	if (!mem_img[CHN_IMAGE]) return (TRUE);

	if (event->is_hint) gdk_device_get_state(event->device, event->window,
		NULL, &state);
	x = event->x;
	y = event->y;
	state = event->state;

	if ((state & (GDK_BUTTON1_MASK | GDK_BUTTON3_MASK)) ==
		(GDK_BUTTON1_MASK | GDK_BUTTON3_MASK)) button = 13;
	else if (state & GDK_BUTTON1_MASK) button = 1;
	else if (state & GDK_BUTTON3_MASK) button = 3;
	else if (state & GDK_BUTTON2_MASK) button = 2;

	/* !!! This uses the fact that zoom factor is either N or 1/N !!! */
	if (can_zoom < 1.0) zoom = rint(1.0 / can_zoom);
	else scale = rint(can_zoom);

	x = ((x - margin_main_x) * zoom) / scale;
	y = ((y - margin_main_y) * zoom) / scale;

	mouse_event(event->type, x, y, state, button, pressure, rm & 1);

	return TRUE;
}

static gboolean configure_canvas( GtkWidget *widget, GdkEventConfigure *event )
{
	int w, h, new_margin_x = 0, new_margin_y = 0;

	canvas_size(&w, &h);

	w = drawing_canvas->allocation.width - w;
	h = drawing_canvas->allocation.height - h;

	if (w > 0) new_margin_x = w >> 1;
	if (h > 0) new_margin_y = h >> 1;

	if ((new_margin_x != margin_main_x) || (new_margin_y != margin_main_y))
	{
		margin_main_x = new_margin_x;
		margin_main_y = new_margin_y;
		gtk_widget_queue_draw(drawing_canvas);
			// Force redraw of whole canvas as the margin has shifted
	}

	return TRUE;
}

void force_main_configure()
{
	if ( drawing_canvas ) configure_canvas( drawing_canvas, NULL );
}

static gint expose_canvas( GtkWidget *widget, GdkEventExpose *event )
{
	int px, py, pw, ph;

	px = event->area.x;		// Only repaint if we need to
	py = event->area.y;
	pw = event->area.width;
	ph = event->area.height;

	repaint_canvas( px, py, pw, ph );

	return FALSE;
}

void set_cursor()			// Set mouse cursor
{
	int i=0;

	if ( tool_type == TOOL_SELECT ) i=1;
	if ( tool_type == TOOL_FLOOD ) i=2;
	gdk_window_set_cursor( drawing_canvas->window, m_cursor[i] );
}

void set_cursor_type(GdkCursorType type)		// Set mouse cursor type
{
	GdkCursor *temp_cursor = NULL;

	temp_cursor = gdk_cursor_new(type);
	gdk_window_set_cursor(main_window->window, temp_cursor);
	gdk_window_set_cursor(drawing_canvas->window, temp_cursor);
	gdk_cursor_destroy(temp_cursor);

	while (gtk_events_pending()) gtk_main_iteration();	// Wait for update
}



void toolbar_icon_event2(GtkWidget *widget, gpointer data)
{
	gint i = tool_type, j = (gint) data;
	gboolean except=FALSE;

	switch (j)
	{
	case MTB_NEW:
		pressed_new( NULL, NULL ); break;
	case MTB_OPEN:
		pressed_open_file( NULL, NULL ); break;
	case MTB_SAVE:
		pressed_save_file( NULL, NULL ); break;
	case MTB_SAVE_AS:
		pressed_save_file_as( NULL, NULL ); break;
	case MTB_CUT:
		pressed_copy(NULL, NULL, 1); break;
	case MTB_COPY:
		pressed_copy(NULL, NULL, 0); break;
	case MTB_PASTE:
		except=TRUE;
		pressed_paste_centre( NULL, NULL ); break;
	case MTB_UNDO:
		main_undo( NULL, NULL ); break;
	case MTB_REDO:
		main_redo( NULL, NULL ); break;
	case MTB_BRCOSA:
		pressed_brcosa( NULL, NULL ); break;
	case MTB_PAN:
		pressed_pan( NULL, NULL ); break;
	case TTB_TEXT:
		pressed_text( NULL, NULL ); break;
	case TTB_PAINT:
		tool_type = brush_type; break;
	case TTB_FLOOD:
		tool_type = TOOL_FLOOD; break;
	case TTB_SELECT:
		tool_type = TOOL_SELECT; break;
	}

	if ( !except && tool_type != i )		// User has changed tool
	{
		if ( marq_status != MARQUEE_NONE)
		{
			marq_status = MARQUEE_NONE;			// Marquee is on so lose it!
			gtk_widget_queue_draw( drawing_canvas );	// Needed to clear selection
		}
		/* Persistent selection frame */
		if ((tool_type == TOOL_SELECT) && (marq_x1 >= 0) &&
			(marq_y1 >= 0) && (marq_x2 >= 0) && (marq_y2 >= 0))
		{
			marq_status = MARQUEE_DONE;
			check_marquee();
			paint_marquee(1, marq_x1, marq_y1);
		}
		update_menus();
		set_cursor();
	}
}





void main_init( int window_id )
{
	GdkPixmap *icon_pix = NULL;
	GtkWidget *vbox_main, *hbox_bottom, *stamp_canvas, *stamp_sw, *stamp_hbox;
	GtkAccelGroup *accel_group;

	accel_group = gtk_accel_group_new ();

///	MAIN WINDOW

#ifndef WIN32
	if ( window_id == 0 )			// No ID so we are not running in SUGAR
	{
		main_window = gtk_window_new (GTK_WINDOW_TOPLEVEL);
	}
	else
	{
		main_window = gtk_plug_new( (GdkNativeWindow) window_id );
	}
#else
		main_window = gtk_window_new (GTK_WINDOW_TOPLEVEL);
#endif

//	if ( gdk_screen_width() < 1000 ) svg_dir = NULL;
			// Ban the use of SVG icons if the screen is too small

	vbox_main = gtk_vbox_new (FALSE, 0);
	gtk_widget_show (vbox_main);
	gtk_container_add (GTK_CONTAINER (main_window), vbox_main);

// we need to realize the window because we use pixmaps for 
// items on the toolbar in the context of it
	gtk_widget_realize( main_window );


	toolbar_init(vbox_main);


///	PALETTE

	hbox_bottom = gtk_hbox_new (FALSE, 0);
	gtk_widget_show (hbox_bottom);
	gtk_box_pack_start (GTK_BOX (vbox_main), hbox_bottom, TRUE, TRUE, 0);

	toolbar_palette_init(hbox_bottom);


	vbox_right = gtk_vbox_new (FALSE, 0);
	gtk_widget_show (vbox_right);
	gtk_box_pack_start (GTK_BOX (hbox_bottom), vbox_right, TRUE, TRUE, 0);


//	MAIN WINDOW

	drawing_canvas = gtk_drawing_area_new ();
	gtk_widget_set_usize( drawing_canvas, 48, 48 );
	gtk_widget_show( drawing_canvas );

	scrolledwindow_canvas = gtk_scrolled_window_new (NULL, NULL);
	gtk_widget_show (scrolledwindow_canvas);
	gtk_box_pack_start (GTK_BOX (vbox_right), scrolledwindow_canvas, TRUE, TRUE, 0);

	gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (scrolledwindow_canvas),
		GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);

	gtk_scrolled_window_add_with_viewport( GTK_SCROLLED_WINDOW(scrolledwindow_canvas),
		drawing_canvas);

	gtk_signal_connect( GTK_OBJECT(drawing_canvas), "configure_event",
		GTK_SIGNAL_FUNC (configure_canvas), NULL );
	gtk_signal_connect( GTK_OBJECT(drawing_canvas), "expose_event",
		GTK_SIGNAL_FUNC (expose_canvas), NULL );
	gtk_signal_connect( GTK_OBJECT(drawing_canvas), "button_press_event",
		GTK_SIGNAL_FUNC (canvas_button), NULL );
	gtk_signal_connect( GTK_OBJECT(drawing_canvas), "button_release_event",
		GTK_SIGNAL_FUNC (canvas_button), NULL );
	gtk_signal_connect( GTK_OBJECT(drawing_canvas), "motion_notify_event",
		GTK_SIGNAL_FUNC (canvas_motion), NULL );
	gtk_signal_connect( GTK_OBJECT(drawing_canvas), "leave_notify_event",
		GTK_SIGNAL_FUNC (canvas_left), NULL );

	gtk_widget_set_events (drawing_canvas, GDK_ALL_EVENTS_MASK);

///	STAMPS AREA

	if (stamp_start>0)
	{
		hbox_bottom = gtk_hbox_new (FALSE, 0);
		gtk_widget_show (hbox_bottom);
		gtk_box_pack_start (GTK_BOX (vbox_main), hbox_bottom, FALSE, FALSE, 0);

		stamp_sw = gtk_scrolled_window_new (NULL, NULL);
		gtk_widget_show (stamp_sw);
		gtk_box_pack_start (GTK_BOX (vbox_right), stamp_sw, FALSE, FALSE, 0);

		gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (stamp_sw),
			GTK_POLICY_AUTOMATIC, GTK_POLICY_NEVER);

		stamp_canvas = gtk_viewport_new (NULL, NULL);
		gtk_widget_show(stamp_canvas);
		gtk_container_add (GTK_CONTAINER (stamp_sw), stamp_canvas);

		stamp_hbox = gtk_hbox_new (FALSE, 0);
		gtk_widget_show (stamp_hbox);
		gtk_container_add (GTK_CONTAINER (stamp_canvas), stamp_hbox);

		if ( toolbar_stamp_init(stamp_hbox) == 0 )
		{
			gtk_widget_destroy(hbox_bottom);
			gtk_widget_destroy(stamp_sw);
				// No valid stamps were loaded so destroy these unused widgets
//printf("Destroying!!\n");
		}
	}


/////////	End of main window widget setup

	gtk_signal_connect_object (GTK_OBJECT (main_window), "delete_event",
		GTK_SIGNAL_FUNC (delete_event), NULL);
	gtk_signal_connect_object (GTK_OBJECT (main_window), "key_press_event",
		GTK_SIGNAL_FUNC (handle_keypress), NULL);


	men_item_state( menu_undo, FALSE );
	men_item_state( menu_redo, FALSE );
	men_item_state( menu_need_marquee, FALSE );
	men_item_state( menu_need_selection, FALSE );
	men_item_state( menu_need_clipboard, FALSE );

	gtk_widget_show_all (main_window);

	/* !!! Have to wait till canvas is displayed, to init keyboard */
	fill_keycodes(main_keys);

	gtk_widget_grab_focus(scrolledwindow_canvas);	// Stops first icon in toolbar being selected
	gdk_window_raise( main_window->window );

	icon_pix = gdk_pixmap_create_from_xpm_d( main_window->window, NULL, NULL, icon_xpm );
	gdk_window_set_icon( main_window->window, NULL, icon_pix, NULL );

	set_cursor();
	toolbar_select_paint_tool();
}

void spot_undo(int mode)
{
	mem_undo_next(mode);		// Do memory stuff for undo
	update_menus();			// Update menu undo issues
}

void update_titlebar()		// Update filename in titlebar
{
	char txt[300], txt2[600], *extra = "-";

	cleanse_txt( txt2, mem_filename );		// Clean up non ASCII chars
	if ( mem_changed == 1 ) extra = _("(Modified)");
	snprintf( txt, 290, "%s %s %s", VERSION, extra, txt2 );
	gtk_window_set_title (GTK_WINDOW (main_window), txt );
}

void notify_changed()		// Image/palette has just changed - update vars as needed
{
	if ( mem_changed != 1 )
	{
		mem_changed = 1;
		update_titlebar();
	}
}

void notify_unchanged()		// Image/palette has just been unchanged (saved) - update vars as needed
{
	if ( mem_changed != 0 )
	{
		mem_changed = 0;
		update_titlebar();
	}
}

