% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/MovingAverages.R
\name{SMA}
\alias{SMA}
\alias{MovingAverages}
\alias{EMA}
\alias{WMA}
\alias{DEMA}
\alias{GD}
\alias{T3}
\alias{EVWMA}
\alias{ZLEMA}
\alias{VWAP}
\alias{VWMA}
\alias{VMA}
\alias{MA}
\alias{HMA}
\alias{ALMA}
\title{Moving Averages}
\usage{
SMA(x, n = 10, ...)

EMA(x, n = 10, wilder = FALSE, ratio = NULL, ...)

DEMA(x, n = 10, v = 1, wilder = FALSE, ratio = NULL)

WMA(x, n = 10, wts = 1:n, ...)

EVWMA(price, volume, n = 10, ...)

ZLEMA(x, n = 10, ratio = NULL, ...)

VWAP(price, volume, n = 10, ...)

VMA(x, w, ratio = 1, ...)

HMA(x, n = 20, ...)

ALMA(x, n = 9, offset = 0.85, sigma = 6, ...)
}
\arguments{
\item{x}{Price, volume, etc. series that is coercible to xts or matrix.}

\item{n}{Number of periods to average over. Must be between 1 and
\code{nrow(x)}, inclusive.}

\item{\dots}{any other passthrough parameters}

\item{wilder}{logical; if \code{TRUE}, a Welles Wilder type EMA will be
calculated; see notes.}

\item{ratio}{A smoothing/decay ratio.  \code{ratio} overrides \code{wilder}
in \code{EMA}, and provides additional smoothing in \code{VMA}.}

\item{v}{The 'volume factor' (a number in [0,1]).  See Notes.}

\item{wts}{Vector of weights.  Length of \code{wts} vector must equal the
length of \code{x}, or \code{n} (the default).}

\item{price}{Price series that is coercible to xts or matrix.}

\item{volume}{Volume series that is coercible to xts or matrix, that
corresponds to price series, or a constant.  See Notes.}

\item{w}{Vector of weights (in [0,1]) the same length as \code{x}.}

\item{offset}{Percentile at which the center of the distribution should occur.}

\item{sigma}{Standard deviation of the distribution.}
}
\value{
A object of the same class as \code{x} or \code{price} or a vector
(if \code{try.xts} fails) containing the columns:
\describe{
   \item{SMA}{ Simple moving average. }
   \item{EMA}{ Exponential moving average. }
   \item{WMA}{ Weighted moving average. }
   \item{DEMA}{ Double-exponential moving average. }
   \item{EVWMA}{ Elastic, volume-weighted moving average. }
   \item{ZLEMA}{ Zero lag exponential moving average. }
   \item{VWMA}{ Volume-weighed moving average (same as \code{VWAP}). }
   \item{VWAP}{ Volume-weighed average price (same as \code{VWMA}). }
   \item{VWA}{ Variable-length moving average. }
   \item{HMA}{ Hull moving average. }
   \item{ALMA}{ Arnaud Legoux moving average. }
}
}
\description{
Calculate various moving averages (MA) of a series.
}
\details{
\code{SMA} calculates the arithmetic mean of the series over the past
\code{n} observations.

\code{EMA} calculates an exponentially-weighted mean, giving more weight to
recent observations.  See Warning section below.

\code{WMA} is similar to an EMA, but with linear weighting if the length of
\code{wts} is equal to \code{n}.  If the length of \code{wts} is equal to the
length of \code{x}, the WMA will use the values of \code{wts} as weights.

\code{DEMA} is calculated as: \code{DEMA = (1 + v) * EMA(x,n) -
EMA(EMA(x,n),n) * v} (with the corresponding \code{wilder} and \code{ratio}
arguments).

\code{EVWMA} uses volume to define the period of the MA.

\code{ZLEMA} is similar to an EMA, as it gives more weight to recent
observations, but attempts to remove lag by subtracting data prior to
\code{(n-1)/2} periods (default) to minimize the cumulative effect.

\code{VWMA} and \code{VWAP} calculate the volume-weighted moving average
price.

\code{VMA} calculate a variable-length moving average based on the absolute
value of \code{w}.  Higher (lower) values of \code{w} will cause \code{VMA}
to react faster (slower).

\code{HMA} a WMA of the difference of two other WMAs, making it very
reponsive.

\code{ALMA} inspired by Gaussian filters. Tends to put less weight on most
recent observations, reducing tendency to overshoot.
}
\note{
For \code{EMA}, \code{wilder=FALSE} (the default) uses an exponential
smoothing ratio of \code{2/(n+1)}, while \code{wilder=TRUE} uses Welles
Wilder's exponential smoothing ratio of \code{1/n}. The \code{EMA} result
is initialized with the \code{n}-period sample average at period \code{n}.
The exponential decay is applied from that point forward.

Since \code{WMA} can accept a weight vector of length equal to the length of
\code{x} or of length \code{n}, it can be used as a regular weighted moving
average (in the case \code{wts=1:n}) or as a moving average weighted by
volume, another indicator, etc.

Since \code{DEMA} allows adjusting \code{v}, it is technically Tim Tillson's
generalized DEMA (GD).  When \code{v=1} (the default), the result is the
standard DEMA.  When \code{v=0}, the result is a regular EMA.  All other
values of \code{v} return the GD result.  This function can be used to
calculate Tillson's T3 indicator (see example below).  Thanks to John Gavin
for suggesting the generalization.

For \code{EVWMA}, if \code{volume} is a series, \code{n} should be chosen so
the sum of the volume for \code{n} periods approximates the total number of
outstanding shares for the security being averaged.  If \code{volume} is a
constant, it should represent the total number of outstanding shares for the
security being averaged.
}
\section{Warning }{
 Some indicators (e.g. EMA, DEMA, EVWMA, etc.) are
calculated using the indicators' own previous values, and are therefore
unstable in the short-term.  As the indicator receives more data, its output
becomes more stable.  See example below.
}

\examples{

data(ttrc)
ema.20 <-   EMA(ttrc[,"Close"], 20)
sma.20 <-   SMA(ttrc[,"Close"], 20)
dema.20 <-  DEMA(ttrc[,"Close"], 20)
evwma.20 <- EVWMA(ttrc[,"Close"], ttrc[,"Volume"], 20)
zlema.20 <- ZLEMA(ttrc[,"Close"], 20)
alma <- ALMA(ttrc[,"Close"])
hma <- HMA(ttrc[,"Close"])

## Example of Tim Tillson's T3 indicator
T3 <- function(x, n=10, v=1) DEMA(DEMA(DEMA(x,n,v),n,v),n,v)
t3 <- T3(ttrc[,"Close"])

## Example of short-term instability of EMA
## (and other indicators mentioned above)
x <- rnorm(100)
tail( EMA(x[90:100],10), 1 )
tail( EMA(x[70:100],10), 1 )
tail( EMA(x[50:100],10), 1 )
tail( EMA(x[30:100],10), 1 )
tail( EMA(x[10:100],10), 1 )
tail( EMA(x[ 1:100],10), 1 )

}
\references{
The following site(s) were used to code/document this
indicator:\cr \url{http://www.fmlabs.com/reference/ExpMA.htm}\cr
\url{http://www.fmlabs.com/reference/WeightedMA.htm}\cr
\url{http://www.fmlabs.com/reference/DEMA.htm}\cr
\url{http://www.fmlabs.com/reference/T3.htm}\cr
\url{https://www.linnsoft.com/techind/evwma-elastic-volume-weighted-moving-average}\cr
\url{http://www.fmlabs.com/reference/ZeroLagExpMA.htm}\cr
\url{http://www.fmlabs.com/reference/VIDYA.htm}\cr
\url{http://www.traderslog.com/hullmovingaverage}\cr
\url{http://www.arnaudlegoux.com/}\cr
}
\seealso{
See \code{\link{wilderSum}}, which is used in calculating a Welles
Wilder type MA.
}
\author{
Joshua Ulrich, Ivan Popivanov (HMA, ALMA)
}
\keyword{ts}
