package org.apache.lucene.search;

/**
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

import org.apache.lucene.util.LuceneTestCase;
import org.apache.lucene.analysis.MockAnalyzer;
import org.apache.lucene.analysis.MockTokenizer;
import org.apache.lucene.document.Document;
import org.apache.lucene.document.Field;
import org.apache.lucene.index.IndexReader;
import org.apache.lucene.index.RandomIndexWriter;
import org.apache.lucene.index.Term;
import org.apache.lucene.search.IndexSearcher;
import org.apache.lucene.search.PhraseQuery;
import org.apache.lucene.store.Directory;

public class TestSloppyPhraseQuery extends LuceneTestCase {

  private static final String S_1 = "A A A";
  private static final String S_2 = "A 1 2 3 A 4 5 6 A";

  private static final Document DOC_1 = makeDocument("X " + S_1 + " Y");
  private static final Document DOC_2 = makeDocument("X " + S_2 + " Y");
  private static final Document DOC_3 = makeDocument("X " + S_1 + " A Y");
  private static final Document DOC_1_B = makeDocument("X " + S_1 + " Y N N N N " + S_1 + " Z");
  private static final Document DOC_2_B = makeDocument("X " + S_2 + " Y N N N N " + S_2 + " Z");
  private static final Document DOC_3_B = makeDocument("X " + S_1 + " A Y N N N N " + S_1 + " A Y");
  private static final Document DOC_4 = makeDocument("A A X A X B A X B B A A X B A A");

  private static final PhraseQuery QUERY_1 = makePhraseQuery( S_1 );
  private static final PhraseQuery QUERY_2 = makePhraseQuery( S_2 );
  private static final PhraseQuery QUERY_4 = makePhraseQuery( "X A A");

  /**
   * Test DOC_4 and QUERY_4.
   * QUERY_4 has a fuzzy (len=1) match to DOC_4, so all slop values > 0 should succeed.
   * But only the 3rd sequence of A's in DOC_4 will do.
   */
  public void testDoc4_Query4_All_Slops_Should_match() throws Exception {
    for (int slop=0; slop<30; slop++) {
      int numResultsExpected = slop<1 ? 0 : 1;
      checkPhraseQuery(DOC_4, QUERY_4, slop, numResultsExpected);
    }
  }

  /**
   * Test DOC_1 and QUERY_1.
   * QUERY_1 has an exact match to DOC_1, so all slop values should succeed.
   * Before LUCENE-1310, a slop value of 1 did not succeed.
   */
  public void testDoc1_Query1_All_Slops_Should_match() throws Exception {
    for (int slop=0; slop<30; slop++) {
      float score1 = checkPhraseQuery(DOC_1, QUERY_1, slop, 1);
      float score2 = checkPhraseQuery(DOC_1_B, QUERY_1, slop, 1);
      assertTrue("slop="+slop+" score2="+score2+" should be greater than score1 "+score1, score2>score1);
    }
  }

  /**
   * Test DOC_2 and QUERY_1.
   * 6 should be the minimum slop to make QUERY_1 match DOC_2.
   * Before LUCENE-1310, 7 was the minimum.
   */
  public void testDoc2_Query1_Slop_6_or_more_Should_match() throws Exception {
    for (int slop=0; slop<30; slop++) {
      int numResultsExpected = slop<6 ? 0 : 1;
      float score1 = checkPhraseQuery(DOC_2, QUERY_1, slop, numResultsExpected);
      if (numResultsExpected>0) {
        float score2 = checkPhraseQuery(DOC_2_B, QUERY_1, slop, 1);
        assertTrue("slop="+slop+" score2="+score2+" should be greater than score1 "+score1, score2>score1);
      }
    }
  }

  /**
   * Test DOC_2 and QUERY_2.
   * QUERY_2 has an exact match to DOC_2, so all slop values should succeed.
   * Before LUCENE-1310, 0 succeeds, 1 through 7 fail, and 8 or greater succeeds.
   */
  public void testDoc2_Query2_All_Slops_Should_match() throws Exception {
    for (int slop=0; slop<30; slop++) {
      float score1 = checkPhraseQuery(DOC_2, QUERY_2, slop, 1);
      float score2 = checkPhraseQuery(DOC_2_B, QUERY_2, slop, 1);
      assertTrue("slop="+slop+" score2="+score2+" should be greater than score1 "+score1, score2>score1);
    }
  }

  /**
   * Test DOC_3 and QUERY_1.
   * QUERY_1 has an exact match to DOC_3, so all slop values should succeed.
   */
  public void testDoc3_Query1_All_Slops_Should_match() throws Exception {
    for (int slop=0; slop<30; slop++) {
      float score1 = checkPhraseQuery(DOC_3, QUERY_1, slop, 1);
      float score2 = checkPhraseQuery(DOC_3_B, QUERY_1, slop, 1);
      assertTrue("slop="+slop+" score2="+score2+" should be greater than score1 "+score1, score2>score1);
    }
  }

  private float  checkPhraseQuery(Document doc, PhraseQuery query, int slop, int expectedNumResults) throws Exception {
    query.setSlop(slop);

    Directory ramDir = newDirectory();
    RandomIndexWriter writer = new RandomIndexWriter(random, ramDir, new MockAnalyzer(random, MockTokenizer.WHITESPACE, false));
    writer.addDocument(doc);

    IndexReader reader = writer.getReader();

    IndexSearcher searcher = newSearcher(reader);
    TopDocs td = searcher.search(query,null,10);
    //System.out.println("slop: "+slop+"  query: "+query+"  doc: "+doc+"  Expecting number of hits: "+expectedNumResults+" maxScore="+td.getMaxScore());
    assertEquals("slop: "+slop+"  query: "+query+"  doc: "+doc+"  Wrong number of hits", expectedNumResults, td.totalHits);

    //QueryUtils.check(query,searcher);
    writer.close();
    searcher.close();
    reader.close();
    ramDir.close();

    return td.getMaxScore();
  }

  private static Document makeDocument(String docText) {
    Document doc = new Document();
    Field f = new Field("f", docText, Field.Store.NO, Field.Index.ANALYZED);
    f.setOmitNorms(true);
    doc.add(f);
    return doc;
  }

  private static PhraseQuery makePhraseQuery(String terms) {
    PhraseQuery query = new PhraseQuery();
    String[] t = terms.split(" +");
    for (int i=0; i<t.length; i++) {
      query.add(new Term("f", t[i]));
    }
    return query;
  }

}
