/*
 * Copyright 2013 Canonical Ltd.
 *
 * This file is part of powerd.
 *
 * powerd is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; version 3.
 *
 * powerd is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#include <glib.h>
#include <glib-object.h>
#include <gio/gio.h>
#include <string.h>
#include <stdio.h>
#include <errno.h>
#include "powerd-internal.h"
#include "powerd-dbus.h"
#include "log.h"

struct _PowerdSourcePrivate {
    GDBusConnection * system_bus;
    const gchar * path;
    ComCanonicalPowerd * skel;
};

static void powerd_source_class_init (PowerdSourceClass * klass);
static void powerd_source_init       (PowerdSource *      self);
static void powerd_source_dispose    (GObject *           object);
static void powerd_source_finalize   (GObject *           object);

G_DEFINE_TYPE (PowerdSource, powerd_source, G_TYPE_OBJECT);

#define POWERD_SOURCE_GET_PRIVATE(o) (G_TYPE_INSTANCE_GET_PRIVATE ((o), POWERD_TYPE_SOURCE, PowerdSourcePrivate))

/* File-local reference to the singleton object */
static PowerdSource *powerd_source = NULL;

/* This hash table is used to track which dbus names we are watching */
static GHashTable *dbus_name_watch_hash = NULL;

/* Instance */
static void
powerd_source_init (PowerdSource *self)
{
    self->priv = POWERD_SOURCE_GET_PRIVATE(self);
    self->priv->system_bus = g_bus_get_sync(G_BUS_TYPE_SYSTEM, NULL, NULL);
    return;
}

/* Class Init */
static void
powerd_source_class_init (PowerdSourceClass *self)
{
    GObjectClass *object_class = G_OBJECT_CLASS (self);
    g_type_class_add_private (self, sizeof (PowerdSourcePrivate));
    object_class->dispose = powerd_source_dispose;
    object_class->finalize = powerd_source_finalize;
    return;
}

/* Clean up references */
static void
powerd_source_dispose (GObject *object)
{
    PowerdSource * self = POWERD_SOURCE(object);
    if (self->priv->skel != NULL) {
        g_dbus_interface_skeleton_unexport(G_DBUS_INTERFACE_SKELETON(self->priv->skel));
        g_clear_object(&self->priv->skel);
    }
    g_clear_object(&self->priv->system_bus);
    return;
}

/* Free memory */
static void
powerd_source_finalize (GObject *object)
{
    return;
}

static gboolean handle_register_client(PowerdSource *obj,
                                       GDBusMethodInvocation *invocation,
                                       const gchar *name)
{
    const char *owner;
    int ret;

    owner = g_dbus_method_invocation_get_sender(invocation);
    ret = powerd_client_register(owner, name);
    if (ret)
        g_dbus_method_invocation_return_error(invocation, G_DBUS_ERROR,
                                              G_DBUS_ERROR_FAILED,
                                              "Could not register client");
    else
        g_dbus_method_invocation_return_value(invocation, NULL);
    return TRUE;
}

static gboolean handle_unregister_client(PowerdSource *obj,
                                         GDBusMethodInvocation *invocation,
                                         const gchar *name)
{
    const char *owner = g_dbus_method_invocation_get_sender(invocation);
    powerd_client_unregister(owner);
    g_dbus_method_invocation_return_value(invocation, NULL);
    return TRUE;
}

static gboolean handle_ack_state_change(PowerdSource *obj,
                                        GDBusMethodInvocation *invocation,
                                        int state)
{
    const char *owner;
    int ret;

    owner = g_dbus_method_invocation_get_sender(invocation);
    ret = powerd_client_ack(owner, state);
    if (ret)
        g_dbus_method_invocation_return_error(invocation, G_DBUS_ERROR,
                                              G_DBUS_ERROR_INVALID_ARGS,
                                              ret == -EINVAL ?
                                              "Invalid state" :
                                              "Client not registered");
    else
        g_dbus_method_invocation_return_value(invocation, NULL);
    return TRUE;
}

void
powerd_bus_acquired_cb (GDBusConnection *connection,
                        const gchar     *name,
                        gpointer         user_data)
{
    GError *error = NULL;

    powerd_debug("Bus acquired (guid %s)", g_dbus_connection_get_guid (connection));
    powerd_source = (PowerdSource *)g_object_new(POWERD_TYPE_SOURCE, NULL);
    powerd_source->priv->skel = com_canonical_powerd_skeleton_new();
    powerd_source->priv->path = "/com/canonical/powerd";

    if (!g_dbus_interface_skeleton_export(G_DBUS_INTERFACE_SKELETON(powerd_source->priv->skel),
            powerd_source->priv->system_bus, powerd_source->priv->path, &error)) {
        powerd_error("Unable to export skeleton on path '%s': %s", powerd_source->priv->path,
            error->message);
        g_error_free(error);
        error = NULL;
        powerd_exit(-1);
    }

    g_signal_connect(G_OBJECT(powerd_source->priv->skel), "handle-request-sys-state",
        G_CALLBACK(handle_request_sys_state), powerd_source);
    g_signal_connect(G_OBJECT(powerd_source->priv->skel), "handle-clear-sys-state",
        G_CALLBACK(handle_clear_sys_state), powerd_source);
    g_signal_connect(G_OBJECT(powerd_source->priv->skel), "handle-list-sys-requests",
        G_CALLBACK(handle_list_sys_requests), powerd_source);

    g_signal_connect(G_OBJECT(powerd_source->priv->skel), "handle-request-display-state",
        G_CALLBACK(handle_add_display_request), powerd_source);
    g_signal_connect(G_OBJECT(powerd_source->priv->skel), "handle-update-display-state",
        G_CALLBACK(handle_update_display_request), powerd_source);
    g_signal_connect(G_OBJECT(powerd_source->priv->skel), "handle-clear-display-state",
        G_CALLBACK(handle_clear_display_request), powerd_source);
    g_signal_connect(G_OBJECT(powerd_source->priv->skel), "handle-register-client",
        G_CALLBACK(handle_register_client), powerd_source);
    g_signal_connect(G_OBJECT(powerd_source->priv->skel), "handle-unregister-client",
        G_CALLBACK(handle_unregister_client), powerd_source);
    g_signal_connect(G_OBJECT(powerd_source->priv->skel), "handle-ack-state-change",
        G_CALLBACK(handle_ack_state_change), powerd_source);
    g_signal_connect(G_OBJECT(powerd_source->priv->skel), "handle-list-display-requests",
        G_CALLBACK(handle_list_display_requests), powerd_source);
    g_signal_connect(G_OBJECT(powerd_source->priv->skel), "handle-get-sys-request-stats",
        G_CALLBACK(handle_get_sys_request_stats), powerd_source);
    g_signal_connect(G_OBJECT(powerd_source->priv->skel), "handle-get-disp-request-stats",
        G_CALLBACK(handle_get_disp_request_stats), powerd_source);
    g_signal_connect(G_OBJECT(powerd_source->priv->skel), "handle-user-autobrightness-enable",
        G_CALLBACK(handle_user_autobrightness_enable), powerd_source);
    g_signal_connect(G_OBJECT(powerd_source->priv->skel), "handle-get-brightness-params",
        G_CALLBACK(handle_get_brightness_params), powerd_source);
    g_signal_connect(G_OBJECT(powerd_source->priv->skel), "handle-set-user-brightness",
        G_CALLBACK(handle_set_user_brightness), powerd_source);

    powerd_dbus_init_complete();
}

void
powerd_name_acquired_cb (GDBusConnection *connection,
                         const gchar     *name,
                         gpointer         user_data)
{
    powerd_debug("name acquired (%s)", name);
}

void
powerd_name_lost_cb (GDBusConnection *connection,
                     const gchar     *name,
                     gpointer         user_data)
{
    fprintf(stderr, "dbus name lost or unable to acquire dbus name, is another copy of powerd running?\n");
    powerd_exit(-2);
}

void
powerd_sys_state_signal_emit (enum SysPowerStates state)
{
    GError *error = NULL;

    /* Make sure dbus has been set up */
    if (!powerd_source)
        return;

    powerd_debug("Emitting signal for transition to state %s (%d)",
            state_to_string(state), state);
    g_dbus_connection_emit_signal(
            POWERD_SOURCE_GET_PRIVATE(powerd_source)->system_bus,
            NULL, /* destination */
            "/com/canonical/powerd",
            "com.canonical.powerd",
            "SysPowerStateChange",
            g_variant_new("(i)", state),
            &error);
    if (error) {
        powerd_warn("Unable to signal a state change update: %s", error->message);
        g_error_free(error);
    }
}

void
powerd_display_state_signal_emit(enum powerd_display_state state,
                                 guint32 flags)
{
    GError *error = NULL;

    /* Make sure dbus has been set up */
    if (!powerd_source)
        return;

    powerd_debug("Emitting signal for display state change: state=%d flags=0x%08x",
                 state, flags);
    g_dbus_connection_emit_signal(
            POWERD_SOURCE_GET_PRIVATE(powerd_source)->system_bus,
            NULL, /* destination */
            "/com/canonical/powerd",
            "com.canonical.powerd",
            "DisplayPowerStateChange",
            g_variant_new("(iu)", state, flags),
            &error);
    if (error) {
        powerd_warn("Unable to signal a state change update: %s", error->message);
        g_error_free(error);
    }
}

void
ofono_manager_proxy_connect_cb(GObject *source_object,
               GAsyncResult *res,
               gpointer user_data)
{
    GError *error = NULL;
    GDBusProxy *ofono_proxy;

    ofono_proxy = g_dbus_proxy_new_finish(res, &error);
    if (error) {
        powerd_warn("%s failed: %s", __func__, error->message);
        g_error_free(error);
        return;
    }

    /* Register for insertion and removal of modems */
    g_signal_connect(ofono_proxy, "g-signal",
                        G_CALLBACK(on_ofono_manager_signal), NULL);

    /* Get current modems */
    g_dbus_proxy_call(ofono_proxy, "GetModems", NULL,
                        G_DBUS_CALL_FLAGS_NONE, -1, NULL,
                        ofono_get_modems_cb, NULL);
}

void
ofono_voicecall_proxy_connect_cb(GObject *source_object,
               GAsyncResult *res,
               gpointer user_data)
{
    GError *error = NULL;
    GDBusProxy *ofono_proxy;
    struct call_data *call;

    ofono_proxy = g_dbus_proxy_new_finish(res, &error);
    if (error) {
        powerd_warn("%s failed: %s", __func__, error->message);
        g_error_free(error);
        return;
    }

    call = user_data;
    call->ofono_proxy = ofono_proxy;

    /* Register for voicecall signals */
    g_signal_connect(ofono_proxy, "g-signal",
                        G_CALLBACK(on_ofono_voicecall_signal), NULL);

    /* Get current voice call state */
    g_dbus_proxy_call(ofono_proxy, "GetProperties", NULL,
                        G_DBUS_CALL_FLAGS_NONE, -1, NULL,
                        ofono_voicecall_get_props_cb, NULL);
}

void
ofono_proxy_connect_cb(GObject *source_object,
               GAsyncResult *res,
               gpointer user_data)
{
    GError *error = NULL;
    GDBusProxy *ofono_proxy = NULL;
    const char *interface_name = "";

    ofono_proxy = g_dbus_proxy_new_finish (res, &error);
    if (error) {
        powerd_warn("ofono_proxy_connect_cb failed: %s", error->message);
        g_error_free(error);
    }
    else {
        interface_name = g_dbus_proxy_get_interface_name(ofono_proxy);
        powerd_debug("ofono_proxy_connect_cb: proxy is %s", interface_name);
        if (!strcmp(interface_name, "org.ofono.VoiceCallManager")) {
            g_signal_connect(ofono_proxy, "g-signal",
                G_CALLBACK (on_ofono_voicecallmanager_signal), NULL);
        }
        else if (!strcmp(interface_name, "org.ofono.MessageManager")) {
            g_signal_connect(ofono_proxy, "g-signal",
                G_CALLBACK (on_ofono_messagemanager_signal), NULL);
        }
        else if (!strcmp(interface_name, "org.ofono.SupplementaryServices")) {
            g_signal_connect(ofono_proxy, "g-signal",
                G_CALLBACK (on_ofono_ussd_signal), NULL);
        }
        else {
            powerd_warn("unknown interface name for this proxy, ignoring");
        }
        powerd_debug("ofono_proxy_connect_cb succeeded");
    }
}

void
powerd_name_vanished_cb(GDBusConnection *connection,
                        const gchar *name,
                        gpointer user_data)
{
    powerd_warn("%s vanished from dbus, clearing associated requests", name);
    clear_sys_state_by_owner(name);
    clear_disp_state_by_owner(name);
    powerd_client_unregister(name);
    // This can return false if the hash entry was removed because the
    // last request was dropped, in other words, this call will fail
    // if the callers cleaned-up properly, so ignore any FALSE return
    // values.
    // Make sure this is called AFTER the clear_* calls to avoid spurious
    // warnings.
    g_hash_table_remove(dbus_name_watch_hash, name);
}

void
powerd_dbus_name_watch_add(const char *owner)
{
    struct DbusNameWatch *dbnw = NULL;
    char *hash_owner = NULL;

    if (strcmp(owner,"internal") == 0) {
        return;
    }

    /* XXX: For debug, remove later */
    powerd_debug("name_watch_add: looking for %s", owner);

    dbnw = g_hash_table_lookup(dbus_name_watch_hash, owner);
    if (dbnw == NULL) {
        hash_owner = strdup(owner);
        powerd_debug("watching %s to see when it disappears on dbus",
            hash_owner);
        dbnw = g_new(struct DbusNameWatch, 1);
        dbnw->watch_id = g_bus_watch_name(G_BUS_TYPE_SYSTEM, hash_owner,
            G_BUS_NAME_WATCHER_FLAGS_NONE, NULL,
            (GBusNameVanishedCallback)powerd_name_vanished_cb, NULL, NULL);

        dbnw->ref_count = 1;
        g_hash_table_insert(dbus_name_watch_hash, hash_owner, dbnw);
    }
    else {
        dbnw->ref_count++;
        /* XXX: For debug, remove later */
        powerd_debug("name_watch: ref_count for %s is now %d", owner,
            dbnw->ref_count);
    }

}

void
powerd_dbus_name_watch_remove(const char *owner)
{
    struct DbusNameWatch *dbnw = NULL;

    if (strcmp(owner,"internal") == 0) {
        return;
    }

    /* XXX: For debug, remove later */
    powerd_debug("name_watch_remove: looking for %s", owner);

    dbnw = g_hash_table_lookup(dbus_name_watch_hash, owner);
    if (dbnw == NULL) {
        powerd_warn("removal request for a non-existant name: %s\n", owner);
        return;
    }
    else {
        dbnw->ref_count--;
        /* XXX: For debug, remove later */
        powerd_debug("name_watch: ref_count for %s is now %d", owner,
            dbnw->ref_count);
        if (dbnw->ref_count <= 0) {
            powerd_debug("no longer watching %s, there are no more "\
                            "requests", owner);
            g_bus_unwatch_name(dbnw->watch_id);
            if (g_hash_table_remove(dbus_name_watch_hash, (char *)owner) == FALSE) {
                powerd_warn("could not remove %s from dbus_name_watch_hash",
                    owner);
            }
        }
    }
}

/* Destructor for key */
static void dbus_name_watch_key_destroy(gpointer data)
{
    if (data)
        g_free(data);
}

void dbus_name_watch_init(void)
{
    dbus_name_watch_hash = g_hash_table_new_full(g_str_hash,
        g_str_equal, dbus_name_watch_key_destroy, NULL);
}

void dbus_name_watch_deinit(void)
{
    g_hash_table_destroy(dbus_name_watch_hash);
}
