/*****************************************************************************
*
* Copyright (c) 2000 - 2006, The Regents of the University of California
* Produced at the Lawrence Livermore National Laboratory
* All rights reserved.
*
* This file is part of VisIt. For details, see http://www.llnl.gov/visit/. The
* full copyright notice is contained in the file COPYRIGHT located at the root
* of the VisIt distribution or at http://www.llnl.gov/visit/copyright.html.
*
* Redistribution  and  use  in  source  and  binary  forms,  with  or  without
* modification, are permitted provided that the following conditions are met:
*
*  - Redistributions of  source code must  retain the above  copyright notice,
*    this list of conditions and the disclaimer below.
*  - Redistributions in binary form must reproduce the above copyright notice,
*    this  list of  conditions  and  the  disclaimer (as noted below)  in  the
*    documentation and/or materials provided with the distribution.
*  - Neither the name of the UC/LLNL nor  the names of its contributors may be
*    used to  endorse or  promote products derived from  this software without
*    specific prior written permission.
*
* THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT  HOLDERS AND CONTRIBUTORS "AS IS"
* AND ANY EXPRESS OR  IMPLIED WARRANTIES, INCLUDING,  BUT NOT  LIMITED TO, THE
* IMPLIED WARRANTIES OF MERCHANTABILITY AND  FITNESS FOR A PARTICULAR  PURPOSE
* ARE  DISCLAIMED.  IN  NO  EVENT  SHALL  THE  REGENTS  OF  THE  UNIVERSITY OF
* CALIFORNIA, THE U.S.  DEPARTMENT  OF  ENERGY OR CONTRIBUTORS BE  LIABLE  FOR
* ANY  DIRECT,  INDIRECT,  INCIDENTAL,  SPECIAL,  EXEMPLARY,  OR CONSEQUENTIAL
* DAMAGES (INCLUDING, BUT NOT  LIMITED TO, PROCUREMENT OF  SUBSTITUTE GOODS OR
* SERVICES; LOSS OF  USE, DATA, OR PROFITS; OR  BUSINESS INTERRUPTION) HOWEVER
* CAUSED  AND  ON  ANY  THEORY  OF  LIABILITY,  WHETHER  IN  CONTRACT,  STRICT
* LIABILITY, OR TORT  (INCLUDING NEGLIGENCE OR OTHERWISE)  ARISING IN ANY  WAY
* OUT OF THE  USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH
* DAMAGE.
*
*****************************************************************************/

// ************************************************************************* //
//                            avtparaDISFileFormat.C                           //
// ************************************************************************* //
#define NO_BOOST 1
#include <avtparaDISFileFormat.h>
#include <avtparaDISOptions.h>
#include <DBOptionsAttributes.h>
#include <DebugStream.h>
#include "paraDIS_lib/paradis_c_interface.h"
#include "RC_cpp_lib/stringutil.h"
#include "RC_c_lib/debugutil.h"
#include "RC_cpp_lib/pathutil.h"
#include "version.h"

#include <string>
#include <vector>

using std::vector;

#ifdef PARALLEL
#include <avtParallel.h>
#endif

#include <vtkFloatArray.h>
#include <vtkPolyData.h>
#include <vtkCellArray.h>
#include <vtkRectilinearGrid.h>
#include <vtkStructuredGrid.h>
#include <vtkUnstructuredGrid.h>

#include <avtDatabaseMetaData.h>
#include <avtMaterial.h>

#include <Expression.h>

#include <InvalidVariableException.h>

using rclib::Point; 
using     std::string;



// ****************************************************************************
//  Method: avtparaDIS constructor
//
//  Programmer: rcook -- generated by xml2avt
//  Creation:   Tue Jan 30 14:56:34 PST 2007
//
// ****************************************************************************
 
avtparaDISFileFormat::avtparaDISFileFormat(const char *filename,
    DBOptionsAttributes *rdatts)
  : avtSTSDFileFormat(filename), 
    mVerbosity(0), mFilename(filename), mMaterialSetChoice(0) {
 
  paraDIS_init(); 
  // DebugStream::GetLevel doesn't exist in 1.12RC, so turning off
  // debugging as a work around.  It does exist in the trunk, so 
  // leave as follows in trunk.  Sigh.  
  // mVerbosity = DebugStream::GetLevel(); 
  mVerbosity = rdatts->GetInt(PARADIS_VERBOSITY); 
  string debugfile ; 

  /*  if (debug1_real) mVerbosity++;
  if (debug2_real) mVerbosity++;
  if (debug3_real) mVerbosity++;
  if (debug4_real) mVerbosity++;
  if (debug5_real) mVerbosity++;
  */
  if (mVerbosity) {
    debugfile = rdatts->GetString(PARADIS_DEBUG_FILE);
  }
  paraDIS_SetVerbosity(mVerbosity, debugfile.c_str()); 
  /*!
    Write out files
  */ 
  paraDIS_EnableDebugOutput(rdatts->GetBool(PARADIS_ENABLE_DEBUG_OUTPUT)); 

  dbg_setverbose(mVerbosity); 

#ifdef PARALLEL
  /*!  I keep this here to remind myself how to use MPI.  LOL
    MPI_Comm_rank(MPI_COMM_WORLD, &mProcNum); 
    MPI_Comm_size(MPI_COMM_WORLD, &mNumProcs);  
  */
  // This is the Visit Way: 
  mProcNum = PAR_Rank(); 
  mNumProcs = PAR_Size(); 
#else
  mProcNum = 0; 
  mNumProcs = 1; 
#endif

  mMaterialSetChoice = rdatts->GetEnum(PARADIS_MATERIAL_SET_CHOICE); 
  paraDIS_SetThreshold(rdatts->GetDouble(PARADIS_NN_ARM_THRESHOLD)); 
 // Adding a material to the mesh, which is the correct thing to do for int-valued data (discrete) as it allows subsetting to work
  // our own data members. 
  mSegmentMNTypes.clear(); 
  mSegmentMNTypes.push_back(string("ARM_UNKNOWN"));    
  mSegmentMNTypes.push_back(string("ARM_UNINTERESTING"));
  mSegmentMNTypes.push_back(string("ARM_LOOP"));
  mSegmentMNTypes.push_back(string("ARM_MM_111"));
  mSegmentMNTypes.push_back(string("ARM_MN_111"));
  mSegmentMNTypes.push_back(string("ARM_NN_111"));
  mSegmentMNTypes.push_back(string("ARM_MM_100"));
  mSegmentMNTypes.push_back(string("ARM_MN_100"));
  mSegmentMNTypes.push_back(string("ARM_NN_100"));
  mSegmentMNTypes.push_back(string("ARM_SHORT_NN_111"));
  mSegmentMNTypes.push_back(string("ARM_SHORT_NN_100"));

  mBurgersTypes.push_back(string("100 arm type"));    
  mBurgersTypes.push_back(string("010 arm type"));
  mBurgersTypes.push_back(string("001 arm type"));
  mBurgersTypes.push_back(string("+++ arm type"));
  mBurgersTypes.push_back(string("++- arm type"));
  mBurgersTypes.push_back(string("+-+ arm type"));
  mBurgersTypes.push_back(string("-++ arm type"));
  mBurgersTypes.push_back(string("unknown arm type"));

  mNodeNeighborValues.push_back(string("0 neighbors"));    
  mNodeNeighborValues.push_back(string("1 neighbor"));    
  mNodeNeighborValues.push_back(string("2 neighbors"));    
  mNodeNeighborValues.push_back(string("3 neighbors"));    
  mNodeNeighborValues.push_back(string("4 neighbors"));    
  mNodeNeighborValues.push_back(string("5 neighbors"));    
  mNodeNeighborValues.push_back(string("6 neighbors"));    
  mNodeNeighborValues.push_back(string("7 neighbors"));    
  mNodeNeighborValues.push_back(string("8 neighbors"));    
  mNodeNeighborValues.push_back(string("more than 8 neighbors"));    
  
  // PARADIS METADATA
  paraDIS_SetDataFile(mFilename.c_str());    
  paraDIS_SetProcNum(mProcNum, mNumProcs);     
  return;   
}


// ****************************************************************************
//  Method: avtparaDISFileFormat::FreeUpResources
//
//  Purpose:
//      When VisIt is done focusing on a particular timestep, it asks that
//      timestep to free up any resources (memory, file descriptors) that
//      it has associated with it.  This method is the mechanism for doing
//      that.
//
//  Programmer: rcook -- generated by xml2avt
//  Creation:   Tue Jan 30 14:56:34 PST 2007
//
// ****************************************************************************

void
avtparaDISFileFormat::FreeUpResources(void)
{
  paraDIS_Clear(); 
}

// ****************************************************************************
//  Method: avtparaDISFileFormat::GetParaDISMetaData
//
//  Purpose:
//      Check to see if the file is paraDIS format, if so, set up the metadata
//      accordingly.  
//
//  Programmer: rcook
//  Creation:   Tue Jan 30 14:56:34 PST 2007
//
// ****************************************************************************
bool avtparaDISFileFormat::PopulateParaDISMetaData(avtDatabaseMetaData *md){ 
  // PopulateDatabaseMetaData() will not get called for every timestep, so:
  // DO NOT INITIALIZE CLASS DATA MEMBERS HERE!  
  md->SetMustRepopulateOnStateChange(true); 

  debug2 << "GetParaDISMetaData called " << endl; 
  //double *extents = NULL; 
  
  double fextents[6] = {0}; 
  if ( ! paraDIS_GetBounds(fextents)) {
    debug1 <<"Could not get bounds.  The file is not a ParaDIS-style file" <<endl; 
    return false; 
  }
  
  // We think we have a valid paraDIS file at this point. 
  debug1 <<"The file is a ParaDIS-style file" <<endl; 
 
 // PARADIS MESH METADATA
  string meshname = "segments"; /* line segments */ 
  avtMeshType meshtype = AVT_UNSTRUCTURED_MESH; 
  int nblocks = 1; /* This is for multi-domain, we are single-domain self-decomposing */ 
  int block_origin = 0; 
  int spatial_dimension = 3; /* 3D space */ 
  int topological_dimension = 1; /* lines */ 
  double *extents = NULL; 
  AddMeshToMetaData(md, meshname, meshtype, extents, nblocks, block_origin,
                    spatial_dimension, topological_dimension);


   /*!
    =======================================================
    segments
    =======================================================
  */ 
  AddScalarVarToMetaData(md, "segmentIndex", "segments", AVT_ZONECENT);
  AddScalarVarToMetaData(md, "burgersType", "segments", AVT_ZONECENT);
  if (mVerbosity > 1) {
    AddScalarVarToMetaData(md, "segmentEngine", "segments", AVT_ZONECENT);
  }

  // VisIt only allows one material set, so we have to let the user choose one
  if (mMaterialSetChoice == 0) {
    AddMaterialToMetaData(md, "Segment_Burgers_Type", "segments", 
                          mBurgersTypes.size(), mBurgersTypes);
  } else {
    AddMaterialToMetaData(md, "Segment_MN_Type", "segments", 
                          mSegmentMNTypes.size(), mSegmentMNTypes);
  }
 
   /*!
    =======================================================
    nodes
    =======================================================
  */ 
  meshname = "nodes"; 
  topological_dimension = 0; /* only points */ 
  meshtype = AVT_POINT_MESH; 
  AddMeshToMetaData(md, meshname, meshtype, extents, nblocks, block_origin,
                    spatial_dimension, topological_dimension);

  //AddScalarVarToMetaData(md, "nodeType", "nodes", AVT_NODECENT);
  // use a material for the node types 
  AddMaterialToMetaData(md, "Node_Num_Neighbors", "nodes", mNodeNeighborValues.size(), mNodeNeighborValues);
  
  AddScalarVarToMetaData(md, "simulationDomain", "nodes", AVT_NODECENT); 
  AddScalarVarToMetaData(md, "simulationID", "nodes", AVT_NODECENT); 
  AddScalarVarToMetaData(md, "nodeIndex", "nodes", AVT_NODECENT); 
  
  if (mVerbosity > 1) {
    AddScalarVarToMetaData(md, "nodeEngine", "nodes", AVT_ZONECENT);
  }
  
  md->SetFormatCanDoDomainDecomposition(true);  

  
  return true; 
}

// ****************************************************************************
//  Method: avtparaDISFileFormat::PopulateDatabaseMetaData
//
//  Purpose:
//      This database meta-data object is like a table of contents for the
//      file.  By populating it, you are telling the rest of VisIt what
//      information it can request from you.
//
//  Programmer: rcook -- generated by xml2avt
//  Creation:   Tue Jan 30 14:56:34 PST 2007
//
// ****************************************************************************

void
avtparaDISFileFormat::PopulateDatabaseMetaData(avtDatabaseMetaData *md)
{
  debug1 << "starting populateDatabaseMetaData" << endl; 
  debug1 << "avtparaDISFileFormat version " << PARADIS_READER_VERSION_NUMBER << ", " << PARADIS_READER_VERSION_DATE << endl;

  /* first segments */ 

  if (!PopulateParaDISMetaData(md)) {
    EXCEPTION1(VisItException, "Cannot get paraDIS metadata from the file."); 
  }

  debug1 << "finishing populateDatabaseMetaData" << endl; 
  return;

  // NOTES FOR DOCUMENTATION
 // CODE TO ADD A MESH
    //
    // string meshname = ...
    //
    // AVT_RECTILINEAR_MESH, AVT_CURVILINEAR_MESH, AVT_UNSTRUCTURED_MESH,
    // AVT_POINT_MESH, AVT_SURFACE_MESH, AVT_UNKNOWN_MESH
    // avtMeshType mt = AVT_RECTILINEAR_MESH;
    //
    // int nblocks = 1;  <-- this must be 1 for STSD
    // int block_origin = 0;
    // int spatial_dimension = 2;
    // int topological_dimension = 2;
    // float *extents = NULL;
    //
    // Here's the call that tells the meta-data object that we have a mesh:
    //
    // AddMeshToMetaData(md, meshname, mt, extents, nblocks, block_origin,
    //                   spatial_dimension, topological_dimension);
    //

    //
    // CODE TO ADD A SCALAR VARIABLE
    //
    // string mesh_for_this_var = meshname; // ??? -- could be multiple meshes
    // string varname = ...
    //
    // AVT_NODECENT, AVT_ZONECENT, AVT_UNKNOWN_CENT
    // avtCentering cent = AVT_NODECENT;
    //
    //
    // Here's the call that tells the meta-data object that we have a var:
    //
    //

    //
    // CODE TO ADD A VECTOR VARIABLE
    //
    // string mesh_for_this_var = meshname; // ??? -- could be multiple meshes
    // string varname = ...
    // int vector_dim = 2;
    //
    // AVT_NODECENT, AVT_ZONECENT, AVT_UNKNOWN_CENT
    // avtCentering cent = AVT_NODECENT;
    //
    //
    // Here's the call that tells the meta-data object that we have a var:
    //
    // AddVectorVarToMetaData(md, varname, mesh_for_this_var, cent,vector_dim);
    //

    //
    // CODE TO ADD A TENSOR VARIABLE
    //
    // string mesh_for_this_var = meshname; // ??? -- could be multiple meshes
    // string varname = ...
    // int tensor_dim = 9;
    //
    // AVT_NODECENT, AVT_ZONECENT, AVT_UNKNOWN_CENT
    // avtCentering cent = AVT_NODECENT;
    //
    //
    // Here's the call that tells the meta-data object that we have a var:
    //
    // AddTensorVarToMetaData(md, varname, mesh_for_this_var, cent,tensor_dim);
    //

    //
    // CODE TO ADD A MATERIAL
    //

  /* 
     string matname = "mymaterial";
     string mesh_for_mat = "segments"; 
     vector<string> mnames;
     mnames.push_back(string("100 arm type"));    
     mnames.push_back(string("010 arm type"));
     mnames.push_back(string("001 arm type"));
     mnames.push_back(string("+++ arm type"));
     mnames.push_back(string("++- arm type"));
     mnames.push_back(string("+-+ arm type"));
     mnames.push_back(string("-++ arm type"));
     mnames.push_back(string("unknown arm type"));
     
     int nmats = 8;
  */ 
  // 
  // Here's the call that tells the meta-data object that we have a mat:
  //
  //AddMaterialToMetaData(md, matname, mesh_for_mat, nmats, mnames);
  //
  //
  // Here's the way to add expressions:
    //Expression momentum_expr;
    //momentum_expr.SetName("momentum");
    //momentum_expr.SetDefinition("{u, v}");
    //momentum_expr.SetType(Expression::VectorMeshVar);
    //md->AddExpression(&momentum_expr);
    //Expression KineticEnergy_expr;
    //KineticEnergy_expr.SetName("KineticEnergy");
    //KineticEnergy_expr.SetDefinition("0.5*(momentum*momentum)/(rho*rho)");
    //KineticEnergy_expr.SetType(Expression::ScalarMeshVar);
    //md->AddExpression(&KineticEnergy_expr);

 }

// ****************************************************************************
//  Method: avtparaDISFileFormat::GetMesh
//
//  Purpose:
//      Gets the mesh associated with this file.  The mesh is returned as a
//      derived type of vtkDataSet (ie vtkRectilinearGrid, vtkStructuredGrid,
//      vtkUnstructuredGrid, etc).
//
//  Arguments:
//      meshname    The name of the mesh of interest.  This can be ignored if
//                  there is only one mesh.
//
//  Programmer: rcook -- generated by xml2avt
//  Creation:   Tue Jan 30 14:56:34 PST 2007
//
// ****************************************************************************

vtkDataSet *
avtparaDISFileFormat::GetMesh(const char *meshname)
{  
  debug2 << "GetMesh("<<meshname<<")"<<endl;
  try {
    //==================================================================
    /*! 
      Nodes or segments
    */ 
    debug1 << "Starting GetParaDISMesh(" <<meshname<<")"<< endl; 
    //return TestGetMesh(); 
    if (paraDIS_GetNumNodes() == 0) {
      paraDIS_SetDataFile(mFilename.c_str()); 
#ifdef TEST_SUBSPACE
      /*! 
        Set a subspace to exercise culling code
      */       
      paraDIS_TestRestrictSubspace(); 
#endif
      paraDIS_ReadData();
      if (dbg_isverbose() > 1) {
        paraDIS_PrintArmStats(); 
      }
      debug2 << "GetMesh: created " << paraDIS_GetNumNodes() << " nodes  and " << paraDIS_GetNumArmSegments() << " arm segments." << endl; 
    }
    
    //=============================================================
    /* nodes or segments */ 
    //===============================================================
    
    /*! 
      The set of points in both segments and node meshes are the same.  This redundancy is the way our data is.
    */ 
    vtkPoints *points =  vtkPoints::New();
    int numpoints = paraDIS_GetNumNodes(); 
    debug2 << "Setting number of points to  " <<numpoints << endl; 
    points->SetNumberOfPoints(numpoints); 
    
    debug2 << "GetMesh: setting locations" << endl; 
    float location[3] ; 
    int nodenum = 0; 
    while (nodenum < numpoints) {
      paraDIS_GetNodeLocation(nodenum, location); 
      debug5 << "Setting point " << nodenum << "to (" << location[0] << ", " << location[1] << ", " << location[2] << endl; 
      points->SetPoint(nodenum, location); 
      ++nodenum; 
    }
    
    
    /*! 
      For segments, build the mesh from the ArmSegments
    */ 
    if (! strcmp(meshname, "segments")) {
      debug2 << "GetMesh: *** creating segment mesh ***" << endl; 
      
      
      //vtkCellArray *lines = vtkCellArray::New(); 
      int numsegments = paraDIS_GetNumArmSegments(); 
      //lines->Allocate(3*numsegments); 
      
      vtkUnstructuredGrid *linemesh = vtkUnstructuredGrid::New(); // reference-counted "new"
      linemesh->SetPoints(points); 
      points->Delete();  // memory is not really freed here, I think due to ref counting. From now on, points are by index    
      
      /*!
        Associate node IDs to each segment
      */ 
      vtkIdType nodeIndices[2]; 
      int index0, index1, segnum = 0; 
      while (segnum < numsegments) {
        index0 = paraDIS_GetEndpointIndex(segnum, 0); 
        index1 = paraDIS_GetEndpointIndex(segnum, 1); 
        debug5 << "GetMesh: Segment " << segnum << " node indices: (" << index0 << ", " << index1 << endl; 
        nodeIndices[0] = index0;
        nodeIndices[1] = index1;
        linemesh->InsertNextCell(VTK_LINE, 2, nodeIndices);
        ++segnum; 
      } 
      /*!
        Assign lines to mesh
      */ 
      //linemesh->SetLines(lines);
      //lines->Delete(); // actually a deref, not deleted...
      debug1 << "GetMesh: Ending GetMesh" << endl; 
      return linemesh; 
    }/* End segments mesh */ 
    else {      
      debug2 << "GetMesh: *** creating node mesh ***" << endl; 
      vtkIdType nodeIndex[1]; //has to be array type
      vtkUnstructuredGrid *nodemesh = vtkUnstructuredGrid::New(); 
      nodemesh->SetPoints(points); 
      points->Delete(); 
      nodenum = 0; 
      while (nodenum != numpoints) {
        debug5 << "Inserting node index " << nodenum << " = " <<  nodeIndex[0] << endl;        
        nodeIndex[0] = nodenum; 
        nodemesh->InsertNextCell(VTK_VERTEX, 1, nodeIndex); 
        
        ++nodenum;
        
      }      
      debug1 << "GetMesh: Ending GetMesh" << endl; 
      return nodemesh;         
    }/* end node mesh */ 
  }catch (string err) {
    cerr << err << endl; 
    EXCEPTION1(VisItException, err.c_str());
  }   
}

vtkDataSet *
avtparaDISFileFormat::TestGetMesh(const char *meshname) 
{
  //----------------------------------------------------------------------
  //OLD CODE OLD CODE OLD CODE   
  // First pass:  just pass back a cube and see what happens
  debug1 << "starting with GetMesh"<<endl; 
  // allocate memory for points 
  int numpoints = 8; 
  vtkPoints *points =  vtkPoints::New();
  points->SetNumberOfPoints(numpoints); 

  // assign some data to "points" 
  // these points are in a skewed hexahedron shape
  float ptsdata[8][3] = {{0.1,0.1,0.1}, {1,0,0}, {0,1,0}, {1,1,0}, 
                     {0,0,1}, {1,0,1}, {0,1,1}, {1.1,1.1,1.1}};
  int i=0; 
  for (i=0; i<numpoints; i++) points->SetPoint(i,ptsdata[i]); 

  
  // determine the mesh to return  
  if (! strcmp(meshname, "segments")) {
    // assign points to the mesh 
    vtkPolyData *linemesh = vtkPolyData::New(); //reference-counted "new"
    linemesh->SetPoints(points); 
    points->Delete(); /* we'll refer to these by id now */ 

    // allocate memory for the lines 
    int numlines = 12;
    vtkCellArray *lines = vtkCellArray::New(); 
    lines->Allocate(3*numlines); //for lines, it's 3*numlines 

    // associate node ids to the lines  
    vtkIdType line_ids[12][2] = { {0,1},{0,2},{3,1}, {3,2},
                                        {0,4},{1,5},{2,6}, {3,7}, 
                                        {4,5}, {4,6}, {7,5}, {7,6} };
    for (i=0; i<numlines; i++) lines->InsertNextCell(2,line_ids[i]); 
     
    // assign lines to mesh
    linemesh->SetLines(lines); 
    lines->Delete(); //deref, actually
    debug1 << "done with GetMesh (lines)"<<endl; 
    return linemesh; 
  }
  else if (!strcmp(meshname, "nodes")){
    vtkUnstructuredGrid *nodemesh = vtkUnstructuredGrid::New(); 
    nodemesh->SetPoints(points); 
    points->Delete(); 
  
    // associate node ids to the nodes 
    int numnodes = 8; 
    vtkIdType node_ids[8][1] = {{0}, {1}, {2}, {3}, {4}, {5}, {6}, {7}}; 
    for (i=0; i<numnodes; i++) nodemesh->InsertNextCell(VTK_VERTEX, 1, node_ids[i]); 
    
    debug1 << "done with GetMesh (nodes)"<<endl; 
    return nodemesh; 
  }
  return NULL; 
}


// ****************************************************************************
//  Method: avtparaDISFileFormat::GetVar
//
//  Purpose:
//      Gets a scalar variable associated with this file.  Although VTK has
//      support for many different types, the best bet is vtkFloatArray, since
//      that is supported everywhere through VisIt.
//
//  Arguments:
//      varname    The name of the variable requested.
//
//  Programmer: rcook -- generated by xml2avt
//  Creation:   Tue Jan 30 14:56:34 PST 2007
//
// ****************************************************************************

vtkDataArray *
avtparaDISFileFormat::GetVar(const char *varname)
{
  debug2 << "Beginning GetVar(" << varname << ")" << endl; 
  int index = 0;
  float f=0.0; 
  vtkFloatArray *scalars = vtkFloatArray::New(); 

  int numsegs = paraDIS_GetNumArmSegments(), 
    numnodes = paraDIS_GetNumNodes();
  if (!strcmp(varname, "segmentIndex")) {
    for (index=0; index<numsegs; index++) {
      f=index; 
      scalars->InsertTuple(index,&f); 
    }
  } else if (!strcmp(varname, "segmentEngine")) {
    for (index=0; index<numsegs; index++) {
      f= mProcNum; 
      scalars->InsertTuple(index,&f);
    }
  } else if (!strcmp(varname, "burgersType")) {
    for (index=0; index<numsegs; index++) {
      f= paraDIS_GetSegmentBurgersType(index); 
      scalars->InsertTuple(index,&f);
    }
  }   else if (!strcmp(varname, "simulationDomain")) {
    for (index=0; index<numnodes; index++) {
      f=paraDIS_GetNodeSimulationDomain(index); 
      scalars->InsertTuple(index,&f); 
    }
  }   else if (!strcmp(varname, "simulationID")) {
    for (index=0; index<numnodes; index++) {
      f=paraDIS_GetNodeSimulationID(index); 
      scalars->InsertTuple(index,&f); 
    }
  }   else if (!strcmp(varname, "nodeIndex")) {
    for (index=0; index<numnodes; index++) {
      f=index;
      scalars->InsertTuple(index,&f); //value(ith element) == i
    }
  } else if (!strcmp(varname, "nodeEngine")) {
    for (index=0; index<numnodes; index++) {
      f= mProcNum; 
      scalars->InsertTuple(index,&f); 
    }
  } 
  debug1 << "done with GetVar"<<endl;
  return scalars;
    //
    // If you have a file format where variables don't apply (for example a
    // strictly polygonal format like the STL (Stereo Lithography) format,
    // then uncomment the code below.
    //
    // EXCEPTION1(InvalidVariableException, varname);
    //

    //
    // If you do have a scalar variable, here is some code that may be helpful.
    //
    // int ntuples = XXX; // this is the number of entries in the variable.
    // vtkFloatArray *rv = vtkFloatArray::New();
    // rv->SetNumberOfTuples(ntuples);
    // for (int i = 0 ; i < ntuples ; i++)
    // {
    //      rv->SetTuple1(i, VAL);  // you must determine value for ith entry.
    // }
    //
    // return rv;
    //
}


// ****************************************************************************
//  Method: avtparaDISFileFormat::GetVectorVar
//
//  Purpose:
//      Gets a vector variable associated with this file.  Although VTK has
//      support for many different types, the best bet is vtkFloatArray, since
//      that is supported everywhere through VisIt.
//
//  Arguments:
//      varname    The name of the variable requested.
//
//  Programmer: rcook -- generated by xml2avt
//  Creation:   Tue Jan 30 14:56:34 PST 2007
//
// ****************************************************************************

vtkDataArray *
avtparaDISFileFormat::GetVectorVar(const char *varname)
{
  return 0; 
  //YOU MUST IMPLEMENT THIS

    //
    // If you have a file format where variables don't apply (for example a
    // strictly polygonal format like the STL (Stereo Lithography) format,
    // then uncomment the code below.
    //
    // EXCEPTION1(InvalidVariableException, varname);
    //

    //
    // If you do have a vector variable, here is some code that may be helpful.
    //
    // int ncomps = YYY;  // This is the rank of the vector - typically 2 or 3.
    // int ntuples = XXX; // this is the number of entries in the variable.
    // vtkFloatArray *rv = vtkFloatArray::New();
    // int ucomps = (ncomps == 2 ? 3 : ncomps);
    // rv->SetNumberOfComponents(ucomps);
    // rv->SetNumberOfTuples(ntuples);
    // float *one_entry = new float[ucomps];
    // for (int i = 0 ; i < ntuples ; i++)
    // {
    //      int j;
    //      for (j = 0 ; j < ncomps ; j++)
    //           one_entry[j] = ...
    //      for (j = ncomps ; j < ucomps ; j++)
    //           one_entry[j] = 0.;
    //      rv->SetTuple(i, one_entry); 
    // }
    //
    // delete [] one_entry;
    // return rv;
    //
}


// NEED METHOD COMMENT

void *
avtparaDISFileFormat::GetAuxiliaryData(const char *var, const char *type,
                                       void *, DestructorFunction &df)
{
  debug3 << "GetAuxiliaryData(" << var << ", " << type << ")" << endl; 
  if (strcmp(type, AUXILIARY_DATA_MATERIAL) != 0) {
    return NULL;
  }

  avtMaterial *mat = NULL;
  
  if (string(var).substr(0,8) == "Segment_") {
    int index; 
    int numsegs = paraDIS_GetNumArmSegments(); 
    int *matId = new int[numsegs];
    int *matptr = matId; 
    if (string(var) == "Segment_Burgers_Type") {
      //---------------------------------------------
      for (index=0; index<numsegs; index++) {
        *(matptr++)=paraDIS_GetSegmentBurgersType(index)-1; 
      }    
      debug3 << "mBurgersTypes.size(): " << mBurgersTypes.size() << endl;
      int i=0; 
      while (i < mBurgersTypes.size()) {
        debug3 << "mBurgersTypes["<<i<<"] = "<<mBurgersTypes[i]<< endl; 
        ++i; 
      }
      mat = new avtMaterial(mBurgersTypes.size(), mBurgersTypes, 
                            numsegs, matId, 0, NULL, NULL, NULL, NULL);
      //---------------------------------------------      
    }     else if (string(var) == "Segment_MN_Type")  {
      //---------------------------------------------      
      for (index=0; index<numsegs; index++) {
        *(matptr++)=paraDIS_GetSegmentMNType(index); 
      }    
      mat = new avtMaterial(mSegmentMNTypes.size(), mSegmentMNTypes, 
                            numsegs, matId, 0, NULL, NULL, NULL, NULL);
      //---------------------------------------------
    } else {
      string err = string("Error: unknown variable: ") + var; 
      EXCEPTION1(VisItException, err.c_str());
    }
    
  }
  else if (string(var) == "Node_Num_Neighbors") {
    int index; 
    int numnodes = paraDIS_GetNumNodes(); 
    int *matId = new int[numnodes];
    int *matptr = matId; 
    for (index=0; index<numnodes; index++) {
      *(matptr++)=paraDIS_GetNumNodeNeighbors(index); 
    }    
    debug3 << "mNodeNeighborValues.size() = " << mNodeNeighborValues.size() << endl; 
    mat =  new avtMaterial(mNodeNeighborValues.size(), mNodeNeighborValues, 
                           numnodes, matId, 0, NULL, NULL, NULL, NULL);
  } else {
    /* 
       string err = string("Error: unknown variable: ") + var; 
       EXCEPTION1(VisItException, err.c_str());
    */
    return NULL; 
  }
  df = avtMaterial::Destruct;
  
  return mat;
}
      


