/* This file is part of the Palabos library.
 *
 * Copyright (C) 2011-2015 FlowKit Sarl
 * Route d'Oron 2
 * 1010 Lausanne, Switzerland
 * E-mail contact: contact@flowkit.com
 *
 * The most recent release of Palabos can be downloaded at 
 * <http://www.palabos.org/>
 *
 * The library Palabos is free software: you can redistribute it and/or
 * modify it under the terms of the GNU Affero General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * The library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

/* Main author: Orestis Malaspinas
 */

/** \file
 * Collision terms with external force -- header file.
 */
#ifndef EXTERNAL_FORCE_DYNAMICS_H
#define EXTERNAL_FORCE_DYNAMICS_H

#include "core/globalDefs.h"
#include "core/dynamics.h"
#include "basicDynamics/isoThermalDynamics.h"

namespace plb {

/** Implementation of the computation of the velocity in the 
  * presence of an external force
*/
template<typename T, template<typename U> class Descriptor>
class ExternalForceDynamics : public IsoThermalBulkDynamics<T,Descriptor> {
public:
/* *************** Construction / Destruction ************************ */
    ExternalForceDynamics(T omega_);

/* *************** Velocity computation ************************* */

    /// Implementation of velocity computation
    virtual void computeVelocity( Cell<T,Descriptor> const& cell, 
                                  Array<T,Descriptor<T>::d>& u ) const;
};


/// Implementation BGK dynamics with a simple, linear external force
template<typename T, template<typename U> class Descriptor>
class NaiveExternalForceBGKdynamics : public ExternalForceDynamics<T,Descriptor> {
public:
/* *************** Construction / Destruction ************************ */
    NaiveExternalForceBGKdynamics(T omega_);

    /// Clone the object on its dynamic type.
    virtual NaiveExternalForceBGKdynamics<T,Descriptor>* clone() const;

    /// Return a unique ID for this class.
    virtual int getId() const;

/* *************** Collision and Equilibrium ************************* */

    /// Implementation of the collision step
    virtual void collide(Cell<T,Descriptor>& cell,
                         BlockStatistics& statistics_);

    /// Compute equilibrium distribution function
    virtual T computeEquilibrium(plint iPop, T rhoBar, Array<T,Descriptor<T>::d> const& j,
                                 T jSqr, T thetaBar=T()) const;
private:
    static int id;
};

/// Implementation BGK dynamics with a simple, linear external force
template<typename T, template<typename U> class Descriptor>
class NaiveExternalForcePrecondBGKdynamics : public ExternalForceDynamics<T,Descriptor> {
public:
/* *************** Construction / Destruction ************************ */
    NaiveExternalForcePrecondBGKdynamics(T omega_, T invGamma_);

    /// Clone the object on its dynamic type.
    virtual NaiveExternalForcePrecondBGKdynamics<T,Descriptor>* clone() const;

    /// Return a unique ID for this class.
    virtual int getId() const;

/* *************** Collision and Equilibrium ************************* */

    /// Implementation of the collision step
    virtual void collide(Cell<T,Descriptor>& cell,
                         BlockStatistics& statistics_);

    /// Compute equilibrium distribution function
    virtual T computeEquilibrium(plint iPop, T rhoBar, Array<T,Descriptor<T>::d> const& j,
                                 T jSqr, T thetaBar=T()) const;
    virtual void serialize(HierarchicSerializer& serializer) const;
    virtual void unserialize(HierarchicUnserializer& unserializer);
private:
    static int id;
    T invGamma;
};

/// Implementation of O(Ma^2) BGK dynamics with an external force (Guo approach)
template<typename T, template<typename U> class Descriptor>
class GuoExternalForceBGKdynamics : public ExternalForceDynamics<T,Descriptor> {
public:
/* *************** Construction / Destruction ************************ */
    GuoExternalForceBGKdynamics(T omega_);

    /// Clone the object on its dynamic type.
    virtual GuoExternalForceBGKdynamics<T,Descriptor>* clone() const;

    /// Return a unique ID for this class.
    virtual int getId() const;

/* *************** Collision and Equilibrium ************************* */

    /// Implementation of the collision step
    virtual void collide(Cell<T,Descriptor>& cell,
                         BlockStatistics& statistics_);

    /// Compute equilibrium distribution function
    virtual T computeEquilibrium(plint iPop, T rhoBar, Array<T,Descriptor<T>::d> const& j,
                                 T jSqr, T thetaBar=T()) const;
private:
    static int id;
};

/// Implementation of O(Ma^2) BGK dynamics with an external force (Shan/Chen approach)
template<typename T, template<typename U> class Descriptor>
class ShanChenExternalForceBGKdynamics : public ExternalForceDynamics<T,Descriptor> {
public:
/* *************** Construction / Destruction ************************ */
    ShanChenExternalForceBGKdynamics(T omega_);

    /// Clone the object on its dynamic type.
    virtual ShanChenExternalForceBGKdynamics<T,Descriptor>* clone() const;

    /// Return a unique ID for this class.
    virtual int getId() const;

/* *************** Collision and Equilibrium ************************* */

    /// Implementation of the collision step
    virtual void collide(Cell<T,Descriptor>& cell,
                         BlockStatistics& statistics_);

    /// Compute equilibrium distribution function
    virtual T computeEquilibrium(plint iPop, T rhoBar, Array<T,Descriptor<T>::d> const& j,
                                 T jSqr, T thetaBar=T()) const;
private:
    static int id;
};

/// Implementation of O(Ma^2) BGK dynamics with an external force (He et al. approach)
template<typename T, template<typename U> class Descriptor>
class HeExternalForceBGKdynamics : public ExternalForceDynamics<T,Descriptor> {
public:
/* *************** Construction / Destruction ************************ */
    HeExternalForceBGKdynamics(T omega_);

    /// Clone the object on its dynamic type.
    virtual HeExternalForceBGKdynamics<T,Descriptor>* clone() const;

    /// Return a unique ID for this class.
    virtual int getId() const;

/* *************** Collision and Equilibrium ************************* */

    /// Implementation of the collision step
    virtual void collide(Cell<T,Descriptor>& cell,
                         BlockStatistics& statistics_);

    /// Compute equilibrium distribution function
    virtual T computeEquilibrium(plint iPop, T rhoBar, Array<T,Descriptor<T>::d> const& j,
                                 T jSqr, T thetaBar=T()) const;
private:
    static int id;
};


/// Incompressible version of the Guo external forcing
template<typename T, template<typename U> class Descriptor>
class IncGuoExternalForceBGKdynamics : public ExternalForceDynamics<T,Descriptor> {
public:
    /* *************** Construction / Destruction ************************ */
    IncGuoExternalForceBGKdynamics(T omega_);
    
    /// Clone the object on its dynamic type.
    virtual IncGuoExternalForceBGKdynamics<T,Descriptor>* clone() const;
    
    /// Return a unique ID for this class.
    virtual int getId() const;
    
    /* *************** Collision and Equilibrium ************************* */
    
    /// Implementation of the collision step
    virtual void collide(Cell<T,Descriptor>& cell,
                         BlockStatistics& statistics_);
    
    /// Compute equilibrium distribution function
    virtual T computeEquilibrium(plint iPop, T rhoBar, Array<T,Descriptor<T>::d> const& j,
                                 T jSqr, T thetaBar=T()) const;
                                 
     ///  Is it in the incompressible BGK model
    virtual bool velIsJ() const;
private:
    static int id;
};

/// Implementation of O(Ma^2) BGK dynamics with an external force (Guo approach)
template<typename T, template<typename U> class Descriptor>
class ShanChenExternalForceRegularizedBGKdynamics : public ExternalForceDynamics<T,Descriptor> {
public:
    /* *************** Construction / Destruction ************************ */
    ShanChenExternalForceRegularizedBGKdynamics(T omega_);
    
    /// Clone the object on its dynamic type.
    virtual ShanChenExternalForceRegularizedBGKdynamics<T,Descriptor>* clone() const;
    
    /// Return a unique ID for this class.
    virtual int getId() const;
    
    /* *************** Collision and Equilibrium ************************* */
    
    /// Implementation of the collision step
    virtual void collide(Cell<T,Descriptor>& cell,
                         BlockStatistics& statistics_);
    
    /// Compute equilibrium distribution function
    virtual T computeEquilibrium(plint iPop, T rhoBar, Array<T,Descriptor<T>::d> const& j,
                                 T jSqr, T thetaBar=T()) const;
private:
    static int id;
};

}  // namespace plb

#endif  // EXTERNAL_FORCE_DYNAMICS_H
