/*
 *  Copyright (c) 2008 Cyrille Berger <cberger@cberger.net>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation;
 * either version 2, or (at your option) any later version of the License.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library; see the file COPYING.  If not, write to
 * the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
 * Boston, MA 02110-1301, USA.
 */

#include "VariableNG_p.h"

#include <llvm/Constants.h>
#include <llvm/Instructions.h>

#include "LLVMBackend/CodeGenerator_p.h"
#include "Debug.h"
#include "LLVMBackend/ExpressionResult_p.h"
#include "LLVMBackend/GenerationContext_p.h"
#include "Type.h"
#include "Type_p.h"
#include "LLVMBackend/Visitor_p.h"

using namespace GTLCore;
using namespace LLVMBackend;

struct VariableNG::Private
{
  const Visitor* visitor;
  const GTLCore::Type* type;
  bool constant;
  bool isInitialised;
  llvm::Value* pointer;
  bool constantPointer;
  bool dependant;
  bool isInMemoryObject()
  {
    return type->dataType() == Type::ARRAY or type->dataType() == Type::STRUCTURE;
  }
  bool storedOnStack;
#ifdef OPENGTL_ENABLE_DEBUG_OUTPUT
  int dependantWasChangedToConst;
#endif
};

VariableNG::VariableNG(const GTLCore::Type* _type, bool _constant, bool _dependant ) : d(new Private)
{
  d->type = _type;
  d->constant = _constant;
  d->isInitialised = false;
  d->visitor = LLVMBackend::Visitor::getVisitorFor( _type );
  d->pointer = 0;
  d->constantPointer = false;
  d->dependant = _dependant;
  d->storedOnStack = false;
#ifdef OPENGTL_ENABLE_DEBUG_OUTPUT
  d->dependantWasChangedToConst = 2;
#endif
}

VariableNG::~VariableNG()
{
  delete d;
}
bool VariableNG::constant() const
{
  return d->constant;
}

void VariableNG::setConstant( bool v )
{
  GTL_ASSERT( d->dependant );
#ifdef OPENGTL_ENABLE_DEBUG_OUTPUT
  if( not v )
  {
    --d->dependantWasChangedToConst;
    GTL_ASSERT( d->dependantWasChangedToConst >= 0 );
  }
#endif
  d->constant = v;
}

const GTLCore::Type* VariableNG::type() const
{
  return d->type;
}

llvm::BasicBlock* VariableNG::initialise( GenerationContext& _generationContext, llvm::BasicBlock* _bb, const ExpressionResult& _initialiser, const std::list<llvm::Value*>& _initialSize )
{
  GTL_DEBUG( _initialSize.size() );
  d->storedOnStack = true;
  llvm::Value* pointer_ = 0;
  if( _initialiser.value() and _initialiser.functionResult()
      and (type()->dataType() == Type::ARRAY or type()->dataType() == Type::STRUCTURE ) )
  {
    pointer_ = new llvm::AllocaInst( d->type->d->pointerType(_generationContext.llvmContext()), llvm::ConstantInt::get(llvm::Type::getInt32Ty(_generationContext.llvmContext()), 1), "Variable", _bb);
    new llvm::StoreInst(_initialiser.value(), pointer_, _bb);
    initialise( _generationContext, _bb, pointer_ );
    _bb = d->visitor->mark( _generationContext, _bb, pointer(_bb), d->type, CodeGenerator::integerToConstant( _generationContext.llvmContext(), INT32_C(1) ) );
  } else {
    if( d->isInMemoryObject() )
    {
      pointer_ = new llvm::AllocaInst( d->type->d->pointerType(_generationContext.llvmContext()), llvm::ConstantInt::get(llvm::Type::getInt32Ty(_generationContext.llvmContext()), 1), "Variable", _bb);
      llvm::Value* value = CodeGenerator::allocateMemory(_generationContext, d->type->d->type(_generationContext.llvmContext()), CodeGenerator::integerToConstant(_generationContext.llvmContext(), INT32_C(1)), _bb);
      new llvm::StoreInst(value, pointer_, _bb);
    } else {
      pointer_ = new llvm::AllocaInst( d->type->d->type(_generationContext.llvmContext()), llvm::ConstantInt::get(llvm::Type::getInt32Ty(_generationContext.llvmContext()), 1), "Variable", _bb);
    }
    initialise( _generationContext, _bb, pointer_ );
    _bb = d->visitor->initialise( _generationContext, _bb, pointer(_bb), d->type, _initialSize);
    
    if( _initialiser.value() )
    {
      _bb = d->visitor->set( _generationContext, _bb, pointer(_bb), d->type, _initialiser.value(), _initialiser.type());
    }
  }
  d->constantPointer = false;
  return _bb;
}

void VariableNG::initialise( GenerationContext& _generationContext, llvm::BasicBlock* _currentBlock, llvm::Value* _pointer )
{
  GTL_ASSERT( _pointer->getType()->getTypeID() == llvm::Type::PointerTyID );
  GTL_ASSERT(not d->isInitialised);
  GTL_DEBUG(*_pointer << *_pointer->getType());
  d->isInitialised = true;
  d->pointer = _pointer;
  d->constantPointer = true;
}

llvm::Value* VariableNG::get(LLVMBackend::GenerationContext& _generationContext, llvm::BasicBlock* _currentBlock)
{
  llvm::Value* v = d->visitor->get( _generationContext, _currentBlock, d->pointer, d->type ).value();
  return v;
}

llvm::BasicBlock* VariableNG::set(LLVMBackend::GenerationContext& _generationContext, llvm::BasicBlock* _currentBlock, llvm::Value* _value, const GTLCore::Type* _valueType )
{
  GTL_ASSERT(not constant() );
  return d->visitor->set( _generationContext, _currentBlock, d->pointer, d->type, _value, _valueType);
}

llvm::BasicBlock* VariableNG::replacePointer( LLVMBackend::GenerationContext& _generationContext, llvm::BasicBlock* _currentBlock, llvm::Value* _pointer)
{
  GTL_DEBUG( "_pointer: " << *_pointer->getType() << " d->pointer: " << *d->pointer->getType() << " " << d->pointer );
  GTL_ASSERT( _pointer->getType() == d->pointer->getType()->getContainedType(0) );
  GTL_ASSERT( not d->constantPointer );
  GTL_ASSERT( d->isInMemoryObject() );
  _currentBlock = cleanUp( _generationContext, _currentBlock, 0 );
  new llvm::StoreInst(_pointer, d->pointer, _currentBlock);
  _currentBlock = d->visitor->mark( _generationContext, _currentBlock, _pointer, d->type, CodeGenerator::integerToConstant( _generationContext.llvmContext(), INT32_C(1) ) );
  return _currentBlock;
}

llvm::Value* VariableNG::pointer(llvm::BasicBlock* _currentBlock)
{
  GTL_DEBUG( *d->pointer->getType() << *d->pointer );
  if(d->isInMemoryObject() and d->storedOnStack)
  {
    return new llvm::LoadInst(d->pointer, "", _currentBlock);
  }
  return d->pointer;
}

bool VariableNG::constantPointer() const
{
  return d->constantPointer;
}

llvm::BasicBlock* VariableNG::cleanUp( LLVMBackend::GenerationContext& _generationContext, llvm::BasicBlock* _currentBlock, llvm::Value* _donttouch )
{
  if( d->isInMemoryObject() and not d->constantPointer )
  {
    llvm::Value* pointer_ = pointer( _currentBlock);
    _currentBlock = d->visitor->mark( _generationContext, _currentBlock, pointer_, d->type, CodeGenerator::integerToConstant( _generationContext.llvmContext(), INT32_C(-1) ) );
    // pointer_ is deallocated inside cleanup
    _currentBlock = d->visitor->cleanUp( _generationContext, _currentBlock, pointer_, d->type, _donttouch, true, true );
  }
  return _currentBlock;
}
