/* pdu-icmp.c
  
   PDU builder for ICMP messages

   Copyright (C) 2007, 2008, 2009, 2010 Eloy Paris

   This is part of Network Expect.

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.
    
   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.
    
   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

#include "pbuild-priv.h"
#include "pdu-icmp.h"

/*
 * Most ICMP builders only need to calculate the ICMP checksum in the
 * second pass. This generic second pass builder can be used for all these
 * ICMP builders.
 */
static void
pdu_icmp_generic_pass2_builder(const GNode *pdu, void *dest,
			       void *prev_pdu_hdr _U_)
{
    struct icmp_hdr *icmp;
    uint16_t cksum;
    void *field;
    size_t hdr_len, opts_len, payload_len;

    icmp = dest;

    if ( (field = _pb_pdata(pdu, "cksum") ) )
	cksum = htons(num_next(field) );
    else {
	/* Must be 0 before calculating cksum! */
	SSVAL(icmp, offsetof(struct icmp_hdr, icmp_cksum), 0);

	hdr_len = ( (struct node_data *) pdu->data)->_data_pdu.hdr_len;
	opts_len = ( (struct node_data *) pdu->data)->_data_pdu.opts_len;
	payload_len = ( (struct node_data *) pdu->data)->_data_pdu.payload_len;

	cksum = _pb_cksum(dest, hdr_len + opts_len + payload_len);
    }

    SSVAL(icmp, offsetof(struct icmp_hdr, icmp_cksum), cksum);
}

/*********************************************************************/
/************************* Raw ICMP message **************************/
/*********************************************************************/

static void
pdu_icmprawhdr_builder(const GNode *pdu, void *dest)
{
    struct icmp_hdr *icmp;

    icmp = dest;

    icmp->icmp_type = num_next(_pb_pdata(pdu, "type") );
    icmp->icmp_code = num_next(_pb_pdata(pdu, "code") );
}

#if 0
static void
pdu_icmprawhdr_dumper(pdu_t *p, const char *prefix)
{
    struct icmphdr_options *hdr_data;

    hdr_data = p->header_data;

    printf("%s  Parameters:\n", prefix);
    printf("%s    ICMP Type: %s\n", prefix, num_info(hdr_data->_raw.type) );
    printf("%s    ICMP Code: %s\n", prefix, num_info(hdr_data->_raw.code) );
    printf("%s    ICMP checksum: %s\n", prefix,
	   hdr_data->cksum ? num_info(hdr_data->cksum) : "automatic");
}
#endif

/*********************************************************************/
/******************** ICMP echo request and reply ********************/
/*********************************************************************/

static void
pdu_icmpecho_builder(const GNode *pdu, void *dest)
{
    struct icmpecho_hdr *icmp;
    uint16_t u16;
    void *field;

    icmp = dest;

    icmp->type = ICMP_ECHO;
    icmp->code = 0;

    field = _pb_pdata(pdu, "id");

    if (!strcmp( ( (struct node_data *) pdu->data)->name, "icmp-echo") )
	/* Use process ID for the ICMP echo ID, if none was specified */
	u16 = htons(field ? num_next(field) : (uint32_t) getpid() );
    else
	u16 = htons(num_next(field) );
    SSVAL(icmp, offsetof(struct icmpecho_hdr, id), u16);

    u16 = htons(num_next(_pb_pdata(pdu, "seq") ) );
    SSVAL(icmp, offsetof(struct icmpecho_hdr, seq), u16);
}

#if 0
static void
pdu_icmpechohdr_dumper(pdu_t *p, const char *prefix)
{
    struct icmphdr_options *hdr_data;

    hdr_data = p->header_data;

    printf("%s  Parameters:\n", prefix);
    printf("%s    ICMP ID: %s\n", prefix, num_info(hdr_data->_echo.id) );
    printf("%s    ICMP Sequence: %s\n", prefix, num_info(hdr_data->_echo.seq) );
    printf("%s    ICMP checksum: %s\n", prefix,
	   hdr_data->cksum ? num_info(hdr_data->cksum) : "automatic");
}
#endif

static void
pdu_icmpechoreply_builder(const GNode *pdu, void *dest)
{
    struct icmpecho_hdr *icmp;
    uint16_t u16;

    icmp = dest;

    icmp->type = ICMP_ECHOREPLY;
    icmp->code = 0;
    u16 = htons(num_next(_pb_pdata(pdu, "id") ) );
    SSVAL(icmp, offsetof(struct icmpecho_hdr, id), u16);
    u16 = htons(num_next(_pb_pdata(pdu, "seq") ) );
    SSVAL(icmp, offsetof(struct icmpecho_hdr, seq), u16);
}

/*********************************************************************/
/*************************** ICMP redirect ***************************/
/*********************************************************************/

static void
pdu_icmpredirect_builder(const GNode *pdu, void *dest)
{
    struct icmpredirect_hdr *icmp;
    ip_addr_t ipaddr;

    icmp = dest;

    icmp->type = ICMP_REDIRECT;
    icmp->code = 0;

    ipaddr = ip_next(_pb_pdata(pdu, "gateway") );
    SIVAL(icmp, offsetof(struct icmpredirect_hdr, gateway), ipaddr);
}

#if 0
static void
pdu_icmpredirecthdr_dumper(pdu_t *p, const char *prefix)
{
    struct icmphdr_options *hdr_data;
    char addr[INET_ADDRSTRLEN];

    hdr_data = p->header_data;

    printf("%s  Parameters:\n", prefix);
    printf("%s    New gateway: %s\n", prefix,
	   inet_ntop(AF_INET, &hdr_data->_redirect.gateway, addr,
		     INET_ADDRSTRLEN) );
    printf("%s    ICMP checksum: %s\n", prefix,
	   hdr_data->cksum ? num_info(hdr_data->cksum) : "automatic");
}
#endif

/*********************************************************************/
/************************* ICMP source quench ************************/
/*********************************************************************/

static void
pdu_icmpsrcquench_builder(const GNode *pdu _U_, void *dest)
{
    struct icmpsrcquench_hdr *icmp;

    icmp = (struct icmpsrcquench_hdr *) dest;

    icmp->type = ICMP_SOURCE_QUENCH;
    icmp->code = 0;
    icmp->unused = 0;
}

/*********************************************************************/
/*********************** ICMP parameter problem **********************/
/*********************************************************************/

static void
pdu_icmpparmproblemhdr_builder(const GNode *pdu, void *dest)
{
    struct icmpparmproblem_hdr *icmp;

    icmp = dest;

    icmp->type = ICMP_PARAMETERPROB;
    icmp->code = 0;
    icmp->ptr = num_next(_pb_pdata(pdu, "ptr") );
    icmp->unused1 = 0;
    icmp->unused2 = 0;
}

#if 0
static void
pdu_icmpparmproblemhdr_dumper(pdu_t *p, const char *prefix)
{
    struct icmphdr_options *hdr_data;

    hdr_data = p->header_data;

    printf("%s  Parameters:\n", prefix);
    printf("%s    Pointer: %s\n", prefix,
	   num_info(hdr_data->_parmproblem.ptr) );
    printf("%s    ICMP checksum: %s\n", prefix,
	   hdr_data->cksum ? num_info(hdr_data->cksum) : "automatic");
}
#endif

/*********************************************************************/
/************************* ICMP time exceeded ************************/
/*********************************************************************/

static void
pdu_icmptimeexceed_builder(const GNode *pdu, void *dest)
{
    struct icmptimeexceed_hdr *icmp;

    icmp = dest;

    icmp->type = ICMP_TIME_EXCEEDED;
    icmp->code = num_next(_pb_pdata(pdu, "code") );
    icmp->unused = 0;
}

#if 0
static void
pdu_icmptimeexceedhdr_dumper(pdu_t *p, const char *prefix)
{
    struct icmphdr_options *hdr_data;

    hdr_data = p->header_data;

    printf("%s  Parameters:\n", prefix);
    printf("%s    ICMP Code: %s\n", prefix,
	   num_info(hdr_data->_timeexceeded.code) );
    printf("%s    ICMP checksum: %s\n", prefix,
	   hdr_data->cksum ? num_info(hdr_data->cksum) : "automatic");
}
#endif

/*********************************************************************/
/************************** ICMP unreachable *************************/
/*********************************************************************/

#if 0
static void
pdu_icmpunreachablehdr_dumper(pdu_t *p, const char *prefix)
{
    struct icmphdr_options *hdr_data;

    hdr_data = p->header_data;

    printf("%s  Parameters:\n", prefix);
    printf("%s    ICMP Code: %s\n", prefix,
	   num_info(hdr_data->_unreachable.code) );
    printf("%s    Next-hop MTU: %s\n", prefix,
	   num_info(hdr_data->_unreachable.nexthop_mtu) );
    printf("%s    ICMP checksum: %s\n", prefix,
	   hdr_data->cksum ? num_info(hdr_data->cksum) : "automatic");
}
#endif

static const pdu_t pdu_icmpraw = {
    .name = "icmp",
    .description = "Raw ICMP message",
    .len = sizeof(struct icmp_hdr),
    .fields = (field_t []) {
	{.name = "cksum", .type = PDU_FTYPE_NUMTYPE},
	{.name = "type", .type = PDU_FTYPE_NUMTYPE},
	{.name = "code", .type = PDU_FTYPE_NUMTYPE},
	{.name = NULL}
    },
    .documented_in = "RFC 792",
    .build = &pdu_icmprawhdr_builder,
    .postbuild = &pdu_icmp_generic_pass2_builder,
};

static const pdu_t pdu_icmpecho = {
    .name = "icmp-echo",
    .description = "ICMP echo request",
    .len = sizeof(struct icmpecho_hdr),
    .fields = (field_t []) {
	{.name = "cksum", .type = PDU_FTYPE_NUMTYPE},
	{.name = "id", .type = PDU_FTYPE_NUMTYPE},
	{.name = "seq", .type = PDU_FTYPE_NUMTYPE},
	{.name = NULL}
    },
    .documented_in = "RFC 792",
    .build = &pdu_icmpecho_builder,
    .postbuild = &pdu_icmp_generic_pass2_builder,
};

static const pdu_t pdu_icmpechoreply = {
    .name = "icmp-echoreply",
    .description = "ICMP echo reply",
    .len = sizeof(struct icmpecho_hdr),
    .fields = (field_t []) {
	{.name = "cksum", .type = PDU_FTYPE_NUMTYPE},
	{.name = "id", .type = PDU_FTYPE_NUMTYPE},
	{.name = "seq", .type = PDU_FTYPE_NUMTYPE},
	{.name = NULL}
    },
    .documented_in = "RFC 792",
    .build = &pdu_icmpechoreply_builder,
    .postbuild = &pdu_icmp_generic_pass2_builder,
};

static const pdu_t pdu_icmpredirect = {
    .name = "icmp-redirect",
    .description = "ICMP redirect",
    .documented_in = "RFC 792",
    .len = sizeof(struct icmpredirect_hdr),
    .build = &pdu_icmpredirect_builder,
    .postbuild = &pdu_icmp_generic_pass2_builder,
};

static const pdu_t pdu_icmpsrcquench = {
    .name = "icmp-srcquench",
    .description = "ICMP source quench",
    .documented_in = "RFC 792",
    .len = sizeof(struct icmpsrcquench_hdr),
    .build = &pdu_icmpsrcquench_builder,
    .postbuild = &pdu_icmp_generic_pass2_builder,
};

static const pdu_t pdu_icmpparmproblem = {
    .name = "icmp-parmprob",
    .description = "ICMP parameter problem",
    .documented_in = "RFC 792",
    .len = sizeof(struct icmpparmproblem_hdr),
    .build = &pdu_icmpparmproblemhdr_builder,
    .postbuild = &pdu_icmp_generic_pass2_builder,
};

static const pdu_t pdu_icmptimeexceed = {
    .name = "icmp-timeexceed",
    .description = "ICMP time exceeded",
    .documented_in = "RFC 792",
    .len = sizeof(struct icmptimeexceed_hdr),
    .build = &pdu_icmptimeexceed_builder,
    .postbuild = &pdu_icmp_generic_pass2_builder,
};

static const pdu_t pdu_icmpunreachable = {
    .name = "icmp-unreachable",
    .description = "ICMP unreachable",
    .documented_in = "RFC 792",
    .len = sizeof(struct icmpunreachable_hdr),
    .fields = (field_t []) {
	{
	    .name = "type",
	    .type = PDU_FTYPE_UINT8,
	    .offset = offsetof(struct icmpunreachable_hdr, type),
	    .defval = (defval_t []) {
		{.type = PDU_DEF_NUM, ._number = ICMP_DEST_UNREACH}
	    }
	},
	{
	    .name = "code",
	    .type = PDU_FTYPE_UINT8,
	    .offset = offsetof(struct icmpunreachable_hdr, code),
	    .data = (struct pdu_dict []) { /* Enumeration */
		{"net",            (uint32_t []) {0x00} },
		{"host",           (uint32_t []) {0x01} },
		{"protocol",       (uint32_t []) {0x02} },
		{"port",           (uint32_t []) {0x03} },
		{"fragneeded",     (uint32_t []) {0x04} },
		{"sr-failed",      (uint32_t []) {0x05} },
		{"admin-prohibit", (uint32_t []) {0x0d} },
		{NULL,	           NULL}
	    },
	    .defval = (defval_t []) { {.type = PDU_DEF_NUM, ._number = 1} }
	},
	{
	    .name = "mtu",
	    .type = PDU_FTYPE_UINT16,
	    .offset = offsetof(struct icmpunreachable_hdr, nexthop_mtu)
	},
	{
	    .name = NULL
	}
    },
    .postbuild = &pdu_icmp_generic_pass2_builder,
};

void
_pb_register_icmp(void)
{
    _pb_register_protocol(&pdu_icmpraw);
    _pb_register_protocol(&pdu_icmpecho);
    _pb_register_protocol(&pdu_icmpechoreply);
    _pb_register_protocol(&pdu_icmpredirect);
    _pb_register_protocol(&pdu_icmpsrcquench);
    _pb_register_protocol(&pdu_icmpparmproblem);
    _pb_register_protocol(&pdu_icmptimeexceed);
    _pb_register_protocol(&pdu_icmpunreachable);
}
