# Licensed under public domain according to web site from
# which it was downloaded
# http://www.jejik.com/articles/2007/02/a_simple_unix_linux_daemon_in_python/
#
# Sander Marechal (http://www.jejik.com)
# Posted on 2008-11-26@16:40
# @Anonymous Coward
#
# This is really basic stuff and not very original. It's Public Domain, so do
# with it as you please.

import sys
import os
import atexit
import time
from signal import SIGTERM


class Daemon:
    """
     A generic daemon class.

     Usage: subclass the Daemon class and override the run() method
     """

    def __init__(self, pidfile, no_fork=False, stdin='/dev/null',
                 stdout='/dev/null', stderr='/dev/null'):
        self.stdin = stdin
        self.stdout = stdout
        self.stderr = stderr
        self.pidfile = pidfile
        self.no_fork = no_fork

    def _get_pid(self):
        try:
            pf = file(self.pidfile, 'r')
            pid = int(pf.read().strip())
            pf.close()
        except IOError:
            pid = None
        except ValueError:
            pid = None
        return pid

    @staticmethod
    def _write_pid(pid_file):
        pid = str(os.getpid())
        with os.fdopen(os.open(pid_file,
            (os.O_CREAT | os.O_RDWR), 0644), "w+") as o:
            o.write("%s\n" % pid)

    def daemonize(self):
        """
          do the UNIX double-fork magic, see Stevens' "Advanced
          Programming in the UNIX Environment" for details (ISBN 0201563177)
          """
        try:
            pid = os.fork()
            if pid > 0:
                # exit first parent
                sys.exit(0)
        except OSError, e:
            sys.stderr.write(
                "fork #1 failed: %d (%s)\n" % (e.errno, e.strerror))
            sys.exit(1)

        # decouple from parent environment
        os.chdir("/")
        os.setsid()
        os.umask(0)

        # do second fork
        try:
            pid = os.fork()
            if pid > 0:
                # exit from second parent
                sys.exit(0)
        except OSError, e:
            sys.stderr.write(
                "fork #2 failed: %d (%s)\n" % (e.errno, e.strerror))
            sys.exit(1)

        # redirect standard file descriptors
        sys.stdout.flush()
        sys.stderr.flush()
        si = file(self.stdin, 'r')
        so = file(self.stdout, 'a+')
        se = file(self.stderr, 'a+', 0)
        os.dup2(si.fileno(), sys.stdin.fileno())
        os.dup2(so.fileno(), sys.stdout.fileno())
        os.dup2(se.fileno(), sys.stderr.fileno())

    def delpid(self):
        os.remove(self.pidfile)

    def start(self):
        """
          Start the daemon
          """
        # Check for a pidfile to see if the daemon already runs
        pid = self._get_pid()

        if pid:
            message = "pidfile %s already exist. Daemon already running?\n"
            sys.stderr.write(message % self.pidfile)
            sys.exit(1)

        # Start the daemon
        if not self.no_fork:
            self.daemonize()

        # write pidfile
        self._write_pid(self.pidfile)
        atexit.register(self.delpid)

        self.run()

    def stop(self):
        """
          Stop the daemon
          """
        # Get the pid from the pidfile
        pid = self._get_pid()

        if not pid:
            message = "pidfile %s does not exist. Daemon not running?\n"
            sys.stderr.write(message % self.pidfile)
            return  # not an error in a restart

        # Try killing the daemon process
        try:
            while 1:
                os.kill(pid, SIGTERM)
                time.sleep(0.1)
        except OSError, err:
            err = str(err)
            if err.find("No such process") > 0:
                if os.path.exists(self.pidfile):
                    os.remove(self.pidfile)
            else:
                print str(err)
                sys.exit(1)

    def restart(self):
        """
          Restart the daemon
          """
        self.stop()
        self.start()

    def run(self):
        """
          You should override this method when you subclass Daemon. It will be
          called after the process has been daemonized by start() or restart().
          """
