/* psn.cc
 */
#include "osl/record/psn.h"
#include "osl/state/simpleState.h"

const std::string osl::record::psn::
show(Square pos)
{
  const int x = pos.x();
  const int y = pos.y();
  std::string result = "XX";
  result[0] = x + '0';
  result[1] = y + 'a' - 1;
  return result;
}

char osl::record::psn::
show(Ptype ptype)
{
  switch (ptype)
  {
  case PAWN:	return 'P';
  case LANCE:	return 'L';
  case KNIGHT:	return 'N';
  case SILVER:	return 'S';
  case GOLD:	return 'G';
  case BISHOP:	return 'B';
  case ROOK:	return 'R';
  case KING:	return 'K';
  default:
    assert("unsupported ptype" == 0);
    return '!';
  }
}

const std::string osl::record::psn::
show(Move m)
{
  const Square from = m.from();
  const Square to = m.to();
  if (from.isPieceStand())
  {
    std::string result = "X*";
    result[0] = show(m.ptype());
    result += show(to);
    return result;
  }
  std::string result = show(from);
  result += show(to);
  if (m.promoteMask())
    result += '+';
  return result;
}

const std::string osl::record::psn::
showXP(Move m)
{
  if (m.isInvalid())
    return "resign";  
  if (m.isPass())
    return "pass";
  const Square from = m.from();
  const Square to = m.to();
  if (from.isPieceStand())
  {
    std::string result = "X*";
    result[0] = show(m.ptype());
    result += show(to);
    return result;
  }
  std::string result = show(from);
  if (m.capturePtype() != PTYPE_EMPTY)
    result += 'x';
  result += show(to);
  if (m.isPromotion())
    result += '+';
  else if (canPromote(m.ptype())
	   && (from.canPromote(m.player()) || to.canPromote(m.player())))
    result += '=';
  return result;
}


const osl::Move osl::record::psn::
strToMove(const std::string& str, const SimpleState& s)
{
  if (str.size() < 4)
    throw ParseError("Invalid move string: " + str);

  const Square to = strToPos(str.substr(2,2));
  if (str[1] == '*')
  {
    const Ptype ptype = charToPtype(str[0]);
    return Move(to, ptype, s.turn());
  }

  const Square from = strToPos(str.substr(0,2));
  const Ptype ptype = s.pieceOnBoard(from).ptype();
  const Ptype captured = s.pieceOnBoard(to).ptype();
  if (! isPiece(ptype))
    throw ParseError("No piece on square: " + str);    
  bool promotion = false;
  if (str.size() > 4)
  {
    assert(str[4] == '+');
    promotion = true;
  }
  return Move(from, to, (promotion ? promote(ptype) : ptype), 
	      captured, promotion, s.turn());
}

const osl::Square osl::record::psn::
strToPos(const std::string& str)
{
  assert(str.size() == 2);
  const int x = str[0] - '0';
  const int y = str[1] - 'a' + 1;
  if (x <= 0 || x > 9 || y <= 0 || y > 9)
    throw ParseError("Invalid square character: " + str);
  return Square(x, y);
}

osl::Ptype osl::record::psn::
charToPtype(char c)
{
  switch (c) 
  {
  case 'P': return PAWN;
  case 'L': return LANCE;
  case 'N': return KNIGHT;
  case 'S': return SILVER;
  case 'G': return GOLD;
  case 'B': return BISHOP;
  case 'R': return ROOK;
  case 'K': return KING;
  default:
    return PTYPE_EMPTY;
  }
}

/* ------------------------------------------------------------------------- */
// ;;; Local Variables:
// ;;; mode:c++
// ;;; c-basic-offset:2
// ;;; End:
