set(derived_headers_location "${CMAKE_CURRENT_BINARY_DIR}/DerivedHeaders")

# Obtain the original headers from their staged location in the build directory.
set(original_headers_location "${CMAKE_BINARY_DIR}/include/lldb")
set(headers_to_process
  SBDefines.h
  lldb-defines.h
  lldb-enumerations.h
  lldb-types.h
)

file(MAKE_DIRECTORY ${derived_headers_location})

# Take the original headers and convert them RPC as necessary using the conversion script.
set(original_headers)
set(derived_headers)
foreach(header ${headers_to_process})
  set(original_header "${original_headers_location}/${header}")

  get_filename_component(header_filename ${header} NAME)
  string(REPLACE "lldb-" "lldb-rpc-" rpc_header_filename "${header_filename}")
  set(derived_header "${derived_headers_location}/${rpc_header_filename}")

  list(APPEND original_headers "${original_header}")
  list(APPEND derived_headers "${derived_header}")
  add_custom_command(OUTPUT ${derived_header}
    COMMAND ${Python3_EXECUTABLE} ${LLDB_SOURCE_DIR}/scripts/convert-lldb-header-to-rpc-header.py
            ${original_header} ${derived_header}
    DEPENDS ${original_header}

    COMMENT "Creating ${derived_header}"
  )
endforeach()

# Do the same thing for any header files that were autogenerated.
set(generated_headers_to_process
  API/SBLanguages.h
)
foreach(header ${generated_headers_to_process})
  set(original_header "${LLDB_OBJ_DIR}/include/lldb/${header}")

  get_filename_component(header_filename ${header} NAME)
  string(REPLACE "lldb-" "lldb-rpc-" rpc_header_filename "${header_filename}")
  set(derived_header "${derived_headers_location}/${rpc_header_filename}")

  list(APPEND original_headers "${original_header}")
  list(APPEND derived_headers "${derived_header}")
  add_custom_command(OUTPUT ${derived_header}
    COMMAND ${CMAKE_COMMAND} -E copy ${original_header} ${derived_header}
    COMMAND ${Python3_EXECUTABLE} ${LLDB_SOURCE_DIR}/scripts/convert-lldb-header-to-rpc-header.py
            ${original_header} ${derived_header}
    DEPENDS lldb-sbapi-dwarf-enums

    COMMENT "Creating ${derived_header}"
  )
endforeach()

add_custom_target(copy-aux-rpc-headers DEPENDS ${derived_headers})
add_dependencies(copy-aux-rpc-headers liblldb-header-staging)

list(APPEND public_headers
  ${derived_headers_location}/SBDefines.h
  ${derived_headers_location}/SBLanguages.h
  ${derived_headers_location}/lldb-rpc-enumerations.h
  ${derived_headers_location}/lldb-rpc-types.h
  ${derived_headers_location}/lldb-rpc-defines.h
)

# Collect and preprocess headers for the framework bundle
set(version_header
  ${derived_headers_location}/lldb-rpc-defines.h
)

function(FixIncludePaths in subfolder out)
  get_filename_component(base_name ${in} NAME)
  set(parked_header ${CMAKE_CURRENT_BINARY_DIR}/ParkedHeaders/${subfolder}/${base_name})
  set(${out} ${parked_header} PARENT_SCOPE)
  find_program(unifdef_EXECUTABLE unifdef)

  add_custom_command(OUTPUT ${parked_header}
    COMMAND ${LLDB_SOURCE_DIR}/scripts/framework-header-fix.py
            -f lldb_rpc -i ${in} -o ${parked_header} -p ${unifdef_EXECUTABLE} --unifdef_guards USWIG
    DEPENDS ${in}
    COMMENT "Fixing includes in ${in}"
  )
endfunction()

set(preprocessed_headers)

# Apply include-paths fix and any version fix on all headers and park them.
foreach(source_header ${public_headers})
  FixIncludePaths(${source_header} Headers parked_header)
  list(APPEND preprocessed_headers ${parked_header})
endforeach()

# Wrap header preprocessing in a target, so liblldbrpc can depend on.
add_custom_target(liblldbrpc-headers DEPENDS ${preprocessed_headers})
add_dependencies(liblldbrpc-headers copy-aux-rpc-headers liblldb-header-staging)
set_target_properties(liblldbrpc-headers PROPERTIES
  LIBRARY_OUTPUT_DIRECTORY ${CMAKE_CURRENT_BINARY_DIR}/ParkedHeaders
)
