﻿// Licensed to the .NET Foundation under one or more agreements.
// The .NET Foundation licenses this file to you under the MIT license.

using System.Collections.Generic;
using System.Diagnostics;
using System.IO;
using System.Linq;
using System.Text.RegularExpressions.Generator;
using System.Threading.Tasks;
using Xunit;

namespace System.Text.RegularExpressions.Tests
{
    [ConditionalClass(typeof(RegexGeneratorOutputTests), nameof(GeneratorOutputTestsSupported))]
    public partial class RegexGeneratorOutputTests
    {
        public static bool GeneratorOutputTestsSupported =>
            PlatformDetection.IsReflectionEmitSupported &&
            PlatformDetection.IsNotMobile &&
            PlatformDetection.IsNotBrowser &&
            typeof(RegexGenerator).Assembly.GetCustomAttributes(false).OfType<DebuggableAttribute>().Any(da => da.IsJITTrackingEnabled); // output differs between debug and release

        // This exists to ensure we're aware of any egregious breaks to formatting / readability.
        // Any updates that impact the generated code in these baselines will need to be updated
        // as changes are made to the code emitted by the generator.

        [Theory]
        [MemberData(nameof(ValidateExpectedOutput_MemberData))]
        public async Task ValidateExpectedOutput(string program, string expected)
        {
            string actual = await RegexGeneratorHelper.GenerateSourceText(program, allowUnsafe: true, checkOverflow: false);

            AssertExtensions.Equal(Normalize(expected), Normalize(actual));

            static string Normalize(string code)
            {
                string versionString = typeof(Generator.RegexGenerator).Assembly.GetName().Version?.ToString() ?? "";

                var input = new StringReader(code);
                var output = new StringBuilder();

                string line;
                while ((line = input.ReadLine()) != null)
                {
                    line = line.Replace("%VERSION%", versionString);

                    if (string.IsNullOrWhiteSpace(line))
                    {
                        line = "";
                    }

                    output.AppendLine(line);
                }

                return output.ToString();
            }
        }

        public static IEnumerable<object[]> ValidateExpectedOutput_MemberData()
        {
            yield return new object[]
            {
                """
                using System.Text.RegularExpressions;
                partial class C
                {
                    [GeneratedRegex(@"^(?<proto>\w+)://[^/]+?(?<port>:\d+)?/")]
                    public static partial Regex Valid();
                }
                """,

                """
                // <auto-generated/>
                #nullable enable
                #pragma warning disable CS0162 // Unreachable code
                #pragma warning disable CS0164 // Unreferenced label
                #pragma warning disable CS0219 // Variable assigned but never used

                partial class C
                {
                    /// <remarks>
                    /// Pattern:<br/>
                    /// <code>^(?&lt;proto&gt;\\w+)://[^/]+?(?&lt;port&gt;:\\d+)?/</code><br/>
                    /// Explanation:<br/>
                    /// <code>
                    /// ○ Match if at the beginning of the string.<br/>
                    /// ○ "proto" capture group.<br/>
                    ///     ○ Match a word character atomically at least once.<br/>
                    /// ○ Match the string "://".<br/>
                    /// ○ Match a character other than '/' lazily at least once.<br/>
                    /// ○ Optional (greedy).<br/>
                    ///     ○ "port" capture group.<br/>
                    ///         ○ Match ':'.<br/>
                    ///         ○ Match a Unicode digit atomically at least once.<br/>
                    /// ○ Match '/'.<br/>
                    /// </code>
                    /// </remarks>
                    [global::System.CodeDom.Compiler.GeneratedCodeAttribute("System.Text.RegularExpressions.Generator", "%VERSION%")]
                    public static partial global::System.Text.RegularExpressions.Regex Valid() => global::System.Text.RegularExpressions.Generated.Valid_0.Instance;
                }

                namespace System.Text.RegularExpressions.Generated
                {
                    using System;
                    using System.Buffers;
                    using System.CodeDom.Compiler;
                    using System.Collections;
                    using System.ComponentModel;
                    using System.Globalization;
                    using System.Runtime.CompilerServices;
                    using System.Text.RegularExpressions;
                    using System.Threading;

                    /// <summary>Custom <see cref="Regex"/>-derived type for the Valid method.</summary>
                    [GeneratedCodeAttribute("System.Text.RegularExpressions.Generator", "%VERSION%")]
                    [SkipLocalsInit]
                    file sealed class Valid_0 : Regex
                    {
                        /// <summary>Cached, thread-safe singleton instance.</summary>
                        internal static readonly Valid_0 Instance = new();

                        /// <summary>Initializes the instance.</summary>
                        private Valid_0()
                        {
                            base.pattern = "^(?<proto>\\w+)://[^/]+?(?<port>:\\d+)?/";
                            base.roptions = RegexOptions.None;
                            ValidateMatchTimeout(Utilities.s_defaultTimeout);
                            base.internalMatchTimeout = Utilities.s_defaultTimeout;
                            base.factory = new RunnerFactory();
                            base.CapNames = new Hashtable { { "0", 0 } ,  { "port", 2 } ,  { "proto", 1 }  };
                            base.capslist = new string[] {"0", "proto", "port" };
                            base.capsize = 3;
                        }

                        /// <summary>Provides a factory for creating <see cref="RegexRunner"/> instances to be used by methods on <see cref="Regex"/>.</summary>
                        private sealed class RunnerFactory : RegexRunnerFactory
                        {
                            /// <summary>Creates an instance of a <see cref="RegexRunner"/> used by methods on <see cref="Regex"/>.</summary>
                            protected override RegexRunner CreateInstance() => new Runner();

                            /// <summary>Provides the runner that contains the custom logic implementing the specified regular expression.</summary>
                            private sealed class Runner : RegexRunner
                            {
                                /// <summary>Scan the <paramref name="inputSpan"/> starting from base.runtextstart for the next match.</summary>
                                /// <param name="inputSpan">The text being scanned by the regular expression.</param>
                                protected override void Scan(ReadOnlySpan<char> inputSpan)
                                {
                                    // The pattern is anchored.  Validate the current position and try to match at it only.
                                    if (TryFindNextPossibleStartingPosition(inputSpan) && !TryMatchAtCurrentPosition(inputSpan))
                                    {
                                        base.runtextpos = inputSpan.Length;
                                    }
                                }

                                /// <summary>Search <paramref name="inputSpan"/> starting from base.runtextpos for the next location a match could possibly start.</summary>
                                /// <param name="inputSpan">The text being scanned by the regular expression.</param>
                                /// <returns>true if a possible match was found; false if no more matches are possible.</returns>
                                private bool TryFindNextPossibleStartingPosition(ReadOnlySpan<char> inputSpan)
                                {
                                    int pos = base.runtextpos;

                                    // Any possible match is at least 6 characters.
                                    if (pos <= inputSpan.Length - 6)
                                    {
                                        // The pattern leads with a beginning (\A) anchor.
                                        if (pos == 0)
                                        {
                                            return true;
                                        }
                                    }

                                    // No match found.
                                    base.runtextpos = inputSpan.Length;
                                    return false;
                                }

                                /// <summary>Determine whether <paramref name="inputSpan"/> at base.runtextpos is a match for the regular expression.</summary>
                                /// <param name="inputSpan">The text being scanned by the regular expression.</param>
                                /// <returns>true if the regular expression matches at the current position; otherwise, false.</returns>
                                private bool TryMatchAtCurrentPosition(ReadOnlySpan<char> inputSpan)
                                {
                                    int pos = base.runtextpos;
                                    int matchStart = pos;
                                    int capture_starting_pos = 0;
                                    int lazyloop_capturepos = 0;
                                    int lazyloop_pos = 0;
                                    int loop_iteration = 0;
                                    int stackpos = 0;
                                    ReadOnlySpan<char> slice = inputSpan.Slice(pos);

                                    // Match if at the beginning of the string.
                                    if (pos != 0)
                                    {
                                        UncaptureUntil(0);
                                        return false; // The input didn't match.
                                    }

                                    // "proto" capture group.
                                    {
                                        capture_starting_pos = pos;

                                        // Match a word character atomically at least once.
                                        {
                                            int iteration = 0;
                                            while ((uint)iteration < (uint)slice.Length && Utilities.IsWordChar(slice[iteration]))
                                            {
                                                iteration++;
                                            }

                                            if (iteration == 0)
                                            {
                                                UncaptureUntil(0);
                                                return false; // The input didn't match.
                                            }

                                            slice = slice.Slice(iteration);
                                            pos += iteration;
                                        }

                                        base.Capture(1, capture_starting_pos, pos);
                                    }

                                    // Match the string "://".
                                    if (!slice.StartsWith("://"))
                                    {
                                        UncaptureUntil(0);
                                        return false; // The input didn't match.
                                    }

                                    // Match a character other than '/' lazily at least once.
                                    //{
                                        if ((uint)slice.Length < 4 || slice[3] == '/')
                                        {
                                            UncaptureUntil(0);
                                            return false; // The input didn't match.
                                        }

                                        pos += 4;
                                        slice = inputSpan.Slice(pos);
                                        lazyloop_pos = pos;
                                        goto LazyLoopEnd;

                                        LazyLoopBacktrack:
                                        UncaptureUntil(lazyloop_capturepos);
                                        if (Utilities.s_hasTimeout)
                                        {
                                            base.CheckTimeout();
                                        }

                                        pos = lazyloop_pos;
                                        slice = inputSpan.Slice(pos);
                                        if (slice.IsEmpty || slice[0] == '/')
                                        {
                                            UncaptureUntil(0);
                                            return false; // The input didn't match.
                                        }
                                        pos++;
                                        slice = inputSpan.Slice(pos);
                                        lazyloop_pos = pos;

                                        LazyLoopEnd:
                                        lazyloop_capturepos = base.Crawlpos();
                                    //}

                                    // Optional (greedy).
                                    //{
                                        loop_iteration = 0;

                                        LoopBody:
                                        Utilities.StackPush(ref base.runstack!, ref stackpos, 143337952);
                                        Utilities.StackPush(ref base.runstack!, ref stackpos, base.Crawlpos(), pos);

                                        loop_iteration++;

                                        // "port" capture group.
                                        {
                                            int capture_starting_pos1 = pos;

                                            // Match ':'.
                                            if (slice.IsEmpty || slice[0] != ':')
                                            {
                                                goto LoopIterationNoMatch;
                                            }

                                            // Match a Unicode digit atomically at least once.
                                            {
                                                pos++;
                                                slice = inputSpan.Slice(pos);
                                                int iteration1 = 0;
                                                while ((uint)iteration1 < (uint)slice.Length && char.IsDigit(slice[iteration1]))
                                                {
                                                    iteration1++;
                                                }

                                                if (iteration1 == 0)
                                                {
                                                    goto LoopIterationNoMatch;
                                                }

                                                slice = slice.Slice(iteration1);
                                                pos += iteration1;
                                            }

                                            base.Capture(2, capture_starting_pos1, pos);
                                        }


                                        // The loop has an upper bound of 1. Continue iterating greedily if it hasn't yet been reached.
                                        if (loop_iteration == 0)
                                        {
                                            goto LoopBody;
                                        }
                                        goto LoopEnd;

                                        // The loop iteration failed. Put state back to the way it was before the iteration.
                                        LoopIterationNoMatch:
                                        if (--loop_iteration < 0)
                                        {
                                            // Unable to match the remainder of the expression after exhausting the loop.
                                            goto LazyLoopBacktrack;
                                        }
                                        pos = base.runstack![--stackpos];
                                        UncaptureUntil(base.runstack![--stackpos]);
                                        Utilities.ValidateStackCookie(143337952, base.runstack![--stackpos]);
                                        slice = inputSpan.Slice(pos);
                                        LoopEnd:;
                                    //}

                                    // Match '/'.
                                    if (slice.IsEmpty || slice[0] != '/')
                                    {
                                        goto LoopIterationNoMatch;
                                    }

                                    // The input matched.
                                    pos++;
                                    base.runtextpos = pos;
                                    base.Capture(0, matchStart, pos);
                                    return true;

                                    // <summary>Undo captures until it reaches the specified capture position.</summary>
                                    [MethodImpl(MethodImplOptions.AggressiveInlining)]
                                    void UncaptureUntil(int capturePosition)
                                    {
                                        while (base.Crawlpos() > capturePosition)
                                        {
                                            base.Uncapture();
                                        }
                                    }
                                }
                            }
                        }

                    }

                    /// <summary>Helper methods used by generated <see cref="Regex"/>-derived implementations.</summary>
                    [GeneratedCodeAttribute("System.Text.RegularExpressions.Generator", "%VERSION%")]
                    file static class Utilities
                    {
                        /// <summary>Default timeout value set in <see cref="AppContext"/>, or <see cref="Regex.InfiniteMatchTimeout"/> if none was set.</summary>
                        internal static readonly TimeSpan s_defaultTimeout = AppContext.GetData("REGEX_DEFAULT_MATCH_TIMEOUT") is TimeSpan timeout ? timeout : Regex.InfiniteMatchTimeout;

                        /// <summary>Whether <see cref="s_defaultTimeout"/> is non-infinite.</summary>
                        internal static readonly bool s_hasTimeout = s_defaultTimeout != Regex.InfiniteMatchTimeout;

                        /// <summary>Determines whether the character is part of the [\w] set.</summary>
                        [MethodImpl(MethodImplOptions.AggressiveInlining)]
                        internal static bool IsWordChar(char ch)
                        {
                            // Mask of Unicode categories that combine to form [\w]
                            const int WordCategoriesMask =
                                1 << (int)UnicodeCategory.UppercaseLetter |
                                1 << (int)UnicodeCategory.LowercaseLetter |
                                1 << (int)UnicodeCategory.TitlecaseLetter |
                                1 << (int)UnicodeCategory.ModifierLetter |
                                1 << (int)UnicodeCategory.OtherLetter |
                                1 << (int)UnicodeCategory.NonSpacingMark |
                                1 << (int)UnicodeCategory.DecimalDigitNumber |
                                1 << (int)UnicodeCategory.ConnectorPunctuation;

                            // Bitmap for whether each character 0 through 127 is in [\w]
                            ReadOnlySpan<byte> ascii = new byte[]
                            {
                                0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xFF, 0x03,
                                0xFE, 0xFF, 0xFF, 0x87, 0xFE, 0xFF, 0xFF, 0x07
                            };

                            // If the char is ASCII, look it up in the bitmap. Otherwise, query its Unicode category.
                            int chDiv8 = ch >> 3;
                            return (uint)chDiv8 < (uint)ascii.Length ?
                                (ascii[chDiv8] & (1 << (ch & 0x7))) != 0 :
                                (WordCategoriesMask & (1 << (int)CharUnicodeInfo.GetUnicodeCategory(ch))) != 0;
                        }

                        /// <summary>Pushes 1 value onto the backtracking stack.</summary>
                        [MethodImpl(MethodImplOptions.AggressiveInlining)]
                        internal static void StackPush(ref int[] stack, ref int pos, int arg0)
                        {
                            // If there's space available for the value, store it.
                            int[] s = stack;
                            int p = pos;
                            if ((uint)p < (uint)s.Length)
                            {
                                s[p] = arg0;
                                pos++;
                                return;
                            }

                            // Otherwise, resize the stack to make room and try again.
                            WithResize(ref stack, ref pos, arg0);

                            // <summary>Resize the backtracking stack array and push 1 value onto the stack.</summary>
                            [MethodImpl(MethodImplOptions.NoInlining)]
                            static void WithResize(ref int[] stack, ref int pos, int arg0)
                            {
                                Array.Resize(ref stack, (pos + 0) * 2);
                                StackPush(ref stack, ref pos, arg0);
                            }
                        }

                        /// <summary>Pushes 2 values onto the backtracking stack.</summary>
                        [MethodImpl(MethodImplOptions.AggressiveInlining)]
                        internal static void StackPush(ref int[] stack, ref int pos, int arg0, int arg1)
                        {
                            // If there's space available for all 2 values, store them.
                            int[] s = stack;
                            int p = pos;
                            if ((uint)(p + 1) < (uint)s.Length)
                            {
                                s[p] = arg0;
                                s[p + 1] = arg1;
                                pos += 2;
                                return;
                            }

                            // Otherwise, resize the stack to make room and try again.
                            WithResize(ref stack, ref pos, arg0, arg1);

                            // <summary>Resize the backtracking stack array and push 2 values onto the stack.</summary>
                            [MethodImpl(MethodImplOptions.NoInlining)]
                            static void WithResize(ref int[] stack, ref int pos, int arg0, int arg1)
                            {
                                Array.Resize(ref stack, (pos + 1) * 2);
                                StackPush(ref stack, ref pos, arg0, arg1);
                            }
                        }

                        /// <summary>Validates that a stack cookie popped off the backtracking stack holds the expected value. Debug only.</summary>
                        internal static int ValidateStackCookie(int expected, int actual)
                        {
                            if (expected != actual)
                            {
                                throw new Exception($"Backtracking stack imbalance detected. Expected {expected}. Actual {actual}.");
                            }
                            return actual;
                        }
                    }
                }
                """
            };

            yield return new object[]
            {
                """
                using System.Text.RegularExpressions;
                partial class C
                {
                    [GeneratedRegex(@"href\s*=\s*(?:[""'](?<1>[^""']*)[""']|(?<1>[^>\s]+))", RegexOptions.None, -1)]
                    public static partial Regex Valid();
                }
                """,

                """
                // <auto-generated/>
                #nullable enable
                #pragma warning disable CS0162 // Unreachable code
                #pragma warning disable CS0164 // Unreferenced label
                #pragma warning disable CS0219 // Variable assigned but never used

                partial class C
                {
                    /// <remarks>
                    /// Pattern:<br/>
                    /// <code>href\\s*=\\s*(?:["'](?&lt;1&gt;[^"']*)["']|(?&lt;1&gt;[^&gt;\\s]+))</code><br/>
                    /// Explanation:<br/>
                    /// <code>
                    /// ○ Match the string "href".<br/>
                    /// ○ Match a whitespace character atomically any number of times.<br/>
                    /// ○ Match '='.<br/>
                    /// ○ Match a whitespace character greedily any number of times.<br/>
                    /// ○ Match with 2 alternative expressions, atomically.<br/>
                    ///     ○ Match a sequence of expressions.<br/>
                    ///         ○ Match a character in the set ["'].<br/>
                    ///         ○ 1st capture group.<br/>
                    ///             ○ Match a character in the set [^"'] atomically any number of times.<br/>
                    ///         ○ Match a character in the set ["'].<br/>
                    ///     ○ 1st capture group.<br/>
                    ///         ○ Match a character in the set [^&gt;\s] atomically at least once.<br/>
                    /// </code>
                    /// </remarks>
                    [global::System.CodeDom.Compiler.GeneratedCodeAttribute("System.Text.RegularExpressions.Generator", "%VERSION%")]
                    public static partial global::System.Text.RegularExpressions.Regex Valid() => global::System.Text.RegularExpressions.Generated.Valid_0.Instance;
                }

                namespace System.Text.RegularExpressions.Generated
                {
                    using System;
                    using System.Buffers;
                    using System.CodeDom.Compiler;
                    using System.Collections;
                    using System.ComponentModel;
                    using System.Globalization;
                    using System.Runtime.CompilerServices;
                    using System.Text.RegularExpressions;
                    using System.Threading;

                    /// <summary>Custom <see cref="Regex"/>-derived type for the Valid method.</summary>
                    [GeneratedCodeAttribute("System.Text.RegularExpressions.Generator", "%VERSION%")]
                    [SkipLocalsInit]
                    file sealed class Valid_0 : Regex
                    {
                        /// <summary>Cached, thread-safe singleton instance.</summary>
                        internal static readonly Valid_0 Instance = new();

                        /// <summary>Initializes the instance.</summary>
                        private Valid_0()
                        {
                            base.pattern = "href\\s*=\\s*(?:[\"'](?<1>[^\"']*)[\"']|(?<1>[^>\\s]+))";
                            base.roptions = RegexOptions.None;
                            base.internalMatchTimeout = Regex.InfiniteMatchTimeout;
                            base.factory = new RunnerFactory();
                            base.capsize = 2;
                        }

                        /// <summary>Provides a factory for creating <see cref="RegexRunner"/> instances to be used by methods on <see cref="Regex"/>.</summary>
                        private sealed class RunnerFactory : RegexRunnerFactory
                        {
                            /// <summary>Creates an instance of a <see cref="RegexRunner"/> used by methods on <see cref="Regex"/>.</summary>
                            protected override RegexRunner CreateInstance() => new Runner();

                            /// <summary>Provides the runner that contains the custom logic implementing the specified regular expression.</summary>
                            private sealed class Runner : RegexRunner
                            {
                                /// <summary>Scan the <paramref name="inputSpan"/> starting from base.runtextstart for the next match.</summary>
                                /// <param name="inputSpan">The text being scanned by the regular expression.</param>
                                protected override void Scan(ReadOnlySpan<char> inputSpan)
                                {
                                    // Search until we can't find a valid starting position, we find a match, or we reach the end of the input.
                                    while (TryFindNextPossibleStartingPosition(inputSpan) &&
                                           !TryMatchAtCurrentPosition(inputSpan) &&
                                           base.runtextpos != inputSpan.Length)
                                    {
                                        base.runtextpos++;
                                    }
                                }

                                /// <summary>Search <paramref name="inputSpan"/> starting from base.runtextpos for the next location a match could possibly start.</summary>
                                /// <param name="inputSpan">The text being scanned by the regular expression.</param>
                                /// <returns>true if a possible match was found; false if no more matches are possible.</returns>
                                private bool TryFindNextPossibleStartingPosition(ReadOnlySpan<char> inputSpan)
                                {
                                    int pos = base.runtextpos;

                                    // Any possible match is at least 6 characters.
                                    if (pos <= inputSpan.Length - 6)
                                    {
                                        // The pattern has the literal "href" at the beginning of the pattern. Find the next occurrence.
                                        // If it can't be found, there's no match.
                                        int i = inputSpan.Slice(pos).IndexOfAny(Utilities.s_indexOfString_href_Ordinal);
                                        if (i >= 0)
                                        {
                                            base.runtextpos = pos + i;
                                            return true;
                                        }
                                    }

                                    // No match found.
                                    base.runtextpos = inputSpan.Length;
                                    return false;
                                }

                                /// <summary>Determine whether <paramref name="inputSpan"/> at base.runtextpos is a match for the regular expression.</summary>
                                /// <param name="inputSpan">The text being scanned by the regular expression.</param>
                                /// <returns>true if the regular expression matches at the current position; otherwise, false.</returns>
                                private bool TryMatchAtCurrentPosition(ReadOnlySpan<char> inputSpan)
                                {
                                    int pos = base.runtextpos;
                                    int matchStart = pos;
                                    char ch;
                                    int capture_starting_pos = 0;
                                    int capture_starting_pos1 = 0;
                                    int charloop_capture_pos = 0;
                                    int charloop_starting_pos = 0, charloop_ending_pos = 0;
                                    ReadOnlySpan<char> slice = inputSpan.Slice(pos);

                                    // Match the string "href".
                                    if (!slice.StartsWith("href"))
                                    {
                                        UncaptureUntil(0);
                                        return false; // The input didn't match.
                                    }

                                    // Match a whitespace character atomically any number of times.
                                    {
                                        int iteration = 4;
                                        while ((uint)iteration < (uint)slice.Length && char.IsWhiteSpace(slice[iteration]))
                                        {
                                            iteration++;
                                        }

                                        slice = slice.Slice(iteration);
                                        pos += iteration;
                                    }

                                    // Match '='.
                                    if (slice.IsEmpty || slice[0] != '=')
                                    {
                                        UncaptureUntil(0);
                                        return false; // The input didn't match.
                                    }

                                    // Match a whitespace character greedily any number of times.
                                    //{
                                        pos++;
                                        slice = inputSpan.Slice(pos);
                                        charloop_starting_pos = pos;

                                        int iteration1 = 0;
                                        while ((uint)iteration1 < (uint)slice.Length && char.IsWhiteSpace(slice[iteration1]))
                                        {
                                            iteration1++;
                                        }

                                        slice = slice.Slice(iteration1);
                                        pos += iteration1;

                                        charloop_ending_pos = pos;
                                        goto CharLoopEnd;

                                        CharLoopBacktrack:
                                        UncaptureUntil(charloop_capture_pos);

                                        if (charloop_starting_pos >= charloop_ending_pos)
                                        {
                                            UncaptureUntil(0);
                                            return false; // The input didn't match.
                                        }
                                        pos = --charloop_ending_pos;
                                        slice = inputSpan.Slice(pos);

                                        CharLoopEnd:
                                        charloop_capture_pos = base.Crawlpos();
                                    //}

                                    // Match with 2 alternative expressions, atomically.
                                    {
                                        int alternation_starting_pos = pos;
                                        int alternation_starting_capturepos = base.Crawlpos();

                                        // Branch 0
                                        {
                                            // Match a character in the set ["'].
                                            if (slice.IsEmpty || (((ch = slice[0]) != '"') & (ch != '\'')))
                                            {
                                                goto AlternationBranch;
                                            }

                                            // 1st capture group.
                                            {
                                                pos++;
                                                slice = inputSpan.Slice(pos);
                                                capture_starting_pos = pos;

                                                // Match a character in the set [^"'] atomically any number of times.
                                                {
                                                    int iteration2 = slice.IndexOfAny('"', '\'');
                                                    if (iteration2 < 0)
                                                    {
                                                        iteration2 = slice.Length;
                                                    }

                                                    slice = slice.Slice(iteration2);
                                                    pos += iteration2;
                                                }

                                                base.Capture(1, capture_starting_pos, pos);
                                            }

                                            // Match a character in the set ["'].
                                            if (slice.IsEmpty || (((ch = slice[0]) != '"') & (ch != '\'')))
                                            {
                                                goto AlternationBranch;
                                            }

                                            pos++;
                                            slice = inputSpan.Slice(pos);
                                            goto AlternationMatch;

                                            AlternationBranch:
                                            pos = alternation_starting_pos;
                                            slice = inputSpan.Slice(pos);
                                            UncaptureUntil(alternation_starting_capturepos);
                                        }

                                        // Branch 1
                                        {
                                            // 1st capture group.
                                            {
                                                capture_starting_pos1 = pos;

                                                // Match a character in the set [^>\s] atomically at least once.
                                                {
                                                    int iteration3 = 0;
                                                    while ((uint)iteration3 < (uint)slice.Length && ((ch = slice[iteration3]) < 128 ? ("쇿\uffff\ufffe뿿\uffff\uffff\uffff\uffff"[ch >> 4] & (1 << (ch & 0xF))) != 0 : RegexRunner.CharInClass((char)ch, "\u0001\u0002\u0001>?d")))
                                                    {
                                                        iteration3++;
                                                    }

                                                    if (iteration3 == 0)
                                                    {
                                                        goto CharLoopBacktrack;
                                                    }

                                                    slice = slice.Slice(iteration3);
                                                    pos += iteration3;
                                                }

                                                base.Capture(1, capture_starting_pos1, pos);
                                            }

                                        }

                                        AlternationMatch:;
                                    }

                                    // The input matched.
                                    base.runtextpos = pos;
                                    base.Capture(0, matchStart, pos);
                                    return true;

                                    // <summary>Undo captures until it reaches the specified capture position.</summary>
                                    [MethodImpl(MethodImplOptions.AggressiveInlining)]
                                    void UncaptureUntil(int capturePosition)
                                    {
                                        while (base.Crawlpos() > capturePosition)
                                        {
                                            base.Uncapture();
                                        }
                                    }
                                }
                            }
                        }

                    }

                    /// <summary>Helper methods used by generated <see cref="Regex"/>-derived implementations.</summary>
                    [GeneratedCodeAttribute("System.Text.RegularExpressions.Generator", "42.42.42.42")]
                    file static class Utilities
                    {
                        /// <summary>Supports searching for the string "href".</summary>
                        internal static readonly SearchValues<string> s_indexOfString_href_Ordinal = SearchValues.Create(["href"], StringComparison.Ordinal);
                    }
                }
                """
            };

            yield return new object[]
            {
                """
                using System.Text.RegularExpressions;
                partial class C
                {
                    [GeneratedRegex(@"[A-Za-z]+")]
                    public static partial Regex Valid();
                }
                """,

                """
                // <auto-generated/>
                #nullable enable
                #pragma warning disable CS0162 // Unreachable code
                #pragma warning disable CS0164 // Unreferenced label
                #pragma warning disable CS0219 // Variable assigned but never used

                partial class C
                {
                    /// <remarks>
                    /// Pattern:<br/>
                    /// <code>[A-Za-z]+</code><br/>
                    /// Explanation:<br/>
                    /// <code>
                    /// ○ Match a character in the set [A-Za-z] atomically at least once.<br/>
                    /// </code>
                    /// </remarks>
                    [global::System.CodeDom.Compiler.GeneratedCodeAttribute("System.Text.RegularExpressions.Generator", "%VERSION%")]
                    public static partial global::System.Text.RegularExpressions.Regex Valid() => global::System.Text.RegularExpressions.Generated.Valid_0.Instance;
                }

                namespace System.Text.RegularExpressions.Generated
                {
                    using System;
                    using System.Buffers;
                    using System.CodeDom.Compiler;
                    using System.Collections;
                    using System.ComponentModel;
                    using System.Globalization;
                    using System.Runtime.CompilerServices;
                    using System.Text.RegularExpressions;
                    using System.Threading;

                    /// <summary>Custom <see cref="Regex"/>-derived type for the Valid method.</summary>
                    [GeneratedCodeAttribute("System.Text.RegularExpressions.Generator", "%VERSION%")]
                    [SkipLocalsInit]
                    file sealed class Valid_0 : Regex
                    {
                        /// <summary>Cached, thread-safe singleton instance.</summary>
                        internal static readonly Valid_0 Instance = new();

                        /// <summary>Initializes the instance.</summary>
                        private Valid_0()
                        {
                            base.pattern = "[A-Za-z]+";
                            base.roptions = RegexOptions.None;
                            ValidateMatchTimeout(Utilities.s_defaultTimeout);
                            base.internalMatchTimeout = Utilities.s_defaultTimeout;
                            base.factory = new RunnerFactory();
                            base.capsize = 1;
                        }

                        /// <summary>Provides a factory for creating <see cref="RegexRunner"/> instances to be used by methods on <see cref="Regex"/>.</summary>
                        private sealed class RunnerFactory : RegexRunnerFactory
                        {
                            /// <summary>Creates an instance of a <see cref="RegexRunner"/> used by methods on <see cref="Regex"/>.</summary>
                            protected override RegexRunner CreateInstance() => new Runner();

                            /// <summary>Provides the runner that contains the custom logic implementing the specified regular expression.</summary>
                            private sealed class Runner : RegexRunner
                            {
                                /// <summary>Scan the <paramref name="inputSpan"/> starting from base.runtextstart for the next match.</summary>
                                /// <param name="inputSpan">The text being scanned by the regular expression.</param>
                                protected override void Scan(ReadOnlySpan<char> inputSpan)
                                {
                                    // Search until we can't find a valid starting position, we find a match, or we reach the end of the input.
                                    while (TryFindNextPossibleStartingPosition(inputSpan) &&
                                           !TryMatchAtCurrentPosition(inputSpan) &&
                                           base.runtextpos != inputSpan.Length)
                                    {
                                        base.runtextpos++;
                                        if (Utilities.s_hasTimeout)
                                        {
                                            base.CheckTimeout();
                                        }
                                    }
                                }

                                /// <summary>Search <paramref name="inputSpan"/> starting from base.runtextpos for the next location a match could possibly start.</summary>
                                /// <param name="inputSpan">The text being scanned by the regular expression.</param>
                                /// <returns>true if a possible match was found; false if no more matches are possible.</returns>
                                private bool TryFindNextPossibleStartingPosition(ReadOnlySpan<char> inputSpan)
                                {
                                    int pos = base.runtextpos;

                                    // Empty matches aren't possible.
                                    if ((uint)pos < (uint)inputSpan.Length)
                                    {
                                        // The pattern begins with a character in the set [A-Za-z].
                                        // Find the next occurrence. If it can't be found, there's no match.
                                        int i = inputSpan.Slice(pos).IndexOfAny(Utilities.s_asciiLetters);
                                        if (i >= 0)
                                        {
                                            base.runtextpos = pos + i;
                                            return true;
                                        }
                                    }

                                    // No match found.
                                    base.runtextpos = inputSpan.Length;
                                    return false;
                                }

                                /// <summary>Determine whether <paramref name="inputSpan"/> at base.runtextpos is a match for the regular expression.</summary>
                                /// <param name="inputSpan">The text being scanned by the regular expression.</param>
                                /// <returns>true if the regular expression matches at the current position; otherwise, false.</returns>
                                private bool TryMatchAtCurrentPosition(ReadOnlySpan<char> inputSpan)
                                {
                                    int pos = base.runtextpos;
                                    int matchStart = pos;
                                    ReadOnlySpan<char> slice = inputSpan.Slice(pos);

                                    // Match a character in the set [A-Za-z] atomically at least once.
                                    {
                                        int iteration = slice.IndexOfAnyExcept(Utilities.s_asciiLetters);
                                        if (iteration < 0)
                                        {
                                            iteration = slice.Length;
                                        }

                                        if (iteration == 0)
                                        {
                                            return false; // The input didn't match.
                                        }

                                        slice = slice.Slice(iteration);
                                        pos += iteration;
                                    }

                                    // The input matched.
                                    base.runtextpos = pos;
                                    base.Capture(0, matchStart, pos);
                                    return true;
                                }
                            }
                        }

                    }

                    /// <summary>Helper methods used by generated <see cref="Regex"/>-derived implementations.</summary>
                    [GeneratedCodeAttribute("System.Text.RegularExpressions.Generator", "%VERSION%")]
                    file static class Utilities
                    {
                        /// <summary>Default timeout value set in <see cref="AppContext"/>, or <see cref="Regex.InfiniteMatchTimeout"/> if none was set.</summary>
                        internal static readonly TimeSpan s_defaultTimeout = AppContext.GetData("REGEX_DEFAULT_MATCH_TIMEOUT") is TimeSpan timeout ? timeout : Regex.InfiniteMatchTimeout;

                        /// <summary>Whether <see cref="s_defaultTimeout"/> is non-infinite.</summary>
                        internal static readonly bool s_hasTimeout = s_defaultTimeout != Regex.InfiniteMatchTimeout;

                        /// <summary>Supports searching for characters in or not in "ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz".</summary>
                        internal static readonly SearchValues<char> s_asciiLetters = SearchValues.Create("ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz");
                    }
                }
                """
            };
        }
    }
}
