/*
 * Copyright (C) 2021 Andreas Steffen, strongSec GmbH
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the
 * Free Software Foundation; either version 2 of the License, or (at your
 * option) any later version.  See <http://www.fsf.org/copyleft/gpl.txt>.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 * or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * for more details.
 */

#include "test_suite.h"

#include <plugins/plugin_feature.h>

typedef struct rsa_oaep_test_t rsa_oaep_test_t;

struct rsa_oaep_test_t {
	chunk_t msg;
	chunk_t label;
	chunk_t ct;
};

/**
 * Private RSA keys to load
 */
static chunk_t rsa_2048_key = chunk_from_chars( /* PKCS#8 format */
		0x30,0x82,0x04,0xbd,0x02,0x01,0x00,0x30,0x0d,0x06,0x09,0x2a,0x86,0x48,0x86,0xf7,
		0x0d,0x01,0x01,0x01,0x05,0x00,0x04,0x82,0x04,0xa7,0x30,0x82,0x04,0xa3,0x02,0x01,
		0x00,0x02,0x82,0x01,0x01,0x00,0xa2,0xb4,0x51,0xa0,0x7d,0x0a,0xa5,0xf9,0x6e,0x45,
		0x56,0x71,0x51,0x35,0x50,0x51,0x4a,0x8a,0x5b,0x46,0x2e,0xbe,0xf7,0x17,0x09,0x4f,
		0xa1,0xfe,0xe8,0x22,0x24,0xe6,0x37,0xf9,0x74,0x6d,0x3f,0x7c,0xaf,0xd3,0x18,0x78,
		0xd8,0x03,0x25,0xb6,0xef,0x5a,0x17,0x00,0xf6,0x59,0x03,0xb4,0x69,0x42,0x9e,0x89,
		0xd6,0xea,0xc8,0x84,0x50,0x97,0xb5,0xab,0x39,0x31,0x89,0xdb,0x92,0x51,0x2e,0xd8,
		0xa7,0x71,0x1a,0x12,0x53,0xfa,0xcd,0x20,0xf7,0x9c,0x15,0xe8,0x24,0x7f,0x3d,0x3e,
		0x42,0xe4,0x6e,0x48,0xc9,0x8e,0x25,0x4a,0x2f,0xe9,0x76,0x53,0x13,0xa0,0x3e,0xff,
		0x8f,0x17,0xe1,0xa0,0x29,0x39,0x7a,0x1f,0xa2,0x6a,0x8d,0xce,0x26,0xf4,0x90,0xed,
		0x81,0x29,0x96,0x15,0xd9,0x81,0x4c,0x22,0xda,0x61,0x04,0x28,0xe0,0x9c,0x7d,0x96,
		0x58,0x59,0x42,0x66,0xf5,0xc0,0x21,0xd0,0xfc,0xec,0xa0,0x8d,0x94,0x5a,0x12,0xbe,
		0x82,0xde,0x4d,0x1e,0xce,0x6b,0x4c,0x03,0x14,0x5b,0x5d,0x34,0x95,0xd4,0xed,0x54,
		0x11,0xeb,0x87,0x8d,0xaf,0x05,0xfd,0x7a,0xfc,0x3e,0x09,0xad,0xa0,0xf1,0x12,0x64,
		0x22,0xf5,0x90,0x97,0x5a,0x19,0x69,0x81,0x6f,0x48,0x69,0x8b,0xcb,0xba,0x1b,0x4d,
		0x9c,0xae,0x79,0xd4,0x60,0xd8,0xf9,0xf8,0x5e,0x79,0x75,0x00,0x5d,0x9b,0xc2,0x2c,
		0x4e,0x5a,0xc0,0xf7,0xc1,0xa4,0x5d,0x12,0x56,0x9a,0x62,0x80,0x7d,0x3b,0x9a,0x02,
		0xe5,0xa5,0x30,0xe7,0x73,0x06,0x6f,0x45,0x3d,0x1f,0x5b,0x4c,0x2e,0x9c,0xf7,0x82,
		0x02,0x83,0xf7,0x42,0xb9,0xd5,0x02,0x03,0x01,0x00,0x01,0x02,0x82,0x01,0x00,0x76,
		0x27,0xee,0xf3,0x56,0x7b,0x2a,0x27,0x26,0x8e,0x52,0x05,0x3e,0xcd,0x31,0xc3,0xa7,
		0x17,0x2c,0xcb,0x9d,0xdc,0xee,0x81,0x9b,0x30,0x6a,0x5b,0x3c,0x66,0xb7,0x57,0x3c,
		0xa4,0xfa,0x88,0xef,0xc6,0xf3,0xc4,0xa0,0x0b,0xfa,0x0a,0xe7,0x13,0x9f,0x64,0x54,
		0x3a,0x4d,0xac,0x3d,0x05,0x82,0x3f,0x6f,0xf4,0x77,0xcf,0xce,0xc8,0x4f,0xe2,0xac,
		0x7a,0x68,0xb1,0x72,0x04,0xb3,0x90,0x23,0x2e,0x11,0x03,0x10,0xc4,0xe8,0x99,0xc4,
		0xe7,0xc1,0x09,0x67,0xdb,0x4a,0xcd,0xe0,0x42,0xdb,0xbf,0x19,0xdb,0xe0,0x0b,0x4b,
		0x47,0x41,0xde,0x10,0x20,0xaa,0xaa,0xff,0xb5,0x05,0x4c,0x79,0x7c,0x9f,0x13,0x6f,
		0x7d,0x93,0xac,0x3f,0xc8,0xca,0xff,0x66,0x54,0x24,0x2d,0x78,0x21,0xeb,0xee,0x51,
		0x7b,0xf5,0x37,0xf4,0x43,0x66,0xa0,0xfd,0xd4,0x5a,0xe0,0x5b,0x99,0x09,0xc2,0xe6,
		0xcc,0x1e,0xd9,0x28,0x1e,0xff,0x43,0x99,0xf7,0x6c,0x96,0xb9,0x62,0x33,0xec,0x29,
		0xae,0x0b,0xbf,0x0d,0x75,0x2b,0x23,0x4f,0xc1,0x97,0x38,0x9f,0x51,0x05,0x0a,0xa1,
		0xac,0xd0,0x1c,0x07,0x4c,0x3a,0xc8,0xfb,0xdb,0x9e,0xa8,0xb6,0x51,0xa9,0x59,0x95,
		0xe8,0xdb,0x4a,0xd5,0xc4,0x3b,0x6c,0x86,0x73,0xe5,0xa1,0x26,0xe7,0xee,0x94,0xb8,
		0xdf,0xf4,0xc5,0xaf,0xc0,0x12,0x59,0xbc,0x8d,0xa7,0x69,0x50,0xba,0xe6,0xf8,0xba,
		0xe7,0x15,0xf5,0x09,0x85,0xb0,0xd6,0xf6,0x6d,0x04,0xc6,0xfe,0xf3,0xb7,0x00,0x72,
		0x0e,0xec,0xdc,0xdf,0x17,0x1b,0xb7,0xb1,0xec,0xbe,0x72,0x89,0xc4,0x67,0xc1,0x02,
		0x81,0x81,0x00,0xdc,0x43,0x10,0x50,0xf7,0x82,0xe8,0x94,0xfb,0x52,0x48,0x24,0x7d,
		0x98,0xcb,0x7d,0x58,0xb8,0xd1,0xe2,0x4f,0x3b,0x55,0xd0,0x41,0xc5,0x6e,0x4d,0xe0,
		0x86,0xb0,0xd5,0xbb,0x02,0x8b,0xda,0x42,0xee,0xb5,0xd2,0x34,0xd5,0x68,0x1e,0x58,
		0x09,0xd4,0x15,0xe6,0xa2,0x89,0xad,0x4c,0xfb,0xf7,0x8f,0x97,0x8f,0x6c,0x35,0x81,
		0x4f,0x50,0xee,0xbf,0xf1,0xc5,0xb8,0x0a,0x69,0xf7,0x88,0xe8,0x1e,0x6b,0xab,0x5d,
		0xda,0xa7,0x83,0x69,0xd6,0x59,0xd1,0x43,0xec,0x6f,0x17,0xe7,0x98,0x13,0xa5,0x75,
		0xcf,0xad,0x9c,0x56,0x91,0x56,0xb9,0x01,0x13,0xe2,0xe9,0x11,0x0a,0xd9,0xe7,0xb4,
		0x8a,0x1c,0x93,0x48,0xa6,0xe6,0x53,0x32,0x11,0x91,0x29,0x0e,0xa3,0x6c,0xfb,0x3a,
		0x5b,0x18,0xf1,0x02,0x81,0x81,0x00,0xbd,0x1a,0x81,0xe7,0x97,0x7f,0x98,0x98,0x12,
		0x22,0x73,0xae,0x32,0x22,0xb5,0x98,0xea,0x5f,0xb1,0x9e,0xb4,0xea,0xbc,0x38,0x30,
		0x8a,0x5e,0x32,0x19,0x66,0x03,0xb2,0xe5,0x00,0xff,0xb7,0x9f,0x5b,0x88,0x68,0x16,
		0x61,0x1d,0xeb,0xc4,0x72,0xfa,0xc4,0x55,0x44,0x07,0x0b,0xeb,0x05,0x7c,0x94,0x13,
		0x78,0xa6,0x86,0x8a,0xf3,0xb7,0xa0,0x3d,0x3f,0x98,0x80,0xec,0x47,0xd5,0xe0,0x89,
		0xb9,0x4f,0xbd,0xe5,0x42,0xab,0xa9,0xae,0x8d,0x72,0xc5,0x70,0x88,0xd7,0xab,0xf5,
		0xb1,0x31,0xf3,0x90,0x98,0xf7,0xbc,0x16,0x0f,0x90,0x53,0x6a,0xbc,0x94,0x92,0xfd,
		0x4e,0x06,0xf3,0xed,0x72,0x99,0xd4,0xb9,0x7b,0xb0,0x36,0x77,0x20,0x7d,0x95,0x66,
		0x9f,0x14,0x0c,0xfb,0xc2,0x0f,0x25,0x02,0x81,0x81,0x00,0xa9,0x4b,0x52,0x8b,0x28,
		0xf2,0x91,0x59,0x91,0x21,0xd9,0x19,0x52,0xff,0xd1,0xc7,0xf2,0x1d,0x7c,0x14,0x79,
		0xd9,0x9d,0x47,0x88,0x85,0xfb,0x16,0x18,0x70,0xee,0x12,0x18,0xbf,0x08,0x47,0x26,
		0x12,0xdb,0xe5,0x49,0x7e,0x8d,0x9c,0x65,0x06,0x88,0xe0,0x9c,0x78,0x69,0x61,0xae,
		0x3e,0x2c,0x35,0x4d,0xc4,0x8a,0xe3,0x45,0x14,0x75,0x9c,0x4c,0x23,0xc4,0x58,0x84,
		0x88,0x96,0x1d,0xc0,0x6b,0x41,0x4e,0x61,0xc0,0xe1,0xe7,0xfb,0xbd,0x29,0x23,0xd3,
		0x15,0x32,0xfe,0x28,0x9f,0x96,0xda,0x22,0x07,0x11,0xe5,0x8c,0x14,0x01,0x98,0x08,
		0xe0,0x04,0x14,0x27,0x69,0x33,0xbb,0x07,0xe4,0xef,0xb9,0xb4,0xa9,0xb3,0x76,0x56,
		0x91,0x72,0x05,0x20,0x9f,0x33,0xf0,0x95,0x15,0xd7,0xc1,0x02,0x81,0x80,0x3a,0xf0,
		0xe7,0x2a,0x93,0x3a,0xef,0x09,0xff,0x25,0x03,0xdf,0x78,0xba,0xfe,0xd5,0x31,0xc0,
		0x2f,0xf1,0xa2,0xbc,0x43,0x7c,0x54,0x0c,0xdc,0xbd,0x4a,0xd3,0x54,0x35,0xcf,0x51,
		0x17,0x63,0x59,0x65,0x43,0x48,0x06,0x29,0xb1,0x14,0xca,0x7f,0x78,0x0f,0xf7,0xef,
		0xa3,0x2e,0xa0,0xcb,0x6e,0x00,0x0d,0x6d,0x9e,0xa1,0xf2,0xef,0x71,0xfd,0x9c,0xf9,
		0x94,0x84,0x22,0xa1,0x65,0x55,0x7e,0x37,0xe7,0x55,0xed,0xfe,0x70,0xd9,0x0b,0x92,
		0x05,0x02,0xeb,0x47,0x8b,0xc9,0x8a,0x63,0xf7,0x88,0xce,0x3a,0x0f,0x85,0x6d,0x6e,
		0xde,0x72,0x51,0xa3,0x83,0xbf,0xa8,0xfa,0x48,0x0a,0x81,0xa9,0x25,0xaf,0x7b,0x3c,
		0xc5,0x38,0xc4,0xba,0xb8,0xc9,0xf7,0x59,0x7f,0xfb,0x68,0x01,0x1d,0x8d,0x02,0x81,
		0x80,0x26,0x40,0xfb,0xfb,0xcf,0xef,0xb1,0x63,0xee,0x7a,0x87,0xb6,0x48,0x3a,0x66,
		0xee,0x41,0xf9,0x56,0xd9,0x0f,0xa8,0xa7,0x93,0x9b,0xfc,0x04,0x2e,0xe0,0x92,0x4b,
		0x1b,0x79,0x93,0xd0,0x44,0x5f,0x75,0x8d,0x51,0x93,0x3e,0x85,0x17,0x9c,0x03,0x20,
		0xb0,0xc9,0x68,0xb4,0x8a,0x91,0xc3,0x8b,0x5b,0xe9,0x23,0xe1,0x09,0x7c,0x0c,0x56,
		0x2f,0x88,0xd4,0x22,0x94,0xb6,0xa2,0x75,0x9b,0xaf,0xa5,0x42,0x8a,0x74,0xf1,0x27,
		0x08,0x74,0xe4,0x5f,0x6f,0xcc,0x60,0xf2,0x16,0x02,0xde,0x5e,0xcc,0xd1,0x43,0xcf,
		0x31,0x24,0x1f,0x59,0x21,0xb5,0xad,0x39,0x83,0xfb,0x54,0xef,0x17,0xbe,0x3b,0x28,
		0x53,0x67,0xe5,0x0c,0x99,0x9c,0x67,0x24,0x7b,0x55,0x2f,0xe4,0xbf,0xce,0x94,0x5f,
		0x7b);

static chunk_t rsa_3072_key = chunk_from_chars( /* PKCS#8 format */
		0x30,0x82,0x06,0xfb,0x02,0x01,0x00,0x30,0x0d,0x06,0x09,0x2a,0x86,0x48,0x86,0xf7,
		0x0d,0x01,0x01,0x01,0x05,0x00,0x04,0x82,0x06,0xe5,0x30,0x82,0x06,0xe1,0x02,0x01,
		0x00,0x02,0x82,0x01,0x81,0x00,0xc6,0xfe,0x23,0x79,0x25,0x66,0x02,0x3c,0x26,0x52,
		0x87,0xc5,0xac,0x6f,0x71,0x54,0x1c,0x09,0x94,0xd1,0x1d,0x05,0x9e,0xe6,0x40,0x39,
		0x86,0xef,0xa2,0x1c,0x24,0xb5,0x1b,0xd9,0x1d,0x88,0x62,0xf9,0xdf,0x79,0xa4,0xe3,
		0x28,0xe3,0xe2,0x7c,0x83,0xdf,0x26,0x0b,0x25,0xa9,0xb4,0x34,0x20,0xaf,0xfc,0x44,
		0xb5,0x1e,0x8d,0x75,0x25,0xb6,0xf2,0x9c,0x37,0x2a,0x40,0x51,0x04,0x73,0x20,0x07,
		0x52,0x7a,0x62,0xed,0x82,0xfa,0xc7,0x3f,0x48,0x92,0xa8,0x0e,0x09,0x68,0x2a,0x41,
		0xa5,0x8c,0xd3,0x47,0x01,0x7f,0x3b,0xe7,0xd8,0x01,0x33,0x4f,0x92,0xd9,0x32,0x1a,
		0xaf,0xd5,0x3b,0x51,0xbf,0xfa,0xbf,0xc7,0x52,0xcf,0xcc,0xae,0x0b,0x1e,0xe0,0x3b,
		0xda,0xff,0x9e,0x42,0x8c,0xc1,0xc1,0x17,0xf1,0xac,0x96,0xb4,0xfe,0x23,0xf8,0xc2,
		0x3e,0x63,0x81,0x18,0x6a,0x66,0xfd,0x59,0x28,0x93,0x39,0xae,0x55,0xc4,0xbc,0xda,
		0xdb,0xff,0x84,0xab,0xda,0xa5,0x32,0x24,0x0d,0x4e,0x1d,0x28,0xb2,0xd0,0x48,0x1d,
		0xad,0xd3,0xb2,0x46,0x55,0x7c,0xa8,0xfe,0x18,0x09,0x28,0x17,0x73,0x0b,0x39,0xe6,
		0xee,0x37,0x8f,0xfc,0xc8,0x5b,0x19,0xff,0xdc,0x91,0x6a,0x9b,0x99,0x1a,0x6b,0x66,
		0xd4,0xa9,0xc7,0xba,0xb5,0xf5,0xe7,0xa3,0x72,0x21,0x01,0x14,0x2e,0x7a,0x41,0x08,
		0xc1,0x5d,0x57,0x3b,0x15,0x28,0x9e,0x07,0xe4,0x6e,0xae,0xa0,0x7b,0x42,0xc2,0xab,
		0xcb,0xa3,0x30,0xe9,0x95,0x54,0xb4,0x65,0x61,0x65,0xbb,0x4c,0x0d,0xb2,0xb6,0x39,
		0x3a,0x07,0xec,0xa5,0x75,0xc5,0x1a,0x93,0xc4,0xe1,0x5b,0xdb,0x0f,0x74,0x79,0x09,
		0x44,0x7e,0x3e,0xfe,0x34,0xc6,0x7c,0xa8,0x95,0x4b,0x53,0x0e,0x56,0xa2,0x0a,0x1b,
		0x6d,0x84,0xd4,0x5e,0xd1,0xbc,0xd3,0xaa,0x58,0xec,0x06,0xf1,0x84,0xee,0x58,0x57,
		0xaa,0xa8,0x19,0xe1,0xcc,0xa9,0xa2,0x6f,0x4e,0x28,0xd6,0xb9,0x77,0xd3,0x39,0x16,
		0xdb,0x98,0x96,0xd2,0x52,0xd1,0xaf,0xa7,0x62,0xe2,0x87,0xcb,0x0d,0x38,0x4c,0xc7,
		0x5b,0xfe,0x53,0xf4,0xe9,0x22,0xd0,0x2d,0xd0,0xa4,0x81,0xc0,0x42,0xe2,0xd3,0x06,
		0xb4,0xb3,0xc1,0x89,0x37,0x1e,0x57,0x5b,0x25,0xe0,0x00,0x5a,0x16,0x4c,0xf6,0x9d,
		0xd0,0x97,0x6e,0x4d,0x5b,0xe4,0x76,0x80,0x6e,0xa6,0xbe,0x60,0x84,0xe7,0x1a,0xb4,
		0xf5,0xac,0x5c,0x1b,0x12,0x03,0x02,0x03,0x01,0x00,0x01,0x02,0x82,0x01,0x80,0x72,
		0xac,0x6b,0xb6,0xd9,0xa5,0x72,0x6e,0x45,0x4b,0x54,0x30,0xc7,0x11,0x25,0xc6,0xe9,
		0xad,0x5f,0xd4,0x2e,0x1c,0x5a,0x18,0xa8,0x34,0x3e,0x9d,0x83,0xd7,0x22,0x14,0x38,
		0x6b,0x23,0x08,0xc0,0xb8,0xec,0x5e,0xc6,0x75,0x9d,0xcf,0xcd,0x6a,0x21,0xf8,0x8b,
		0x8c,0xea,0xf4,0x64,0x03,0x92,0x3e,0xb8,0x6a,0xc3,0xd1,0x4a,0x85,0x92,0xe9,0x5d,
		0xe0,0x46,0x2e,0x14,0x08,0x5c,0x3f,0x17,0xdb,0x00,0x5d,0xc4,0xfa,0xc8,0x7b,0x4a,
		0x2d,0x1e,0xde,0x5c,0xf8,0x51,0xd5,0x74,0x5c,0x86,0x51,0xa4,0x43,0x8c,0x0a,0x4d,
		0x74,0x6a,0xd7,0x2e,0x41,0x92,0x07,0x96,0x47,0x28,0xc3,0x01,0xbf,0x37,0x9a,0x01,
		0xc0,0x94,0xe9,0x69,0x33,0x76,0xf7,0x21,0x13,0x7d,0x3d,0xc7,0x6e,0xe4,0x7c,0x97,
		0x90,0xfb,0xd5,0x90,0xb7,0xd6,0xa8,0xd6,0x26,0xe2,0x1b,0x27,0x7e,0xf1,0x7a,0x4e,
		0x4f,0x7e,0x01,0x71,0xc1,0x14,0x6e,0x1e,0xc3,0x24,0xfa,0x97,0xf3,0x0d,0x3a,0x1b,
		0xae,0x08,0xf8,0xd5,0xf6,0xe9,0x2c,0xfc,0x12,0x16,0x65,0x23,0x9c,0x42,0x91,0x67,
		0x35,0x9e,0x96,0x50,0x43,0x4b,0x29,0xd2,0x01,0x51,0x90,0x35,0x6a,0xdf,0xee,0x12,
		0xf2,0x5b,0x34,0x1b,0x08,0xf1,0x2b,0x7f,0xec,0x63,0x79,0x59,0x8a,0xf7,0xd5,0xcc,
		0x24,0xfe,0x7f,0x00,0xde,0x1d,0x47,0x13,0x3c,0xe3,0xad,0x8b,0x6b,0xe1,0xc9,0xa8,
		0x54,0xe3,0x3f,0xb9,0x52,0xe1,0x64,0xac,0x6d,0xd2,0xa9,0x05,0x21,0x86,0xee,0x14,
		0x4e,0xe7,0xdd,0x98,0x6a,0x8f,0x03,0x89,0x1d,0x0d,0xa2,0x1e,0xd7,0x85,0x16,0xdc,
		0xdc,0x2a,0xc8,0x9c,0xdd,0xdc,0x8b,0x54,0x47,0x31,0xd6,0x6f,0x9d,0x89,0xbf,0x17,
		0xa5,0x0c,0x6d,0x98,0x7a,0x59,0x8b,0x02,0xc9,0x38,0xdc,0x36,0x52,0x1b,0x88,0x1e,
		0xa9,0x94,0xe4,0xc8,0xfb,0x2b,0xa8,0xfd,0x00,0x1f,0x73,0x33,0x5d,0x4d,0xd1,0xbd,
		0xbe,0x17,0x7d,0x30,0x93,0xcf,0x38,0x83,0x65,0x7c,0x9f,0xf9,0x44,0xe8,0xf5,0xc9,
		0xcd,0xe5,0x48,0xb7,0xc1,0xb0,0x74,0x19,0x29,0xb0,0xd7,0x49,0x77,0xec,0xda,0x69,
		0x4d,0x94,0x0a,0xef,0xd9,0xd2,0xfc,0x75,0x32,0x3e,0x0b,0x3a,0x11,0x4b,0x99,0xfe,
		0xaf,0x3e,0x25,0x18,0xf5,0x15,0x8d,0x1f,0xd9,0xd9,0x53,0xaa,0x20,0xaf,0x15,0x8e,
		0x67,0xd2,0x7e,0x2c,0xe2,0xf1,0x8d,0x97,0xfd,0x02,0xf3,0x69,0x98,0x19,0x79,0x02,
		0x81,0xc1,0x00,0xf5,0xec,0xa1,0x6e,0x0e,0x83,0x69,0x6b,0x0e,0xd9,0xac,0x8a,0x81,
		0x25,0x45,0xda,0xba,0x55,0xf2,0x0a,0x96,0x4c,0x4e,0x63,0x43,0x60,0x4a,0x7f,0x2b,
		0xe2,0x86,0x0f,0xce,0x9f,0xa1,0x6a,0x1c,0xc9,0x21,0x20,0x93,0x9d,0xeb,0x88,0xdf,
		0xf6,0x85,0x50,0x38,0x3e,0xad,0x85,0x1f,0xac,0x07,0xad,0x1b,0x2e,0x8a,0x9b,0x2b,
		0xb6,0x95,0x25,0xd9,0x6c,0xea,0xbb,0x7e,0xe8,0x3c,0xe5,0x0f,0x08,0xd6,0x49,0x10,
		0x7f,0x44,0x9a,0x14,0x52,0x1a,0x68,0x93,0xf3,0xf3,0xc5,0xc5,0xa7,0x03,0xb2,0xfc,
		0x28,0xbf,0xcf,0xe2,0x61,0xa4,0xf7,0xf4,0x50,0x55,0x80,0x80,0xde,0xae,0xaa,0xb6,
		0x51,0xc7,0xa9,0xae,0x58,0x6c,0x1e,0x7f,0x5c,0x52,0xcd,0xa9,0x3e,0x40,0xaa,0xc9,
		0x08,0xe4,0xe3,0x35,0x79,0x84,0xfc,0x11,0x6a,0xf9,0xcb,0xe9,0x53,0x9b,0xc7,0xa8,
		0xd3,0xb3,0x51,0xa7,0x3e,0xa5,0xc2,0x41,0x3d,0x1d,0xa2,0xe0,0xb4,0x48,0xb4,0x54,
		0x67,0x0a,0xca,0x89,0xff,0xe7,0x3b,0x14,0x01,0xe9,0xb8,0x55,0x4f,0xc3,0xf2,0x3d,
		0x6c,0x90,0x46,0x23,0x25,0x1a,0x1d,0x29,0x96,0x2c,0xa9,0xb2,0x6d,0x97,0x33,0x45,
		0xbc,0x4c,0x5f,0x02,0x81,0xc1,0x00,0xcf,0x25,0x44,0x6f,0x59,0xcf,0x51,0x29,0x19,
		0xdd,0xbf,0xcf,0xa2,0xd9,0x67,0x04,0x95,0xad,0x92,0xb6,0xf2,0x95,0xd6,0x10,0x32,
		0x05,0x7f,0x9d,0xa6,0xdb,0xef,0xc4,0x51,0x0a,0x62,0x3c,0x2b,0x47,0xa5,0x22,0x00,
		0x82,0xa3,0xbc,0x42,0xaf,0x1a,0x14,0x4f,0x98,0xc9,0xee,0x4f,0xda,0xe4,0x1b,0xe0,
		0xec,0x50,0x1c,0xcc,0x94,0xb2,0xb0,0x64,0x01,0x91,0x09,0x9b,0x35,0x56,0x11,0x16,
		0x0d,0xeb,0x32,0x7e,0x8a,0xce,0x01,0x8b,0x89,0x80,0x25,0xef,0x47,0x0e,0x43,0x73,
		0xec,0x1d,0x97,0xf6,0x69,0xe2,0x98,0xe1,0xd8,0x45,0xc6,0x55,0x3c,0x0a,0x54,0x6c,
		0xcb,0x16,0x8d,0x5b,0x51,0x0d,0xbe,0x60,0x18,0xfd,0x4e,0xd9,0xa3,0x54,0x5f,0x9b,
		0xdb,0x81,0x96,0x8f,0x4a,0x6d,0x7c,0x79,0x0e,0x5c,0x34,0x72,0x9a,0x8e,0xfb,0x49,
		0x60,0x86,0xfa,0x13,0x00,0x24,0x9a,0xb8,0xb2,0x8f,0x38,0x95,0x1d,0x7b,0xee,0x1c,
		0x12,0x7a,0xc3,0xc4,0xd0,0xbd,0x59,0x6e,0xde,0xe1,0xe9,0xd1,0x77,0x81,0xdb,0xb8,
		0x22,0x7d,0x7b,0x5d,0x76,0xce,0x8b,0x8b,0xce,0x03,0xc5,0xd3,0x39,0xb9,0x75,0x79,
		0x81,0x61,0x08,0x48,0xc5,0x5c,0xdd,0x02,0x81,0xc0,0x63,0x57,0xa5,0x96,0x79,0xd2,
		0x68,0x01,0x51,0x4c,0x69,0x40,0xc2,0x0e,0xb6,0x7b,0x37,0x0e,0x84,0xe9,0xf5,0xf0,
		0xf9,0x31,0x6c,0x04,0x37,0xd3,0xcb,0x7c,0x84,0x3f,0x5a,0x6e,0x6d,0x9c,0x19,0xe8,
		0xbd,0xb3,0x15,0x2e,0x93,0xf9,0x04,0xcf,0xe6,0xe6,0x92,0xf1,0xee,0xd2,0x7a,0x0a,
		0xda,0x46,0xf9,0x56,0x01,0xb3,0xd1,0x22,0xbe,0x79,0x3d,0xad,0x9b,0xdd,0x05,0xd4,
		0xf6,0xd4,0x69,0x10,0x5e,0xcf,0xc1,0x14,0x48,0x38,0x1d,0xc1,0x54,0xdd,0xad,0xf6,
		0xbc,0x20,0xc6,0x49,0x43,0x5b,0x48,0x35,0x85,0xd6,0x8a,0x52,0x7b,0x7b,0x96,0x7b,
		0xe5,0x2e,0x35,0xe0,0xbe,0x9a,0x43,0x70,0x21,0xc1,0xcf,0xa5,0xf4,0x77,0x15,0x67,
		0xcc,0x23,0x3c,0x1c,0xe3,0xae,0x99,0xeb,0x37,0xda,0xf8,0xbd,0x10,0x15,0x6b,0x4b,
		0xd5,0x80,0xa3,0xce,0x9c,0x7d,0x39,0x1b,0xdb,0xb2,0x3e,0x67,0x36,0x3a,0x94,0x74,
		0x05,0xc6,0xc8,0x12,0xcb,0xd3,0xdc,0xcc,0x8b,0x35,0x6a,0x2d,0xaf,0xd0,0xd3,0xb2,
		0x3a,0x21,0xb6,0x84,0xb4,0x58,0xe4,0xab,0x38,0x54,0xbc,0xd9,0xbe,0x04,0xcd,0xc9,
		0xd6,0x5c,0xee,0xb1,0x0a,0x85,0x31,0xc4,0x70,0xed,0x02,0x81,0xbf,0x04,0xda,0xda,
		0xbf,0xc1,0x5b,0x1a,0x8b,0xdc,0x0f,0x56,0x6f,0x87,0x61,0x91,0x08,0x8a,0x79,0x86,
		0xf6,0xc2,0xb8,0xc0,0x4b,0xa0,0xe0,0x80,0x1d,0x31,0xcb,0xf5,0xd2,0xa4,0x13,0x9a,
		0x39,0xce,0xc9,0xdf,0x14,0xec,0xee,0x22,0xe8,0x46,0xa7,0xd3,0xf4,0xa5,0xe8,0xee,
		0xd2,0xa7,0x0c,0x7a,0x4c,0x2c,0xf9,0x5c,0xe7,0x4f,0xe4,0x2c,0x4b,0xf6,0x0c,0x13,
		0x5a,0x26,0x49,0x19,0xbb,0x4c,0xc9,0x06,0xba,0x28,0x3d,0x18,0x96,0xf0,0xae,0x48,
		0x52,0x9b,0x49,0x0f,0x0c,0x85,0xab,0x03,0x06,0x8c,0xbf,0xee,0x8f,0xa6,0xbb,0x6a,
		0xe7,0x3b,0x18,0x2d,0x25,0xcd,0x66,0xf5,0x20,0x5b,0x03,0x8b,0x4e,0xea,0xf1,0xaa,
		0xfe,0x2e,0x1b,0xa5,0xde,0x97,0xc8,0x8d,0x40,0xfa,0x1a,0xc4,0x76,0x26,0x60,0x2f,
		0xc9,0x0a,0xe6,0x94,0x73,0x4f,0x44,0xf3,0xe4,0xe8,0x8d,0x18,0x4e,0x88,0x05,0xa7,
		0x55,0xac,0x29,0x04,0xbe,0x8f,0xe9,0xde,0xf6,0xb7,0xa6,0x2c,0xc9,0xeb,0xcf,0x4d,
		0x7c,0x2d,0x6c,0x9f,0x9e,0x86,0xb2,0x48,0x3e,0x9b,0xf2,0x2c,0xe5,0x18,0x61,0xbb,
		0xb4,0xe7,0x3e,0x73,0x1a,0x4d,0xbe,0xba,0x87,0x77,0x2d,0x29,0x02,0x81,0xc0,0x21,
		0x4a,0x1f,0x73,0x13,0x0e,0x48,0xb3,0x36,0xfe,0x01,0xb9,0x50,0x88,0x5e,0xcd,0xb3,
		0x44,0x3d,0x93,0xe7,0xe8,0xca,0x62,0xfb,0x0d,0xa9,0x6b,0xd4,0x23,0x75,0x9d,0x8b,
		0xe5,0x52,0xc8,0xbe,0x44,0xf1,0x39,0xfb,0xee,0x6e,0xc2,0x4b,0x75,0xfb,0xf0,0x74,
		0x4f,0xac,0x4d,0xaa,0xbf,0x54,0x88,0xfe,0x6c,0x36,0x00,0xd9,0xb8,0xe9,0xa9,0x22,
		0x48,0x1f,0xc7,0x4a,0x7a,0x3d,0x62,0x26,0x62,0xdb,0x8c,0x85,0x31,0x8d,0xe4,0x8e,
		0xe8,0xb7,0x16,0xf1,0x94,0x29,0xfb,0x59,0x49,0x90,0xda,0x70,0x5e,0xbd,0xf7,0xef,
		0x66,0x13,0xdd,0x6b,0xf8,0x85,0xc1,0x6a,0xd6,0x5e,0x9f,0xe6,0xc2,0x80,0x38,0x6b,
		0xee,0x97,0x6c,0x25,0xdb,0xaf,0xf8,0xfb,0xf6,0x9b,0xae,0xd9,0x51,0x0b,0xe5,0xed,
		0xed,0x3f,0x90,0xe0,0xba,0x4a,0x97,0xe5,0xc8,0x1a,0x21,0x89,0xf1,0x14,0x67,0x07,
		0x45,0xab,0x95,0xed,0xda,0x21,0x5b,0xd0,0x5f,0xdc,0x78,0x92,0x9f,0xa0,0xcf,0xe8,
		0xb0,0x1c,0x83,0xf2,0xae,0xc9,0x3e,0x3a,0xd1,0xa3,0x34,0xfd,0x85,0xaa,0x87,0x94,
		0xea,0xcf,0x95,0x5a,0xe5,0xda,0xcd,0x45,0xb2,0x68,0x74,0x1f,0xca,0x19,0x5c);

/**
 * RSA OAEP SHA256 decryption test vectors from https://boringssl.googlesource.com/
 */
static rsa_oaep_test_t rsa_2048_oaep_sha256_tests[] = {
	{ chunk_from_chars(                                                          /* 2 */
		0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
		0x00,0x00,0x00,0x00),
	  chunk_from_chars(),
	  chunk_from_chars(
	  	0x20,0x71,0x80,0xc3,0x40,0x65,0x8b,0x51,0x54,0xae,0x45,0xd2,0xe4,0xe7,0x32,0x6a,
	  	0x09,0x97,0xc6,0x83,0xa2,0x6b,0x59,0x5e,0x53,0x6a,0x29,0x33,0x3c,0x4b,0x66,0x14,
	  	0x9a,0xf8,0x5e,0x02,0x9d,0x54,0x19,0xa3,0x9e,0x3a,0x14,0x7b,0x22,0x15,0x16,0xff,
	  	0xd8,0x6b,0x6b,0x4b,0x66,0xc3,0xe0,0xc4,0xc4,0x9f,0xe8,0xc5,0x7a,0x2f,0x5c,0x37,
	  	0xb8,0x70,0x4b,0x9b,0x59,0x2b,0x80,0xdb,0x9c,0xd7,0x88,0xa4,0xed,0x51,0xab,0x4f,
	  	0x0a,0x1c,0xbe,0xd6,0x3b,0xd1,0x8d,0x1f,0x06,0xa2,0x2f,0x22,0x58,0x66,0xb0,0xc2,
	  	0xc4,0x17,0xcb,0x23,0x47,0x3b,0x7b,0xa4,0x25,0x0b,0x13,0x53,0xbd,0x2e,0x5b,0x4f,
	  	0x0f,0x93,0x7c,0xd2,0xef,0xe5,0xfa,0x38,0xdb,0x3c,0x29,0x5f,0x77,0x48,0xb9,0x70,
	  	0x08,0x86,0x57,0xdb,0x4a,0xa9,0xa7,0x6e,0x1e,0xe6,0xfb,0xff,0x16,0x6e,0xc1,0x86,
	  	0x1d,0x00,0xd0,0x85,0x32,0x6c,0x73,0x84,0xbd,0xd1,0xbc,0x2f,0x40,0x0d,0x4f,0x74,
	  	0xdb,0xdf,0xad,0xaf,0x3f,0xdc,0x46,0x07,0x3e,0x66,0x85,0x73,0xe0,0x20,0x30,0xb9,
	  	0xeb,0x5a,0xf5,0x8e,0xb5,0x40,0xc6,0x66,0x77,0xa7,0x71,0x19,0x44,0x79,0xec,0x00,
	  	0x98,0xd8,0x58,0xa2,0xea,0x45,0xd0,0xba,0x1e,0x6b,0x32,0x44,0x0d,0xfb,0xac,0x74,
	  	0x50,0x00,0x55,0x4d,0x51,0xa1,0x76,0x84,0xca,0x96,0x4b,0x02,0xa7,0x4d,0x47,0x9f,
	  	0x1d,0x43,0x2e,0xf7,0x63,0xef,0x40,0x59,0x71,0x5a,0x43,0x48,0xcf,0xe3,0x6a,0x21,
	  	0x53,0x59,0x71,0x2f,0x25,0xb6,0x97,0x79,0x03,0xbe,0x4a,0xdb,0x92,0xfe,0xbb,0xf6)
	},
	{ chunk_from_chars(                                                          /* 3 */
		0x54,0x65,0x73,0x74),
	  chunk_from_chars(),
	  chunk_from_chars(
	  	0x5e,0xab,0x3f,0x07,0x41,0xe6,0x39,0x86,0xed,0x64,0x7d,0x53,0xe1,0xcd,0x71,0xdf,
	  	0x04,0x19,0x86,0x90,0x08,0x03,0xd0,0xf9,0x9c,0x68,0x35,0x5d,0x24,0x9a,0x15,0xa4,
	  	0x7d,0xc5,0xb4,0xf7,0x0a,0x19,0x14,0x77,0x65,0x42,0x99,0xe5,0xa2,0x73,0x1f,0x3b,
	  	0x4e,0xec,0x76,0xde,0xa1,0x82,0x62,0xfc,0x69,0x6a,0xc7,0x94,0xe5,0xf6,0x6c,0xbf,
	  	0xcd,0xda,0xc4,0x47,0x2c,0x57,0x8e,0x24,0x6c,0x26,0x70,0x75,0x98,0x05,0x55,0x84,
	  	0x54,0x0b,0x83,0x98,0x36,0xb1,0x40,0x4c,0x56,0x11,0xae,0x55,0x8a,0x98,0x4c,0xee,
	  	0x8f,0xd0,0x36,0xce,0xa9,0x24,0xe0,0xbe,0x24,0x74,0xa9,0x40,0xf6,0x1e,0x0a,0xcc,
	  	0x14,0xfc,0xae,0x95,0xeb,0xdc,0x59,0x94,0x2a,0x9c,0xe9,0xaf,0x9a,0x9c,0x81,0x99,
	  	0x9f,0x7f,0x68,0x15,0xf0,0x57,0xff,0xdc,0x25,0x33,0xcb,0x15,0xd6,0x39,0x1d,0x1e,
	  	0x2d,0x95,0xf1,0x6f,0x9c,0x04,0x20,0x9c,0x88,0x9a,0x4c,0x35,0x9c,0x7d,0x29,0x26,
	  	0xd2,0x8a,0x66,0xe2,0xb0,0x30,0xa4,0x16,0xb9,0x28,0xd2,0x82,0x56,0x27,0x99,0x8e,
	  	0x51,0x91,0xfb,0x49,0x83,0xa6,0xe6,0x50,0x24,0x26,0x2d,0x94,0xfc,0x09,0x18,0x7a,
	  	0x2d,0x78,0x16,0x21,0x22,0x43,0x32,0x51,0xd1,0xbf,0xcc,0x8e,0x50,0x7d,0x06,0xeb,
	  	0xa2,0xd2,0x29,0xc1,0x00,0x31,0x26,0x1d,0xa3,0x2a,0xb8,0xcc,0xd1,0x5f,0x1c,0x5f,
	  	0x9f,0xbf,0x07,0xed,0x15,0x84,0x83,0xd7,0x36,0xa1,0x10,0xaf,0x4b,0x44,0xd6,0xa4,
	  	0xda,0x60,0xd6,0xcb,0x51,0x9b,0x44,0x54,0x21,0x3c,0xf9,0xf0,0xdc,0x56,0x0f,0x2b)
	},
	{ chunk_from_chars(                                                          /* 4 */
		0x31,0x32,0x33,0x34,0x30,0x30),
	  chunk_from_chars(),
	  chunk_from_chars(
		0x0d,0xa2,0x90,0xb0,0xbf,0x71,0xa1,0x14,0x1b,0x2a,0xdc,0x62,0xb5,0xb4,0x2b,0x07,
		0xfc,0x52,0x52,0x0c,0xb0,0x5d,0x1f,0x39,0xcc,0xca,0x7b,0x7c,0x22,0xd3,0x79,0xa6,
		0xc2,0xf2,0xd9,0x31,0x43,0xd0,0x57,0x05,0x01,0x40,0x52,0x7a,0x1e,0x56,0x38,0x24,
		0x39,0x38,0xb5,0x31,0xec,0x3d,0xe7,0x01,0x4b,0x01,0x51,0x30,0x1e,0x49,0xb9,0xfa,
		0x43,0x34,0x82,0xa0,0x2a,0xbd,0xfd,0x94,0x19,0x3d,0xc9,0xc7,0x01,0x57,0xe5,0x57,
		0x77,0x6a,0x69,0xde,0xd0,0x1e,0xcb,0xea,0xc4,0x05,0x13,0x35,0x95,0xc6,0x11,0x65,
		0xa4,0x28,0xb6,0x28,0x47,0x29,0xa4,0x74,0x6b,0x47,0xd2,0xbb,0xac,0xa9,0xd7,0x43,
		0x2c,0x3b,0x56,0x93,0x59,0x1c,0xfe,0xe4,0x88,0xeb,0x3c,0x68,0xdd,0xb6,0xcd,0xe0,
		0xdd,0x61,0xcf,0xc6,0x95,0x24,0x23,0xf9,0x94,0xab,0xec,0xee,0x34,0xf5,0x68,0x37,
		0x32,0xb2,0x99,0x34,0xa2,0xc4,0x98,0xad,0x48,0xcd,0xd3,0x0c,0x14,0x91,0x77,0x18,
		0x9f,0x48,0xee,0xfd,0x9c,0xc7,0x23,0x2d,0xf1,0x8b,0xe1,0x1f,0xf5,0xf7,0xe3,0xaf,
		0x73,0x19,0xe3,0x11,0x59,0x97,0xc5,0x99,0xe2,0xd8,0xf7,0xf4,0xf6,0x63,0xdc,0xc4,
		0x00,0x32,0xd4,0x03,0x43,0x6d,0x30,0x58,0xa5,0xea,0x66,0x47,0x36,0x60,0xf0,0xe1,
		0x95,0x28,0x1e,0xc6,0xeb,0xa9,0x97,0xd2,0xa9,0x51,0xca,0x6c,0xa5,0xf3,0xc1,0x12,
		0x71,0x5c,0x89,0xda,0x1d,0x3d,0xfb,0x20,0xa2,0x19,0x40,0x59,0x4e,0xd1,0x0a,0xde,
		0x90,0xed,0x74,0x52,0xb7,0x53,0x75,0x7d,0x72,0x41,0xcb,0x8a,0x80,0x3c,0x37,0x3a)
	},
	{ chunk_from_chars(                                                          /* 8 */
		0x31,0x32,0x33,0x34,0x30,0x30),
	  chunk_from_chars(
	  	0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00),
	  chunk_from_chars(
	  	0x65,0x83,0xe2,0xf1,0x76,0xaa,0x7e,0x7f,0x65,0x5d,0x2c,0x53,0x49,0x73,0x49,0xc1,
	  	0x56,0xc8,0x85,0x1f,0xb2,0x33,0x25,0x58,0x9e,0x85,0xfb,0x83,0xbf,0xa8,0x57,0x34,
	  	0x6c,0xab,0xa2,0x22,0xcd,0xaa,0x32,0x34,0xe7,0x15,0x64,0x15,0x42,0x98,0xc2,0x4d,
	  	0xbb,0x85,0xe1,0x88,0x22,0xa1,0xd5,0xe7,0xfa,0xa4,0x78,0x63,0xa6,0x4d,0x76,0x87,
	  	0x4a,0x3c,0xbc,0x70,0xf4,0xd9,0xf1,0x37,0x42,0x6a,0x34,0x4c,0x47,0x3f,0xac,0x1d,
	  	0xd7,0x00,0x8a,0x99,0x73,0x76,0x5e,0x9f,0x66,0xc5,0xb4,0x92,0x53,0x5a,0x64,0x7c,
	  	0x27,0x3c,0x4f,0x78,0xce,0xb5,0xaa,0x7b,0xa9,0x63,0xa2,0x14,0x2f,0x2c,0xe4,0xa8,
	  	0x1f,0x80,0x4c,0x00,0x2b,0x9b,0x2e,0xab,0xb3,0xc7,0x5e,0x80,0xa3,0xc6,0xce,0xaf,
	  	0xe5,0x38,0x4a,0x54,0x4c,0x67,0x2a,0x5d,0x28,0xd3,0x2b,0xb8,0x71,0x15,0xf4,0x3e,
	  	0xb7,0x97,0x75,0xfd,0x9b,0x3f,0x4a,0x2f,0x6e,0x6a,0x89,0x36,0x8b,0xdd,0x95,0xef,
	  	0x1d,0x01,0x48,0x77,0xb6,0x0a,0xfd,0xb1,0x23,0x4a,0xcd,0x57,0x65,0x3a,0x65,0x45,
	  	0x9f,0x01,0xb2,0xfb,0xe3,0x81,0xf2,0x2a,0x73,0x95,0x04,0xb4,0x89,0x7a,0x7e,0x6c,
	  	0x33,0xb6,0x34,0x9b,0x27,0x6d,0xb6,0x08,0x3a,0xba,0xd9,0xc1,0x69,0x40,0x58,0x59,
	  	0xb8,0x00,0xc8,0x12,0x23,0x76,0x34,0xb5,0x03,0xde,0x6a,0xda,0x43,0x01,0x3c,0x1d,
	  	0x86,0x69,0x7a,0x13,0x5b,0xe7,0x8a,0x97,0x84,0x57,0x6d,0x79,0x6d,0x62,0xaa,0x78,
	  	0x19,0xe2,0xea,0x0e,0x2d,0x90,0x2f,0xfd,0xd9,0xcf,0xdd,0x1a,0xe6,0x62,0x12,0xee)
	},
	{ chunk_from_chars(                                                          /* 9 */
		0x31,0x32,0x33,0x34,0x30,0x30),
	  chunk_from_chars(
		0x00,0x01,0x02,0x03,0x04,0x05,0x06,0x07,0x08,0x09,0x0a,0x0b,0x0c,0x0d,0x0e,0x0f,
		0x10,0x11,0x12,0x13),
	  chunk_from_chars(
		0xa0,0x44,0x3b,0xa4,0x34,0x15,0x6d,0x0b,0x50,0x3e,0xc6,0x62,0xf5,0xeb,0x5b,0x10,
		0xe2,0x0a,0xd0,0xcb,0x82,0x33,0x72,0x0e,0xe1,0x87,0xba,0x98,0x6e,0x48,0x11,0xdd,
		0x31,0x28,0x44,0xd3,0xed,0xb2,0x6b,0xfa,0xf5,0x1b,0x4b,0x92,0x68,0xdc,0x3c,0x76,
		0x07,0x2d,0xd4,0x7c,0x19,0x9d,0x71,0x3c,0x91,0x82,0x4d,0xa2,0x3f,0xf0,0x04,0x81,
		0xee,0x69,0xe9,0xd4,0xcc,0x54,0x31,0x20,0xfc,0x33,0xb7,0x24,0x4b,0xf0,0xc1,0xad,
		0x5f,0xdd,0x1a,0xe9,0xcb,0xad,0xa7,0xfe,0x9a,0x70,0xad,0x0a,0xfa,0xaa,0xad,0x83,
		0x61,0xe8,0xdc,0x4b,0x31,0x98,0xae,0x66,0x1a,0x84,0xe2,0x75,0xb6,0x0b,0xf2,0xeb,
		0xb8,0x5e,0x51,0x2d,0xa7,0x85,0xd2,0xfc,0x34,0x82,0x29,0x4f,0xa1,0x19,0x67,0x68,
		0x1d,0x7b,0xce,0xff,0xc0,0x8c,0xe0,0xe3,0x6f,0x0a,0x8a,0xf7,0xfc,0xfb,0x13,0x37,
		0x18,0x68,0x63,0xc2,0xc1,0xc1,0xb9,0x4c,0x9e,0xc9,0x78,0x5c,0xd3,0xd9,0x4d,0x15,
		0x43,0x7c,0x23,0xb7,0x75,0x67,0x7f,0x3d,0x29,0xa4,0xc9,0xe5,0x2f,0x13,0x39,0x8f,
		0xd1,0x46,0x61,0x16,0x0e,0x58,0x68,0xbc,0xa9,0x76,0x25,0xaa,0x6c,0x7e,0xcb,0x07,
		0xbb,0xb4,0x79,0x64,0x4d,0xef,0x35,0x3f,0x1f,0x01,0xa4,0xc4,0x10,0x0f,0x9a,0xdb,
		0x82,0xc4,0xf6,0xa2,0x65,0xa5,0xee,0x96,0x2d,0xa5,0x8c,0x3c,0x04,0x2a,0xa5,0x49,
		0xc9,0xd2,0xde,0x30,0x08,0xe7,0x44,0x8e,0x0c,0x4b,0x9b,0x4a,0xc8,0xf5,0xe4,0xd8,
		0x62,0x98,0x73,0x90,0x9b,0xb9,0x95,0xcc,0xc0,0x82,0x5f,0xe8,0x7d,0x81,0xd5,0x96)
	},
	{ chunk_from_chars(                                                         /* 10 */
		0x31,0x32,0x33,0x34,0x30,0x30),
	  chunk_from_chars(
		0x00,0x01,0x02,0x03,0x04,0x05,0x06,0x07,0x08,0x09,0x0a,0x0b,0x0c,0x0d,0x0e,0x0f,
		0x10,0x11,0x12,0x13,0x14,0x15,0x16,0x17,0x18,0x19,0x1a,0x1b,0x1c,0x1d,0x1e,0x1f),
	  chunk_from_chars(
		0x26,0xaa,0x8e,0x79,0x31,0xed,0x62,0x4b,0xfd,0x40,0x77,0xe3,0xb8,0x3e,0xc0,0x8d,
		0xe0,0x80,0x48,0x3a,0x61,0x64,0x1e,0x87,0x7f,0x49,0x3f,0xb7,0xd0,0xae,0x4d,0xfe,
		0xbd,0x4f,0x05,0x61,0x2a,0x2e,0x48,0x69,0xd2,0x0d,0x44,0x0a,0x64,0xe9,0x28,0xb1,
		0x3d,0xaa,0x3b,0x79,0xda,0x2e,0xd6,0x74,0xb9,0x94,0x21,0xdf,0x7e,0x7f,0x62,0x5a,
		0x22,0xb1,0xc7,0x11,0x36,0xda,0x27,0xdb,0x69,0x9d,0x3d,0xd9,0x6e,0x33,0x64,0xee,
		0x0c,0xd2,0x12,0x3a,0xb6,0x80,0x89,0x30,0xc6,0xbc,0x28,0xa5,0xdc,0x30,0x78,0x80,
		0xd1,0xab,0x4b,0x03,0xbc,0xd6,0x17,0x8a,0x81,0xb8,0xad,0x52,0xaa,0xff,0xfb,0xab,
		0x38,0x7d,0x40,0x35,0x2d,0xfe,0xa5,0x26,0xab,0xed,0xca,0x01,0x6c,0x87,0xe9,0xe5,
		0x6c,0xcc,0x4c,0x88,0xe8,0x0f,0x57,0x9d,0xa0,0x15,0xb9,0x79,0xbc,0xdd,0x88,0x61,
		0x8b,0x2a,0x32,0xce,0x07,0x29,0x18,0xb2,0xe2,0x23,0x53,0x53,0x59,0xf1,0xce,0x4e,
		0xab,0xa5,0xe6,0x92,0xe6,0x29,0x6b,0x21,0x40,0xdc,0x23,0x04,0x09,0x2e,0xbd,0x6f,
		0x13,0x6a,0x48,0x09,0x2b,0x38,0x49,0x08,0x2b,0x57,0xe7,0x0c,0x93,0xb5,0x4d,0xb5,
		0x50,0x45,0xdd,0x60,0x94,0xef,0x3d,0x2c,0xfa,0x8b,0xc9,0xe2,0xfd,0x2b,0x1b,0xbe,
		0x0c,0x7c,0x60,0x3a,0xd3,0x8d,0x3f,0x40,0xc9,0xea,0xc8,0xae,0x5e,0x28,0xcb,0xbb,
		0x03,0x1c,0x38,0xd9,0x3d,0x3b,0x25,0x41,0xd9,0x4e,0xab,0x3a,0x1e,0x89,0x92,0xa4,
		0x44,0xee,0x4c,0xe7,0xb8,0xd0,0x8c,0x0b,0x9a,0x4f,0x62,0x3d,0x32,0xfc,0xba,0x14)
	},
	{ chunk_from_chars(                                                         /* 11 */
		0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,
		0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,
		0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,
		0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,
		0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,
		0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,
		0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,
		0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,
		0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,
		0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,
		0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,
		0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78),
	  chunk_from_chars(),
	  chunk_from_chars(
	  	0x7e,0xfb,0x69,0xf1,0x13,0x7d,0x6a,0x6e,0x81,0x3b,0x7a,0xb7,0x5b,0xf0,0x40,0x0b,
	  	0x3d,0x07,0xa4,0x42,0xb8,0x8a,0xb0,0x48,0x67,0x5d,0xc0,0x6b,0x02,0x15,0xfc,0x1a,
	  	0x2e,0x03,0x32,0x63,0xec,0x31,0xa6,0xc1,0xd2,0xea,0xc5,0x6c,0xb0,0x47,0x0d,0x69,
	  	0x02,0x2a,0x48,0x13,0x1d,0x1f,0x00,0x0b,0xbe,0xd7,0x05,0x86,0xb8,0x0c,0xf6,0x35,
	  	0x64,0x65,0xc8,0x83,0x4d,0xac,0xed,0x7e,0xa2,0xa5,0xff,0x8e,0xf9,0xc4,0x4d,0x5a,
	  	0xb8,0x28,0xff,0xbf,0x05,0x56,0xa6,0x39,0x47,0x52,0xa4,0xa2,0x8a,0x70,0xca,0xe2,
	  	0x00,0x84,0xe1,0x23,0x6f,0x04,0x2f,0x6c,0x29,0xde,0x5c,0xb3,0x4e,0xf7,0x3a,0xcb,
	  	0xa5,0xab,0xcc,0x7c,0xcb,0x3a,0x26,0x34,0x27,0x01,0xdf,0x3b,0x9d,0xaa,0x94,0x5d,
	  	0x9f,0xa5,0xbf,0x0b,0x9b,0x10,0x30,0x66,0x55,0xe5,0x63,0x70,0x18,0x3f,0x50,0xfb,
	  	0x83,0x21,0xf8,0xf0,0xcd,0x1c,0x72,0x11,0x47,0x91,0xfc,0xa5,0xdf,0x21,0x66,0x29,
	  	0x6b,0x50,0x9b,0x01,0xa0,0xb2,0x91,0xc4,0x61,0x10,0x78,0x7c,0xeb,0xe6,0x9d,0x41,
	  	0xb3,0xb1,0xe8,0x95,0x90,0xbc,0x2f,0x5e,0x5d,0x49,0xea,0x24,0xae,0x0f,0x12,0x07,
	  	0xee,0xf1,0xae,0xe5,0x4b,0x97,0x60,0x55,0x3e,0x80,0xc5,0x50,0x6a,0x8a,0x8a,0x75,
	  	0x73,0x2e,0x92,0x87,0x50,0x25,0xf0,0xbf,0xd5,0xea,0xd7,0x1e,0x43,0x40,0xc8,0xa9,
	  	0xfa,0x16,0xdc,0xd5,0xa7,0xdc,0x96,0xd8,0xc4,0xa7,0xdc,0x4e,0x91,0xf4,0x7a,0x69,
	  	0x36,0x64,0x45,0xc4,0x69,0x5c,0x8b,0xad,0x57,0x8f,0xfe,0x52,0xbb,0x67,0x2f,0x65)
	}
};

static rsa_oaep_test_t rsa_3072_oaep_sha256_tests[] = {
	{ chunk_from_chars(                                                          /* 2 */
		0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
		0x00,0x00,0x00,0x00),
	  chunk_from_chars(),
	  chunk_from_chars(
		0x4a,0xae,0x85,0x95,0x88,0x3d,0xed,0xea,0x13,0xa1,0xfd,0x65,0x64,0x05,0xae,0x1a,
		0x94,0x78,0x5e,0xb8,0x63,0x18,0xe1,0x81,0xb7,0x47,0x33,0x10,0x19,0x08,0x76,0x86,
		0xcb,0x81,0xb2,0x59,0xf8,0x64,0xc6,0xa5,0x24,0x13,0x7a,0x31,0x6f,0x74,0x4c,0x0d,
		0x09,0x23,0x48,0xfe,0x04,0x28,0xe1,0xd9,0x55,0x1b,0xd4,0x02,0x40,0x43,0x42,0xe1,
		0xdf,0x33,0xf7,0x49,0x1c,0x38,0x1a,0x4f,0x32,0xae,0x56,0x87,0xbf,0xae,0x07,0xe1,
		0xc4,0x08,0xc3,0xe6,0x57,0x20,0xf5,0x4c,0x5a,0x16,0x8a,0x29,0xdf,0x6b,0x7f,0xbc,
		0x18,0x35,0xfa,0x49,0xed,0x5a,0x8b,0x42,0xf2,0xac,0xbb,0xde,0xfb,0x3c,0xf5,0x8f,
		0xb6,0x87,0x6c,0x2a,0x0a,0x71,0x0b,0xdc,0x5c,0x90,0x32,0x01,0x8c,0x6b,0x32,0x63,
		0x89,0xb3,0xc3,0x2e,0xfb,0xd7,0xae,0xfa,0x8d,0xfd,0xe0,0xae,0x7b,0xbd,0x82,0x9e,
		0x61,0x60,0xce,0xf2,0xc8,0x63,0x23,0x35,0x00,0x6b,0xaa,0xd3,0x2f,0xa7,0xaf,0x1f,
		0xc9,0xfc,0x3e,0xc5,0xac,0x9c,0x4f,0x7a,0xf2,0x26,0xe7,0x56,0x1c,0xb4,0x43,0x38,
		0xd4,0x28,0xd9,0x78,0x86,0xd2,0x90,0x09,0x98,0x9e,0x1d,0x87,0x56,0x50,0xc9,0xf3,
		0xfa,0x8a,0xf2,0xa0,0x92,0x4f,0x93,0xd0,0x0d,0x2c,0xfc,0xaa,0xe6,0x7e,0x8a,0x89,
		0xef,0x34,0xd9,0x6a,0xfb,0xa4,0x78,0xa2,0x50,0xb7,0x99,0x5e,0x78,0x82,0x87,0x3b,
		0xec,0x86,0x95,0x41,0xc0,0x2e,0xd4,0x6b,0x41,0x0a,0xb7,0x5a,0x21,0x4f,0x0f,0xf0,
		0x8c,0xea,0xed,0x44,0x44,0x08,0x0a,0x4a,0xde,0x6b,0xaf,0x39,0xaa,0xa9,0xd9,0xba,
		0x52,0x48,0x7d,0xca,0x33,0x75,0x91,0x53,0x34,0x55,0x38,0x14,0x3d,0x11,0xaf,0x5f,
		0x0c,0xa3,0x35,0xa2,0xa3,0xb9,0x4e,0x10,0x45,0xda,0x7f,0xf6,0x90,0xa4,0xbd,0x45,
		0x4f,0xb7,0x88,0xbd,0x7c,0x73,0xb0,0xb1,0x6e,0x35,0xec,0x27,0x66,0x02,0x4f,0xd0,
		0xb7,0xfa,0x44,0xe0,0xca,0x01,0xfc,0x61,0xe3,0x31,0xb1,0xfc,0x8b,0x09,0xf5,0x49,
		0xa9,0x46,0x10,0x1e,0x2b,0x60,0xc3,0x39,0x30,0x4a,0x48,0xd0,0x44,0xfe,0x23,0x12,
		0x88,0x12,0x9d,0x2b,0x70,0xd6,0xb0,0x85,0x92,0x28,0x8d,0x17,0x51,0x83,0xe4,0x42,
		0xb9,0xaa,0x4f,0x5e,0xc6,0xea,0xc0,0x14,0x8b,0x6b,0xe1,0xf9,0x89,0x26,0x3d,0x3c,
		0xf3,0x4a,0x7f,0xf2,0x65,0x53,0x9b,0x8e,0x34,0xc2,0x11,0x7b,0x7a,0x1c,0x76,0x16)
	},
	{ chunk_from_chars(                                                          /* 3 */
		0x54,0x65,0x73,0x74),
	  chunk_from_chars(),
	  chunk_from_chars(
		0x8f,0x71,0x39,0x1b,0x8e,0x3a,0x07,0x9a,0x51,0xd0,0x15,0xe8,0x6e,0x05,0x72,0xac,
		0xa1,0x34,0xe5,0xaf,0x78,0x20,0x00,0x54,0x5d,0xa1,0xcb,0xbb,0x4c,0xdf,0xaf,0x83,
		0x65,0x42,0xca,0xd1,0x9b,0xbc,0x6a,0xca,0x73,0x36,0x7d,0xff,0x9c,0xd1,0xe3,0x36,
		0x31,0x26,0x0d,0x39,0x0b,0x6d,0x83,0x88,0x2f,0x82,0x60,0x6e,0x28,0xab,0xd2,0xd8,
		0xaa,0x22,0x9e,0xc1,0xd9,0x3b,0x4c,0xf1,0x65,0x99,0xdf,0xad,0xd2,0x17,0x80,0x29,
		0x36,0x0e,0xf5,0xa3,0xbe,0xa6,0xa1,0x36,0xcd,0x43,0x7f,0x59,0x28,0x9d,0x33,0x4f,
		0x20,0xe7,0xba,0xe9,0x1e,0xcc,0xa1,0x8b,0x7e,0x4f,0xa5,0xe3,0x61,0x3e,0x48,0x9e,
		0xc8,0x06,0xa5,0x2b,0xc4,0x72,0x73,0x33,0xbd,0x7e,0xbd,0x5b,0xcc,0x75,0x68,0xf6,
		0x97,0xd4,0x01,0x3d,0x0f,0x6f,0x54,0xca,0xf1,0xba,0xf3,0x67,0x91,0xfd,0x21,0xf8,
		0xce,0x84,0x64,0x30,0x31,0xff,0xe4,0x74,0x90,0x19,0x20,0xeb,0x4f,0x59,0x2f,0xfa,
		0x30,0x6a,0x97,0x9e,0xdf,0xd6,0xc2,0x87,0xee,0x5b,0x1f,0xb0,0x00,0x5a,0x21,0xd8,
		0x40,0x37,0x2c,0x60,0xa6,0x49,0xa1,0x70,0x2f,0xbc,0x46,0x48,0x92,0x11,0x70,0x0e,
		0x82,0x62,0xbb,0x1a,0x2f,0x54,0xfd,0x38,0xda,0x6a,0x07,0xe1,0x58,0x50,0x40,0x5a,
		0x27,0x77,0x05,0xa6,0x6e,0x70,0x88,0xb9,0x8b,0xfa,0xee,0x8b,0x2b,0x51,0x99,0x27,
		0x97,0xd5,0xe5,0x71,0x21,0x47,0xb7,0x56,0x36,0x71,0x70,0xb6,0xfd,0xe2,0x6c,0x3f,
		0xb5,0x98,0x8e,0xe0,0x7f,0x91,0x6d,0x18,0x9a,0x3a,0x18,0xcd,0xc0,0xf3,0xb1,0xaf,
		0xfe,0x63,0x53,0x8c,0xbc,0x18,0x7d,0x4b,0xfd,0xcf,0x51,0x49,0xb4,0xd2,0x6b,0xb2,
		0x07,0xa1,0x49,0x5d,0x51,0xa2,0xee,0x1c,0x24,0x15,0xf4,0xa0,0xd6,0x43,0x4d,0x50,
		0x95,0x3f,0xf2,0xfe,0x84,0x80,0xd2,0x76,0x49,0x1c,0xa1,0x46,0x8e,0x86,0xc2,0x31,
		0x8e,0x94,0x08,0x7e,0x98,0x44,0x44,0xb0,0x4e,0x7b,0x80,0x4c,0xb1,0xf6,0x2c,0x66,
		0x57,0x3e,0x82,0xfa,0x68,0x00,0xac,0x3a,0x42,0xce,0x0d,0x8e,0x59,0x09,0xf3,0xdf,
		0xba,0x02,0x4a,0x40,0xc5,0x1b,0x21,0xb5,0x45,0x58,0x36,0xa9,0x24,0x4e,0xeb,0xa2,
		0x51,0x57,0xe2,0x46,0x76,0xe1,0xa3,0x17,0xd1,0xaf,0x9b,0xf4,0x77,0x39,0x13,0xec,
		0x3b,0xa6,0xed,0x1d,0xf5,0x0d,0x38,0x10,0xeb,0x98,0xbc,0xd2,0x90,0x75,0x7d,0xaa)
	},
	{ chunk_from_chars(                                                          /* 4 */
		0x31,0x32,0x33,0x34,0x30,0x30),
	  chunk_from_chars(),
	  chunk_from_chars(
		0x15,0x75,0x62,0xf6,0x1b,0x51,0x2b,0x11,0xb0,0x2d,0x1f,0x82,0x99,0xe6,0x2f,0x55,
		0xff,0xb9,0xc1,0xcb,0x5f,0xac,0xb3,0x3c,0x75,0xfe,0xe7,0x9e,0xa8,0xbf,0xb6,0x83,
		0x14,0xd4,0x8a,0x78,0x68,0xc0,0x5b,0x8c,0x97,0xde,0x4a,0xf5,0x4e,0x49,0x60,0x86,
		0x8a,0x69,0x4a,0x3c,0x30,0x2c,0x73,0x19,0x5a,0x8a,0xcf,0xdc,0x78,0x13,0x72,0x31,
		0xdc,0x96,0x0d,0xab,0x44,0xba,0xd1,0x91,0xc4,0xbb,0x2b,0x9f,0x8b,0x16,0x2a,0x73,
		0x79,0x9b,0xa2,0x08,0x3d,0x93,0xc5,0x5b,0x42,0x5f,0x58,0x57,0xc9,0x25,0xf2,0xae,
		0xa6,0xaf,0x19,0x51,0x48,0x25,0xfb,0x18,0x72,0x45,0x55,0xa5,0xb1,0x1e,0xf5,0x53,
		0x88,0x9a,0xf3,0x8c,0x75,0x85,0xc3,0x50,0x33,0xd1,0xeb,0x9b,0x71,0xe5,0x6e,0xc0,
		0xa1,0xa1,0x18,0x5e,0x4e,0x6e,0x5b,0x7a,0xab,0x35,0x14,0x04,0xf0,0xd4,0xa4,0x41,
		0x60,0x23,0x44,0x9d,0x84,0xf3,0xbe,0xbd,0xf5,0x7d,0x56,0x66,0xff,0xae,0xa8,0x0b,
		0xe1,0x95,0x8b,0xe5,0xd7,0xda,0x7a,0xd2,0xc0,0x81,0xaa,0x33,0x82,0x03,0x51,0x6a,
		0xba,0x1e,0x11,0x64,0x21,0x92,0xaa,0x40,0x24,0xe5,0xfe,0xd6,0x11,0x9a,0xad,0x74,
		0x56,0xbf,0x13,0xc2,0xdd,0x56,0x1c,0x4a,0xc9,0xac,0x88,0x7e,0xea,0xf4,0xb5,0x3b,
		0xf1,0x5e,0x00,0x7b,0x04,0x3d,0xe9,0xab,0xd6,0x61,0x9a,0x6f,0xad,0x5d,0xf3,0x5a,
		0x33,0x6f,0x74,0x4d,0x50,0x34,0xaf,0x72,0xa6,0xce,0xec,0xb9,0x30,0x5d,0xa9,0x73,
		0xc5,0x06,0x15,0xc2,0x9b,0x98,0xa8,0xd0,0xaf,0xa8,0xc5,0x4f,0x5c,0xfe,0x80,0xf8,
		0xef,0xba,0x51,0xb0,0x8c,0xf9,0x21,0xcf,0x55,0xc6,0x93,0x14,0x52,0x36,0x19,0xe7,
		0x53,0xf3,0x05,0x6e,0xe7,0x04,0xa9,0x47,0xc2,0x4d,0x12,0xe8,0x81,0x7d,0x53,0x12,
		0xea,0x5f,0x6d,0x84,0x44,0xda,0x40,0x61,0xd2,0x6f,0x18,0xab,0x66,0x0f,0x0e,0xb8,
		0x80,0x19,0x50,0x6e,0xda,0x48,0x01,0x95,0x39,0x5e,0x6e,0x94,0x8d,0x42,0xe9,0x6c,
		0x3a,0xc8,0x72,0x3f,0x55,0x14,0xd8,0x7d,0xa3,0xbc,0x47,0x43,0x2a,0xb6,0x0e,0x93,
		0xef,0xd8,0xe6,0x10,0x05,0x89,0x31,0x8f,0x1b,0x85,0xd1,0xc4,0xbd,0x04,0x82,0x0b,
		0xe2,0x89,0x2a,0x8a,0xcc,0xa1,0xe3,0x63,0x39,0x50,0xb1,0x11,0x85,0x9c,0x6e,0xe2,
		0x06,0x3c,0xc0,0x61,0xa4,0x4a,0x73,0x7f,0xfa,0xa2,0xc1,0x58,0xda,0x80,0x68,0x64)
	},
	{ chunk_from_chars(                                                          /* 8 */
		0x31,0x32,0x33,0x34,0x30,0x30),
	  chunk_from_chars(
	  	0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00),
	  chunk_from_chars(
		0x7a,0x9f,0x4a,0x75,0x37,0x50,0x02,0x88,0x9b,0xb9,0xc4,0xca,0xfe,0x5f,0x04,0x4e,
		0x2f,0x28,0x99,0x74,0x74,0x46,0x52,0x51,0x24,0x67,0x88,0xe5,0x1b,0x0c,0xd5,0xc5,
		0x2c,0x80,0x9a,0x69,0xf1,0xcc,0xef,0x4c,0x11,0x72,0x3a,0xf0,0x30,0xfb,0x69,0x8a,
		0x64,0x1b,0x98,0xb8,0x8b,0x39,0xc1,0x52,0xe7,0x41,0xb8,0xee,0x29,0xb1,0x38,0x57,
		0xec,0x14,0x4f,0x20,0x0c,0x66,0x90,0x10,0xa6,0x57,0x44,0x17,0x01,0xe9,0x29,0xed,
		0x7d,0xf1,0x95,0x66,0x91,0x97,0xf4,0x69,0x09,0xac,0xc6,0x93,0x88,0x12,0x8b,0xb7,
		0x56,0x45,0xf2,0x70,0xf2,0x0d,0xf2,0x56,0xf0,0xd0,0xc0,0xed,0x48,0x8e,0xfc,0xf2,
		0x6c,0x44,0xe4,0xd4,0xe5,0x4a,0x87,0x57,0xf6,0xc5,0xbc,0x70,0x08,0xc6,0x8b,0x7f,
		0xec,0x43,0x77,0x87,0x43,0xa7,0x6a,0x89,0x0d,0x38,0x3e,0x29,0x83,0xa7,0x54,0x09,
		0x52,0x24,0xc5,0x68,0x62,0xb4,0xb6,0x2e,0x20,0xf1,0x12,0xd1,0xbd,0x96,0xf3,0x0a,
		0x3e,0x66,0xa2,0x0b,0x01,0x06,0x9c,0x0e,0xd9,0x73,0x0f,0x9f,0x7d,0xe6,0xcb,0x13,
		0xe2,0x27,0x26,0x40,0xca,0x5c,0xf8,0x07,0xe6,0x4f,0x44,0x29,0x7e,0x3e,0x58,0xee,
		0x93,0x31,0xe1,0xf0,0x46,0x10,0x69,0x4a,0x0d,0x5b,0xe7,0x00,0x6f,0x96,0x74,0x7c,
		0xf7,0x30,0xff,0x3f,0xc4,0xbf,0x88,0x84,0xb3,0xf9,0xf0,0xcb,0xa9,0xc4,0xa8,0xf3,
		0x8a,0x01,0xb2,0x25,0xb0,0x83,0xd7,0x75,0x16,0xd4,0x6a,0xde,0x25,0x82,0x42,0xb0,
		0xea,0x4f,0x36,0x7b,0xcd,0xff,0x49,0x0e,0x2f,0x12,0x7f,0x01,0x38,0x08,0xcf,0xee,
		0xd4,0x51,0xaa,0x7c,0x0f,0x64,0xb3,0x15,0x6f,0xc9,0x68,0x50,0x7e,0xc7,0xc8,0x05,
		0x72,0x69,0x3d,0x15,0x4b,0x92,0x4f,0xe1,0x8d,0xfe,0xa9,0x46,0xd5,0x2d,0xa8,0x1a,
		0x5a,0xd0,0xbf,0xb5,0xfa,0xc7,0x01,0x0e,0xd5,0xb4,0xf1,0x8a,0x0b,0xd1,0xef,0x40,
		0x0a,0x28,0x04,0xb9,0xca,0x2b,0x2d,0xd0,0xbb,0x8b,0x8b,0x40,0x29,0x52,0x94,0x9b,
		0xba,0x93,0x5b,0x67,0x4c,0x48,0x63,0xea,0x4a,0x9e,0x38,0xdd,0x70,0x1d,0x23,0xc7,
		0x1e,0x29,0xb4,0xa7,0xc6,0x95,0x85,0x6d,0xb5,0x97,0x8b,0x7e,0x3c,0x96,0x4a,0xd8,
		0xfd,0x4d,0xec,0xb1,0xcb,0x5e,0xb3,0xf8,0x3e,0x8e,0xf0,0xbe,0x4e,0x5f,0xef,0xd2,
		0x7f,0x86,0x04,0xb1,0xd6,0xfa,0x06,0xdd,0x4f,0x97,0x11,0x03,0x17,0xbb,0x19,0xd9)
	},
	{ chunk_from_chars(                                                          /* 9 */
		0x31,0x32,0x33,0x34,0x30,0x30),
	  chunk_from_chars(
		0x00,0x01,0x02,0x03,0x04,0x05,0x06,0x07,0x08,0x09,0x0a,0x0b,0x0c,0x0d,0x0e,0x0f,
		0x10,0x11,0x12,0x13),
	  chunk_from_chars(
		0x1f,0x1e,0xf3,0xe2,0x95,0x6c,0x31,0xf6,0xb9,0x8f,0xb4,0x14,0x1a,0x42,0xfb,0x2b,
		0x1a,0x14,0x33,0x1b,0x55,0x91,0xf4,0x0b,0x5f,0x34,0xfb,0x3c,0xff,0x53,0x67,0x53,
		0xee,0xd8,0x5f,0x9c,0x54,0xf2,0x4c,0x76,0xb2,0x9c,0xf7,0x01,0x08,0x75,0xb1,0x7e,
		0x2b,0xda,0xdd,0xdf,0xfe,0x23,0x0d,0x41,0x1c,0x87,0x71,0x8a,0x5f,0x0c,0xdb,0xe6,
		0xc1,0x20,0xad,0xe6,0x6b,0x21,0x71,0x94,0x9b,0xcf,0x7d,0xf6,0x5e,0x6c,0x68,0x72,
		0x49,0xc0,0xde,0xdc,0xd3,0x58,0xc3,0x8a,0xa1,0xb6,0x18,0x67,0xe2,0x43,0x5c,0x47,
		0x8f,0x46,0xbe,0x88,0x53,0x90,0x6c,0xc0,0x87,0x41,0xa1,0xc0,0x92,0xc2,0xfc,0x57,
		0x7a,0x0b,0x8a,0x32,0xc8,0x18,0x43,0x45,0x75,0xb5,0x4b,0xd5,0x8e,0x17,0xfc,0xd1,
		0xa6,0xf9,0x10,0xb1,0x85,0xdf,0x53,0x44,0x9f,0xa8,0x8a,0x77,0x7f,0x1f,0x79,0x5a,
		0x89,0xee,0xe2,0x58,0x6c,0x1c,0xac,0xd4,0xd1,0x27,0x73,0x33,0xca,0x2e,0xbb,0xe0,
		0x43,0x01,0x90,0x28,0x96,0x65,0xc3,0xef,0xea,0x05,0x87,0x12,0x27,0x67,0x0f,0xd0,
		0x72,0x50,0x65,0x2a,0x1d,0xd2,0xaf,0xa6,0xed,0x9c,0x29,0x5e,0xbc,0x68,0xb1,0x5a,
		0x25,0x47,0x25,0xdd,0x60,0x06,0xd2,0x8a,0xec,0x6b,0x1a,0xcd,0xaf,0xba,0x69,0x24,
		0x77,0x90,0xce,0xac,0xed,0xf5,0xef,0xf4,0x37,0xd5,0x69,0xa7,0x1e,0x55,0x64,0xd6,
		0xec,0x6c,0x90,0xd2,0x95,0xe2,0x25,0xfb,0xf5,0x25,0xe7,0x6b,0x2e,0x46,0x69,0x85,
		0x48,0x29,0x27,0x34,0x57,0x37,0xea,0xbf,0xa3,0x40,0x46,0xef,0x61,0x84,0x94,0xaf,
		0x1c,0x84,0x7b,0x1d,0x81,0x9f,0x2f,0x99,0xb1,0x5c,0x13,0xb8,0xe8,0x17,0xd4,0x37,
		0xdb,0x38,0xa5,0x74,0xf6,0x6d,0xa8,0x45,0xf5,0x48,0x95,0x71,0x0f,0x1b,0x68,0x5c,
		0xbd,0xd8,0x3f,0xbf,0xf9,0x83,0x5a,0x26,0xde,0x07,0x4c,0x5b,0x17,0x12,0xd9,0x8d,
		0x4e,0x03,0x6c,0x97,0x20,0xd9,0x21,0x73,0x9b,0x8f,0x3d,0xc3,0x20,0xf1,0xd2,0x66,
		0xae,0x5b,0x36,0x7b,0x92,0xf5,0xe3,0x3a,0xe2,0x8a,0x7a,0x06,0xf7,0x27,0xf2,0xdf,
		0x54,0x37,0xe5,0x5c,0x32,0x5c,0xb2,0x33,0xc8,0x01,0xdd,0xf6,0xb1,0x22,0x40,0xcc,
		0xd0,0xb8,0xdb,0x8c,0x31,0xa5,0xca,0x65,0x2c,0xd0,0xe3,0x3b,0xc1,0x08,0x7f,0x4b,
		0xbd,0x70,0xa0,0x14,0x26,0x17,0x90,0x39,0x56,0x44,0x3f,0x4c,0xce,0xa5,0xcd,0x93)
	},
	{ chunk_from_chars(                                                         /* 10 */
		0x31,0x32,0x33,0x34,0x30,0x30),
	  chunk_from_chars(
		0x00,0x01,0x02,0x03,0x04,0x05,0x06,0x07,0x08,0x09,0x0a,0x0b,0x0c,0x0d,0x0e,0x0f,
		0x10,0x11,0x12,0x13,0x14,0x15,0x16,0x17,0x18,0x19,0x1a,0x1b,0x1c,0x1d,0x1e,0x1f),
	  chunk_from_chars(
		0x85,0x18,0x6e,0x5b,0xd1,0x21,0x79,0x6c,0x60,0x4c,0xee,0xae,0xbe,0x2f,0x12,0xfe,
		0x87,0xf4,0x1f,0x50,0x2c,0x26,0x22,0xf0,0xac,0x85,0x99,0x42,0x8f,0x55,0x6f,0x7d,
		0xdb,0xa8,0x23,0x55,0xe6,0x5d,0xde,0x44,0xfe,0x16,0x91,0xb7,0x90,0x1f,0xcf,0x6e,
		0xd8,0x3f,0xb8,0x37,0xc1,0xf7,0xca,0x54,0x9b,0x25,0x65,0xda,0x1a,0x24,0x9d,0xc7,
		0x17,0xc2,0x45,0xa9,0xc6,0x36,0x2c,0x8e,0x83,0x1e,0xe4,0x15,0x6d,0xfd,0xec,0x6d,
		0x9e,0x19,0xee,0x50,0x02,0x43,0xed,0xf4,0x41,0x9d,0x15,0x2c,0x64,0x33,0x60,0xe3,
		0x59,0x35,0xd8,0x8a,0x03,0x64,0xb2,0x80,0x38,0xcb,0xab,0x30,0xd2,0x3f,0x65,0x71,
		0x05,0x3b,0xe9,0x9e,0x0e,0x40,0x55,0x38,0xc5,0x4a,0x71,0xc4,0x48,0x3c,0xaa,0x32,
		0x8a,0x8a,0x49,0xb1,0xe9,0xa5,0xab,0x01,0x72,0xd4,0x29,0xb5,0x63,0x5f,0xa7,0xcc,
		0x11,0xf5,0x70,0x3b,0xb9,0x55,0x36,0xd2,0x18,0xbd,0xb4,0xed,0x55,0x5a,0x2b,0x1b,
		0x71,0xc7,0x17,0xbd,0x65,0xe3,0x60,0x7d,0xfd,0x7e,0x7c,0xfc,0x1f,0xa7,0xf4,0x2e,
		0x72,0xa5,0xda,0xae,0x3e,0x6b,0xd6,0x94,0xa4,0xa6,0x19,0x4b,0xce,0x97,0x0c,0xf1,
		0x02,0xba,0xfc,0x55,0x73,0xb8,0xde,0x18,0x54,0x40,0x80,0xdb,0x3e,0x89,0xa7,0x53,
		0xb9,0x24,0x5e,0xd9,0xcf,0x7a,0x17,0x28,0x20,0x17,0xc8,0x27,0x6b,0xfb,0xe7,0x37,
		0xbf,0x0d,0x80,0x2a,0x7d,0x2e,0x9a,0x9f,0xb5,0xf3,0x73,0xf9,0x76,0x07,0x08,0x21,
		0x1f,0x9b,0x7a,0x99,0xd3,0xdc,0xe0,0xbe,0xcf,0xa7,0x7c,0x96,0xb1,0xf5,0xfe,0x1c,
		0x4a,0xe8,0xcb,0x2b,0x4f,0x7c,0x74,0x1e,0x90,0xe4,0xbd,0x2f,0xd1,0xb9,0x36,0x48,
		0x25,0x2e,0x45,0x76,0xb5,0x67,0x0d,0x51,0xe8,0x60,0x68,0x49,0x7d,0xdf,0x16,0x2a,
		0x80,0xdb,0xc2,0xa1,0x6b,0x73,0xe1,0x6e,0x36,0x61,0xf3,0x1a,0x8f,0x2f,0xdb,0xb6,
		0x67,0x64,0xa7,0x0a,0x1c,0xec,0x5e,0x0e,0x24,0x7e,0x38,0xf3,0xfe,0x95,0x94,0x9a,
		0x57,0x42,0xb1,0x98,0x72,0x80,0xba,0x9d,0x4a,0xfe,0x69,0xcd,0x00,0xfe,0x10,0xfe,
		0xee,0x87,0xb9,0xd0,0xb4,0x0a,0x2e,0x94,0xfc,0x04,0x2a,0x4a,0x51,0x47,0x4d,0x7e,
		0x16,0x2a,0x1e,0x72,0x61,0xb2,0x79,0x1f,0x8a,0xf2,0xaf,0x69,0x78,0xbb,0x87,0xea,
		0x10,0x59,0x55,0x3a,0xf7,0x17,0x77,0xfd,0x04,0xfb,0x92,0xc4,0xfe,0xca,0xb9,0x72)
	},
{ chunk_from_chars(                                                         /* 11 */
		0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,
		0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,
		0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,
		0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,
		0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,
		0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,
		0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,
		0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,
		0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,
		0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,
		0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,
		0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,
		0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,
		0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,
		0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,
		0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,
		0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,
		0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,
		0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,
		0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78),
	  chunk_from_chars(),
	  chunk_from_chars(
		0x6a,0xa1,0x69,0xad,0x08,0x9b,0x2c,0xd0,0xc5,0xc0,0xd0,0x74,0x2e,0x45,0x2c,0x4f,
		0x1b,0xfe,0x9a,0xa0,0x4f,0x4d,0xf9,0x3e,0xeb,0x38,0xad,0x6d,0xe8,0xcc,0x93,0xda,
		0x66,0xaf,0x9b,0xa5,0x89,0x35,0x24,0x46,0xb8,0xc3,0xa5,0x79,0x7c,0xb0,0xfb,0x8b,
		0x0b,0x6d,0x6b,0x6f,0xca,0x1f,0x9a,0xe4,0xd5,0x08,0x95,0x1b,0xf0,0xb5,0x26,0xa9,
		0x43,0x3a,0x9b,0x26,0xb5,0xce,0xe4,0xaa,0x94,0x3f,0x0d,0x2c,0x0d,0x94,0xc4,0x79,
		0xef,0x8f,0x99,0xe2,0xbb,0x68,0x0a,0xa4,0xbd,0xde,0xf3,0x8e,0xf6,0x07,0xce,0x27,
		0x70,0xf4,0xd6,0x06,0x29,0xeb,0xfa,0xbd,0xd9,0xfc,0x20,0xc6,0xb8,0x8c,0x8a,0x02,
		0x17,0x9d,0xce,0x5e,0x9a,0xd9,0xbe,0xb9,0x0c,0xf7,0x5b,0xaf,0x59,0x80,0xed,0xbd,
		0x5b,0x92,0x79,0x4b,0xdf,0x9b,0xc5,0x8d,0xcc,0x44,0xa1,0x3b,0x18,0x11,0x79,0xe9,
		0x2f,0xa3,0x94,0x15,0x25,0x06,0x84,0x51,0x88,0xb8,0x0a,0xed,0xc2,0xfe,0x55,0x9a,
		0x3b,0xbc,0x28,0x9b,0x93,0x3f,0xa2,0xcd,0xde,0xfe,0x08,0x08,0x17,0xf6,0xfb,0x3f,
		0x1e,0x8f,0x58,0x53,0x89,0xeb,0xbe,0xf0,0x6c,0x33,0x76,0xca,0xa9,0x75,0xc0,0x7e,
		0xfe,0x68,0x3a,0x32,0xe6,0x49,0x17,0x56,0x66,0x45,0x33,0x0d,0xca,0x75,0x48,0x0e,
		0x7a,0xd2,0xea,0xe5,0xb4,0x50,0x5b,0x6a,0xd5,0x23,0x3e,0x7c,0xf4,0xc3,0xd8,0x67,
		0x7e,0x03,0x50,0xb8,0xe1,0xdb,0x4c,0xbf,0xd7,0x33,0x44,0xe1,0x4f,0x45,0x8f,0x49,
		0x84,0x9c,0xd9,0x91,0xd3,0xb4,0xfd,0x19,0x06,0x1b,0xf2,0x57,0x84,0xe1,0x7b,0x04,
		0x23,0x0b,0x9a,0x1d,0xb1,0x63,0x7a,0x7b,0x8c,0x0f,0x4e,0x3b,0xfe,0x00,0x95,0x22,
		0x01,0xc0,0x39,0x48,0x35,0x51,0x93,0x1f,0x84,0x98,0x0b,0xf8,0xa6,0x33,0xa1,0xbb,
		0xf4,0xb6,0x09,0xad,0x64,0xac,0x6d,0x26,0x51,0x4c,0x86,0x9c,0x1d,0x7b,0x91,0x5b,
		0x22,0x33,0x56,0xfb,0xb0,0x7f,0xad,0x30,0x0c,0xc9,0xa8,0xe6,0xc7,0x68,0x92,0x7a,
		0x85,0x7c,0x7e,0x9a,0x58,0xb0,0xd7,0x84,0xa0,0x4e,0x24,0x08,0x2a,0x39,0x3e,0x65,
		0xf0,0x98,0xff,0x21,0x1d,0xcd,0xe8,0xa5,0x15,0x55,0x50,0xc1,0x8b,0xb2,0x2d,0x64,
		0xd3,0x24,0x22,0x4d,0x77,0xe1,0x50,0xbf,0x42,0x42,0x62,0xf1,0x78,0x27,0xf1,0x39,
		0x20,0xec,0x7d,0xf2,0x59,0xcc,0xfb,0xa1,0x58,0x76,0x6b,0x0b,0x52,0xbf,0x25,0xb9)
	}
};

static void test_rsa_oaep(chunk_t key, encryption_scheme_t encryption_scheme,
						  rsa_oaep_test_t *test)
{
	private_key_t *privkey;
	public_key_t *pubkey;
	chunk_t msg = chunk_empty, ct = chunk_empty;
	bool success;

	privkey = lib->creds->create(lib->creds, CRED_PRIVATE_KEY, KEY_ANY,
								 BUILD_BLOB_ASN1_DER, key, BUILD_END);
	ck_assert(privkey != NULL);
	ck_assert(privkey->get_type(privkey) == KEY_RSA);

	pubkey = privkey->get_public_key(privkey);
	ck_assert(pubkey != NULL);

	/* decrypt */
	success = privkey->decrypt(privkey, encryption_scheme, &test->label,
							   test->ct, &msg);
	if (test->label.len > 0 && !success)
	{
		warn("RSA OAEP labels might not be supported");
		goto end;
	}
	ck_assert(success);
	ck_assert_chunk_eq(msg, test->msg);
	chunk_free(&msg);

	/* encrypt and decrypt again */
	ck_assert(pubkey->encrypt(pubkey, encryption_scheme, &test->label,
							  test->msg, &ct));
	ck_assert(privkey->decrypt(privkey, encryption_scheme, &test->label,
							   ct, &msg));
	ck_assert_chunk_eq(msg, test->msg);
	chunk_free(&ct);
	chunk_free(&msg);

end:
	pubkey->destroy(pubkey);
	privkey->destroy(privkey);
}

START_TEST(test_rsa_2048_oaep_sha256)
{
	test_rsa_oaep(rsa_2048_key, ENCRYPT_RSA_OAEP_SHA256,
				 &rsa_2048_oaep_sha256_tests[_i]);
}
END_TEST

START_TEST(test_rsa_3072_oaep_sha256)
{
	test_rsa_oaep(rsa_3072_key, ENCRYPT_RSA_OAEP_SHA256,
				 &rsa_3072_oaep_sha256_tests[_i]);
}
END_TEST

Suite *rsa_oaep_sha256_suite_create()
{
	Suite *s;
	TCase *tc;

	s = suite_create("rsa_oaep_sha256");

	tc = tcase_create("rsa_2048_oaep_sha256");
	tcase_add_loop_test(tc, test_rsa_2048_oaep_sha256, 0, countof(rsa_2048_oaep_sha256_tests));
	suite_add_tcase(s, tc);
	tc = tcase_create("rsa_3072_oaep_sha256");
	tcase_add_loop_test(tc, test_rsa_3072_oaep_sha256, 0, countof(rsa_3072_oaep_sha256_tests));
	suite_add_tcase(s, tc);

	return s;
}
