/*
    Tucnak - VHF contest log
    Copyright (C) 2009 Ladislav Vaiz <ok1zia@nagano.cz>

    This program is free software; you can redistribute it and/or
    modify it under the terms of the GNU General Public License
    version 2 as published by the Free Software Foundation.

*/

#include "header.h"

struct qrvdb *qrvdb;

int xyz=1;
struct qrvdb *init_qrv(void){
    struct qrvdb *qrvdb;
    int i;

    qrvdb = g_new0(struct qrvdb, 1);
    qrvdb->qrvs = g_index_array_new();
    qrvdb->hash = g_hash_table_new(g_str_hash, g_str_equal);
    qrvdb->def_bands = 0;

    for (i=0; i<cfg->bands->len; i++){
        struct config_band *confb = g_index_array_index(cfg->bands, i);
        if (confb->qrv || confb->qrvnow) qrvdb->def_bands |= 1 << (toupper(confb->bandchar)-'A');
    }    
    qrvdb->sort = qrv_compare_call;
    qrvdb->sort2 = NULL;
    qrvdb->showall = 0;
    strcpy(qrvdb->search, "");
    return qrvdb;
}

void free_qrv_item(struct qrv_item *qi){
    CONDGFREE(qi->call);
    CONDGFREE(qi->wwl);
    CONDGFREE(qi->text);
    g_free(qi);
}

gboolean qrv_free_hash_item(gpointer key, gpointer value, gpointer user_data){
    //dbg("%s: %d\n", __FUNCTION__, xyz++);
    g_free(key);
    //dbg("%d\n", xyz++);
    // value is already freeed
    return TRUE;

}
void clear_qrv(struct qrvdb *qrvdb){
    int i;
    struct subwin *sw;
    //dbg("%s: a%d\n", __FUNCTION__, xyz++);
    if (gses && gses->subwins){  // gses==NULL when terminating
        for (i=0;i<gses->subwins->len;i++){
            sw=(struct subwin *)g_index_array_index(gses->subwins, i);
            if (sw->type!=SWT_MAP) continue;
#ifdef HAVE_SDL            
    //dbg("%s: b%d\n", __FUNCTION__, xyz++);
            CONDGFREE(sw->minqrvcall);
    //dbg("%s: c%d\n", __FUNCTION__, xyz++);
#endif
        }
    }

    for (i = qrvdb->qrvs->len - 1; i >= 0; i--){
        struct qrv_item *qi = (struct qrv_item*)g_index_array_index(qrvdb->qrvs, i);
        g_index_array_remove_index(qrvdb->qrvs, i);
    //dbg("%s: d%d\n", __FUNCTION__, xyz++);
        free_qrv_item(qi);
    //dbg("%s: e%d\n", __FUNCTION__, xyz++);
    }
    //dbg("%s: f%d\n", __FUNCTION__, xyz++);
    g_hash_table_foreach_remove(qrvdb->hash, qrv_free_hash_item, NULL);
    //dbg("%s: g%d\n", __FUNCTION__, xyz++);
}


void free_qrv(struct qrvdb *qrvdb){
    xyz = 1;
    //dbg("%s: u%d\n", __FUNCTION__, xyz++);
    clear_qrv(qrvdb);
    //dbg("%s: v%d\n", __FUNCTION__, xyz++);

    g_index_array_free(qrvdb->qrvs, 1);
    //dbg("%s: w%d\n", __FUNCTION__, xyz++);
    g_hash_table_destroy(qrvdb->hash);
    //dbg("%s: x%d\n", __FUNCTION__, xyz++);
    g_free(qrvdb);
    //dbg("%s: y%d\n", __FUNCTION__, xyz++);
}


struct qrv_item *qrv_add(gchar *call, gchar *wwl, int qrv_int, int wkd[32], const gchar *text, time_t kst_time){
    struct qrv_item *qi;
    char raw[20];
     

    if (!wwl || !*wwl){
        wwl = find_wwl_by_call(cw, call);
        if (!wwl) wwl="";
    }
    //dbg("qrv_add('%s', '%s', %d, (%d;%d;%d), '%s', %d)\n", call, wwl, qrv_int, wkd[2], wkd[4], wkd[6], text, (unsigned int)kst_time);
    

    qi = qrv_get(qrvdb, call);
    if (qi){
 //       dbg("qrv_update(%s)\n", call);
        if (wwl && *wwl) {
            CONDGFREE(qi->wwl);
            qi->wwl = uc(g_strdup(wwl)); 
            qrv_compute_qrbqtf(qi);
        }
        //q->bands |= qrv_int;  why comment ???
        if (text && strlen(text) > strlen(qi->text)){
            CONDGFREE(qi->text);
            qi->text = g_strdup(text);
        }
        qi->kst_time = kst_time;
        return qi;
    }
    
//    dbg("qrv_new(%s)\n", call);
    qi = g_new0(struct qrv_item, 1);
    qi->call  = uc(g_strdup(call));
    qi->wwl   = uc(g_strdup(wwl));
    qi->bands_qrv = qrv_int;
    memcpy(qi->wkd, wkd, sizeof(qi->wkd));
    qi->text  = g_strdup(text);
    qi->kst_time = kst_time;
    qrv_compute_qrbqtf(qi);
    g_index_array_add(qrvdb->qrvs, qi);

    get_raw_call(raw, call);
    g_hash_table_insert(qrvdb->hash, g_strdup(raw), qi); 
//    dbg("add_to_list(%s %s %s)\r\n", qi->call, qi->wwl, qi->text);
    return qi;
}

int is_qrv(char *s){
    char *c, d, last = '\0';

    if (!s) return 0;
    if (strlen(s)==0) return 0;
    for (c=s; *c!='\0'; c++){
        d = toupper(*c);
        if (d < 'A' || d > 'Z') return 0;
        if (d < last) return 0;
        last = d;
    }
    return 1;
}

#define QRV_DELIM " \t\r\n"
void load_one_qrv(struct qrvdb *qrvdb, gchar *s){
	struct zstring *zs;
    gchar *call, *wwl, *text, *str, *c;
    int qrv_int = 0, wkd[32];
    time_t kst_time = 0;

	zs = zstrdup(s);
	//dbg("ztokens=%d '%s'\n", ztokens(zs), s);
	if (ztokens(zs) >= 6){ 
		ztokenize(zs, 1);  // first ';'
		
        call = ztokenize(zs, 0);

		wwl = ztokenize(zs, 0);

		str = ztokenize(zs, 0);
		for (c = str; *c != '\0'; c++) qrv_int |= 1 << (toupper(*c) - 'A');

		str = ztokenize(zs, 0);
        memset(wkd, 0, sizeof(wkd));
		for (c = str; *c != '\0'; c++) {
            if (*c < 'A' || *c > 'Z') continue;
            wkd[*c - 'A'] = atoi(c+1);
            if (c[1] == '\0' || c[1] > '9') wkd[*c - 'A'] = 1; // for compatibility, 'ABC' means A=1, B=1, C=1 
            // number will be skipped by c++ and next condition
        }

		text = ztokenize(zs, 0);

		str = ztokenize(zs, 0);
        if (str) kst_time = atoi(zs->str);
        
		qrv_add(call, wwl, qrv_int, wkd, text, kst_time);
	}
    zfree(zs);
}

void old_load_one_qrv(struct qrvdb *qrvdb, gchar *s){
    gchar *call, *wwl, *qrv_str, *text;
    int qrv_int, free_text = 0, wkd[32];
    char *token_ptr, *c;

   // dbg("load_one_qrv('%s')\n", s);
    call = strtok_r(s, QRV_DELIM, &token_ptr);
    if (!call) return;

    wwl = strtok_r(NULL, QRV_DELIM, &token_ptr);
    if (!wwl) return;

    qrv_str = strtok_r(NULL, QRV_DELIM, &token_ptr);
    if (!qrv_str) qrv_str = "";

    text = token_ptr;

    memset(wkd, 0, sizeof(wkd));
    if (!is_qrv(qrv_str)){
//        dbg("!isqrv(%s)\n", qrv_str);
        qrv_int = qrvdb->def_bands;
        //bands_wkd_me = qrvdb->def_bands;
        text = g_strconcat(qrv_str, " ", text, NULL);
        free_text = 1;
    }else{
        qrv_int = 0;
        for (c=qrv_str;*c!='\0';c++){
            if (isupper(*c)) {
                //dbg("wkd[%d]=1\n", *c-'A');
                wkd[*c-'A'] = 1;
            }
            *c=upcase(*c);
            qrv_int|=1<<(*c-'A');
        }
    }

    qrv_add(call, wwl, qrv_int, wkd, text, 0);
    if (free_text) g_free(text);
}


int load_qrv_from_file(struct qrvdb *qrvdb, gchar *filename){
    FILE *f;
    char s[202];

    f = fopen(filename, "rt");
    if (!f){
        return -1;
    }

    while((fgets(s, 200, f))!=NULL){
        chomp(s);
		if (s[0]==';')
			load_one_qrv(qrvdb,s);
		else
			old_load_one_qrv(qrvdb,s);
    }
    fclose(f);
    g_index_array_qsort(qrvdb->qrvs, qrvdb->sort);
    return 0;
}


int save_qrv_to_file(char *filename){
    FILE *f;
    char qrv_str[32], wkd_str[32*30], kst_str[20];
    int i;
    char  x[2], d;
	
    f = fopen(filename, "wt");
    if (!f){
        log_addf(VTEXT(T_CANT_WRITE_S), filename);
        return errno;
    }

    for (i=0; i<qrvdb->qrvs->len; i++){
		struct zstring *zs;
        struct qrv_item *qi = (struct qrv_item*)g_index_array_index(qrvdb->qrvs, i);
#if 0
        for (j=0, c=qrv_str; j<32; j++){
            if (!(qi->bands_qrv & (1<<j))) continue;
            if (qi->bands_wkd_me & (1<<j))
                *c='A'+j;
            else
                *c='a'+j;
            c++;
        }
        *c='\0';
        fprintf(f, "%s\t%s\t%s\t%s\n", qi->call, qi->wwl, qrv_str, qi->text); 
#endif
		x[1]='\0';
		strcpy(qrv_str, "");
        for (x[0]='A'; x[0]<'Z'; x[0]++) if (qi->bands_qrv & ( 1 << (x[0] - 'A'))) strcat(qrv_str, x);

		strcpy(wkd_str, "");
        for (d='A'; d<'Z'; d++) {
//            dbg("band=%c i=%d wkd=%d \n", d, d - 'A', qi->wkd[d - 'A']);
            if (qi->wkd[d - 'A'] == 0) continue;
//            dbg("%c=%d\n", d, qi->wkd[d - 'A']);
            sprintf(wkd_str+strlen(wkd_str), "%c%d", d, qi->wkd[d - 'A']);
        }

        sprintf(kst_str, "%d", (int)qi->kst_time);

		zs = zconcatesc("", qi->call, qi->wwl, qrv_str, wkd_str, qi->text, kst_str, NULL);
		fprintf(f, "%s\n", zs->str);
		//dbg("zs->str='%s'\n", zs->str);
		zfree(zs);
    }

    fclose(f);
    return 0;
}

int qrv_skip(struct qrv_item *qi, int bi){
    //if (get_qso_by_callsign(aband, qi->call)!=NULL) return 1;
    int ret = 0;

    if ((qi->bands_wkd & (1<<bi))!=0) ret = 2;
    if (qrvdb->showall && !*qrvdb->search) return ret;
    if ((qi->bands_qrv & (1<<bi))==0) ret = 1;
    if (strlen(qrvdb->search) > 1){// here's leading /
        ret = 3;
        if (my_strcasestr(qi->call, qrvdb->search + 1)) ret = 0;
        if (my_strcasestr(qi->wwl,  qrvdb->search + 1)) ret = 0;
        if (my_strcasestr(qi->text, qrvdb->search + 1)) ret = 0;
    }
//    dbg("qrv_skip(%s, %d)=%d  qrv=0x%x  wkd=0x%x\n", qi->call, bi, ret, qi->bands_qrv, qi->bands_wkd);
    return ret;
}


int qrv_compare_call(const void *a, const void *b){
    struct qrv_item **qa, **qb;
	int ret;

    qa = (struct qrv_item **)a;
    qb = (struct qrv_item **)b;
    
    if (!*qa && !*qb) return 0;
    if (!*qa) return -1;
    if (!*qb) return +1;
    ret = strcmp((*qa)->call, (*qb)->call);
    if (ret) return ret;
    if (qrvdb->sort2 == qrv_compare_call) return 0;
    return qrvdb->sort2(a, b);
}

int qrv_compare_wwl(const void *a, const void *b){
    struct qrv_item **qa, **qb;
	int ret;

    qa = (struct qrv_item **)a;
    qb = (struct qrv_item **)b;
    
    if (!*qa && !*qb) return 0;
    if (!*qa) return -1;
    if (!*qb) return +1;
    ret = strcmp((*qa)->wwl, (*qb)->wwl);
    if (ret) return ret;
    if (qrvdb->sort2 == qrv_compare_wwl) return 0;
    return qrvdb->sort2(a, b);
}

int qrv_compare_wkd(const void *a, const void *b){
    struct qrv_item **qa, **qb;
	int ret, bi;

    qa = (struct qrv_item **)a;
    qb = (struct qrv_item **)b;
    
    if (!*qa && !*qb) return 0;
    if (!*qa) return -1;
    if (!*qb) return +1;
    if (!aband) return 0;
    bi = toupper(aband->bandchar) - 'A';
    ret = (*qa)->wkd[bi] - (*qb)->wkd[bi]; 
    if (ret) return -ret;
    if (qrvdb->sort2 == qrv_compare_wkd) return 0;
    return qrvdb->sort2(a, b);
}

int qrv_compare_qrb(const void *a, const void *b){
    struct qrv_item **qa, **qb;
	double d;

    qa = (struct qrv_item **)a;
    qb = (struct qrv_item **)b;
    
    if (!*qa && !*qb) return 0;
    if (!*qa) return -1;
    if (!*qb) return +1;
    d = (*qa)->qrb - (*qb)->qrb; 
    if (d != 0) return d < 0 ? 1 : -1; // odx on top
    if (qrvdb->sort2 == qrv_compare_qrb) return 0;
    return qrvdb->sort2(a, b);
}

int qrv_compare_qtf(const void *a, const void *b){
    struct qrv_item **qa, **qb;
	int ret;
    qa = (struct qrv_item **)a;
    qb = (struct qrv_item **)b;
    
    if (!*qa && !*qb) return 0;
    if (!*qa) return -1;
    if (!*qb) return +1;
    if (!aband) return 0;
    ret = (*qa)->qtf - (*qb)->qtf; 
    if (ret) return ret;
    if (qrvdb->sort2 == qrv_compare_qtf) return 0;
    return qrvdb->sort2(a, b);
}

int qrv_compare_kst_time(const void *a, const void *b){
    struct qrv_item **qa, **qb;
	long int ret;

    qa = (struct qrv_item **)a;
    qb = (struct qrv_item **)b;
    
    if (!*qa && !*qb) return 0;
    if (!*qa) return -1;
    if (!*qb) return +1;
    if (!aband) return 0;
    ret = (int)((*qa)->kst_time - (*qb)->kst_time); 
    if (ret) return ret;
    if (qrvdb->sort2 == qrv_compare_kst_time) return 0;
    return qrvdb->sort2(a, b);
}

int qrv_compare_text(const void *a, const void *b){
    struct qrv_item **qa, **qb;
	int ret;

    qa = (struct qrv_item **)a;
    qb = (struct qrv_item **)b;
    
    if (!*qa && !*qb) return 0;
    if (!*qa) return -1;
    if (!*qb) return +1;
    ret = strcmp((*qa)->text, (*qb)->text);
    if (ret) return ret;
    if (qrvdb->sort2 == qrv_compare_text) return 0;
    return qrvdb->sort2(a, b);
}


void sw_qrv_seek(struct subwin *sw, int value){
    int bi;
    struct qrv_item *qi;

    if (!aband) return;
    bi = toupper(aband->bandchar) - 'A';

    while (value > 0){
        sw->cur++;
        if (sw->cur >= qrvdb->qrvs->len) break;
        qi = (struct qrv_item *)g_index_array_index(qrvdb->qrvs, sw->cur);
        if (qrv_skip(qi, bi)) continue;
        value--;
    }
    
    while (value < 0){
        sw->cur--;
        if (sw->cur < 0) break;
        qi = (struct qrv_item *)g_index_array_index(qrvdb->qrvs, sw->cur);
        if (qrv_skip(qi, bi)) continue;
        value++;
    }
}

int sw_qrv_kbd_func(struct subwin *sw, struct event *ev, int fw){
    int bi, len;
    struct qrv_item *qi;
    
    if (!aband) return 0;
    bi = toupper(aband->bandchar) - 'A';
    
    switch(kbd_action(KM_MAIN,ev)){
        case ACT_ESC:
            return 0;
            break;
        case ACT_DOWN:
            sw_qrv_seek(sw, 1);
            sw_qrv_check_bounds(sw);
            redraw_later();
            return 1;
        case ACT_UP:
            sw_qrv_seek(sw, -1);
            sw_qrv_check_bounds(sw);
            redraw_later();
            return 1;
        case ACT_PAGE_DOWN:
            sw_qrv_seek(sw, sw->h - 1);
            sw_qrv_check_bounds(sw);
            redraw_later();
            return 1;
        case ACT_PAGE_UP:
            sw_qrv_seek(sw, -sw->h + 1);
            sw_qrv_check_bounds(sw);
            redraw_later();
            return 1;
        case ACT_HOME:
            sw->cur = 0;
            sw_qrv_check_bounds(sw);
            redraw_later();
            return 1;
        case ACT_END:
            sw->cur = qrvdb->qrvs->len - 1;
            sw_qrv_check_bounds(sw);
            redraw_later();
            return 1;
		case ACT_SCROLL_LEFT:
    		if (sw->ho>0) sw->ho--;
            redraw_later();
            return 1;
		case ACT_SCROLL_RIGHT:
			sw->ho++;
            redraw_later();
            return 1;
        case ACT_ENTER:
            if (*qrvdb->search){
                strcpy(qrvdb->search, "");
                redraw_later();
                if (qrvdb->qrvs->len == 0) return 1;
                qi = (struct qrv_item*)g_index_array_index(qrvdb->qrvs, sw->cur);
                if (!qi) return 1;
                qi->bands_qrv |= 1<<bi;
                return 1;
            }
            sw_unset_focus();
            il_set_focus(aband->il);
            if (qrvdb->qrvs->len == 0) break;
            qi = (struct qrv_item*)g_index_array_index(qrvdb->qrvs, sw->cur);
            if (!qi) break;
            clear_tmpqsos(aband,1);
            process_input(aband, qi->call, 1);
            redraw_later();
/*#ifdef HAVE_HAMLIB                
            if (trig){
                trig_set_qrg(trig, qi->qrg*1000.0);
#endif                */
            return 1;
        case ACT_BACKSPACE:
            len = strlen(qrvdb->search);
            if (len == 0) break;
            qrvdb->search[len - 1] = '\0';
            redraw_later();
            return 1;
        case ACT_DELETE:
            if (qrvdb->qrvs->len == 0) break;
            qi = (struct qrv_item*)g_index_array_index(qrvdb->qrvs, sw->cur);
            if (!qi) break;
            qi->bands_qrv &= ~(1<<bi);
            redraw_later();
            return 1;
        case ACT_INSERT:
            edit_qrv(CBA0);
            return 1;
        default:
            //dbg("default: ev->x='%c' search='%s'\n", ev->x, qrvdb->search);
            if (*qrvdb->search && ev->y == 0){
                if (!isprint(ev->x)) break;
                len = strlen(qrvdb->search);
                if (len >= QRVSSIZE - 2) break;
                qrvdb->search[len] = toupper(ev->x);
                qrvdb->search[len+1] = '\0';
                //dbg("search='%s'\n", qrvdb->search);
                redraw_later();
                break;
            }

            switch(ev->x){
                case ' ':
                    if (qrvdb->qrvs->len == 0) break;
                    qi = (struct qrv_item*)g_index_array_index(qrvdb->qrvs, sw->cur);
                    if (!qi) break;
                    clear_tmpqsos(aband,1);
                    process_input(aband, qi->call, 0);
                    redraw_later();
                    break;
                case '1':
                    qrvdb->sort2 = qrvdb->sort;
                    qrvdb->sort = qrv_compare_call;
                    g_index_array_qsort(qrvdb->qrvs, qrvdb->sort);
                    redraw_later();
                    break;
                case '2':
                    qrvdb->sort2 = qrvdb->sort;
                    qrvdb->sort = qrv_compare_wwl;
                    g_index_array_qsort(qrvdb->qrvs, qrvdb->sort);
                    redraw_later();
                    break;
                case '3':
                    qrvdb->sort2 = qrvdb->sort;
                    qrvdb->sort = qrv_compare_wkd;
                    g_index_array_qsort(qrvdb->qrvs, qrvdb->sort);
                    redraw_later();
                    break;
                case '4':
                    qrvdb->sort2 = qrvdb->sort;
                    qrvdb->sort = qrv_compare_qrb;
                    g_index_array_qsort(qrvdb->qrvs, qrvdb->sort);
                    redraw_later();
                    break;
                case '5':
                    qrvdb->sort2 = qrvdb->sort;
                    qrvdb->sort = qrv_compare_qtf;
                    g_index_array_qsort(qrvdb->qrvs, qrvdb->sort);
                    redraw_later();
                    break;
                case '6':
                    qrvdb->sort2 = qrvdb->sort;
                    qrvdb->sort = qrv_compare_kst_time;
                    g_index_array_qsort(qrvdb->qrvs, qrvdb->sort);
                    redraw_later();
                    break;
                case '7':
                    qrvdb->sort2 = qrvdb->sort;
                    qrvdb->sort = qrv_compare_text;
                    g_index_array_qsort(qrvdb->qrvs, qrvdb->sort);
                    redraw_later();
                    break;
                case 'a':
                case 'A':
                    qrvdb->showall = !qrvdb->showall;
                    redraw_later();
                    break;
                case 'e':
                case 'E':
                    {
						cba_t cba;
						if (qrvdb->qrvs->len == 0) break;
						qi = (struct qrv_item*)g_index_array_index(qrvdb->qrvs, sw->cur);
						if (!qi) break;
                    
						SETCBA(cba, qrv_item, qi);
						edit_qrv(cba);
						break;
					}
                case 'u':
                case 'U':
                    if (qrvdb->qrvs->len == 0) break;
                    qi = (struct qrv_item*)g_index_array_index(qrvdb->qrvs, sw->cur);
                    if (!qi) break;
                    qi->bands_qrv |= 1<<bi;
                    redraw_later();
                    break;
                case '/':
                    strcpy(qrvdb->search, "/");
                    redraw_later();
                    break;
            }
    }
    return 0;
}

int sw_qrv_mouse_func(struct subwin *sw, struct event *ev, int fw){
    return 0;
}

void sw_qrv_draw_qrv(struct subwin *sw, int i, struct qrv_item *qi, int color){
    gchar *c;
    char degree=' ';
    struct tm utc;
    char kst_time_str[20];
    char qrb_str[20];
    char qtf_str[20];

#ifdef HAVE_SDL    
    //if (sdl) degree='';
    if (sdl) degree=0xb0;
#endif

    if (!qi || !qi->call){
        c = g_strdup("~");
    }else{
        int bi = toupper(aband->bandchar) - 'A';
        
        strcpy(qrb_str, "");
        if (qi->qrb >= 0) sprintf(qrb_str, "%5.0fkm", qi->qrb);

        strcpy(qtf_str, "");
        if (qi->qtf >= 0) sprintf(qtf_str, "%d%c", qi->qtf, degree);

        strcpy(kst_time_str, "");
        if (qi->kst_time != 0){
            gmtime_r(&qi->kst_time, &utc);
            sprintf(kst_time_str, "%2d:%02d", utc.tm_hour, utc.tm_min);
        }

        c = g_strdup_printf(" %-12s  %6s %4d %-7s  "
                            "%-4s  %5s  %s", 
                qi->call, qi->wwl, qi->wkd[bi], qrb_str, 
                qtf_str, kst_time_str, qi->text);
    }

    if (c && strlen(c)>sw->ho){
        print_text(sw->x, sw->y+i, sw->w, c + sw->ho, color);
    }
    g_free(c);
}


void sw_qrv_redraw(struct subwin *sw, int flags){
    struct qrv_item *qi;
    int i,bi, y, len;
    char *c;

    if (!aband) return;
    
    bi = toupper(aband->bandchar)-'A';
    fill_area(sw->x, sw->y+sw->h/2, sw->w, 1, COL_INV);
    clip_printf(sw, -sw->ho +  2, 0,  qrvdb->sort==qrv_compare_call?COL_WHITE:COL_NORM, "Callsign");
    clip_printf(sw, -sw->ho + 16, 0,  qrvdb->sort==qrv_compare_wwl?COL_WHITE:COL_NORM,  "WWL");
    clip_printf(sw, -sw->ho + 23, 0,  qrvdb->sort==qrv_compare_wkd?COL_WHITE:COL_NORM,   "WKD");
    clip_printf(sw, -sw->ho + 29, 0,  qrvdb->sort==qrv_compare_qrb?COL_WHITE:COL_NORM,  "QRB");
    clip_printf(sw, -sw->ho + 36, 0,  qrvdb->sort==qrv_compare_qtf?COL_WHITE:COL_NORM,  "QTF");
    clip_printf(sw, -sw->ho + 42, 0,  qrvdb->sort==qrv_compare_kst_time?COL_WHITE:COL_NORM, "AcKst");
    clip_printf(sw, -sw->ho + 50, 0,  qrvdb->sort==qrv_compare_text?COL_WHITE:COL_NORM, "Text");

    len = strlen(qrvdb->search);
    if (len > 0){
        clip_printf(sw, 0, sw->h - 1, COL_NORM, "%s", qrvdb->search);
        if (gses->focused) set_cursor(sw->x + len,  sw->y + sw->h - 1, sw->x + len,  sw->y + sw->h - 1); 
    }else{
        int x = 0;
        c = "1-6: sort, ";
        clip_printf(sw, x, sw->h - 1, COL_NORM, c); 
        x += strlen(c);
        c = "A: all";
        clip_printf(sw, x, sw->h - 1, qrvdb->showall?COL_WHITE:COL_NORM, c);
        x += strlen(c);
        c = ", Enter, Space, E: edit, Insert, Delete, U: undel, /: search";
        clip_printf(sw, x, sw->h - 1, COL_NORM, c);
        x += strlen(c);
    }

    if (!qrvdb->qrvs->len) return;
    sw_qrv_check_bounds(sw);

    qi = (struct qrv_item *)g_index_array_index(qrvdb->qrvs, sw->cur);
//    dbg("%s: cur=%d qi=%p\n", __FUNCTION__, sw->cur, qi);
    if (!qi || qrv_skip(qi, bi)) {
        print_text(sw->x, sw->y + sw->h/2, sw->w, "~", COL_INV);
        return;
    }
    sw_qrv_draw_qrv(sw, sw->h/2, qi, COL_INV);
    
    y = sw->h / 2 - 1;
    for(i = sw->cur - 1; i >= 0; i--){
        if (y <= 0) break;
        qi = (struct qrv_item *)g_index_array_index(qrvdb->qrvs, i);
        if (qrv_skip(qi, bi)) continue;
        sw_qrv_draw_qrv(sw, y--, qi, COL_NORM);
    } 

    y = sw->h / 2 + 1;
    for(i = sw->cur + 1; i < qrvdb->qrvs->len; i++){
        if (y >= sw->h - 1) break;
        qi = (struct qrv_item *)g_index_array_index(qrvdb->qrvs, i);
        if (qrv_skip(qi, bi)) continue;
        sw_qrv_draw_qrv(sw, y++, qi, COL_NORM);
    } 
}

void sw_qrv_check_bounds(struct subwin *sw){
    int len, bi;
    struct qrv_item *qi;

    if (!aband) return;

    len = qrvdb->qrvs->len;
    if (len == 0) return;

    bi = toupper(aband->bandchar) - 'A';

    if (sw->cur < 0) sw->cur = 0;
    if (sw->cur >= qrvdb->qrvs->len) sw->cur = qrvdb->qrvs->len - 1;;

    while (sw->cur < qrvdb->qrvs->len){
        qi = (struct qrv_item *)g_index_array_index(qrvdb->qrvs, sw->cur);
        if (!qrv_skip(qi, bi)) break;
        sw->cur++; 
    }
    if (sw->cur >= qrvdb->qrvs->len) sw->cur = qrvdb->qrvs->len - 1;;
             
    while (sw->cur >= 0){
        qi = (struct qrv_item *)g_index_array_index(qrvdb->qrvs, sw->cur);
        if (!qrv_skip(qi, bi)) break;
        sw->cur--; 
    }
    if (sw->cur < 0) sw->cur = 0;
}

void sw_qrv_raise(struct subwin *sw){
    sw_qrv_sort(qrvdb);
}
void sw_qrv_sort(struct qrvdb *qrvdb){
    g_index_array_qsort(qrvdb->qrvs, qrvdb->sort);
}
    
void qrv_compute_qrbqtf(struct qrv_item *q){
    double qtf, qtfrad;

    if (!q) return;
    if (!q->wwl || !*q->wwl){
        q->qrb = -1;
        q->qtf = -1;
        return;
    }
    /* CHANGE look at add_tmpqso_locator */    
    qrbqtf(ctest->pwwlo, q->wwl, &q->qrb, &qtf, NULL, 2);
    q->qtf = (int) (qtf+0.5);
    if (q->qrb < 0.1) {
        q->qrb = 0;
        q->qtf = 0;
    } 
    qtfrad=qtf*MY_PI/180.0;
    q->kx = (int)(  q->qrb*sin(qtfrad));
    q->ky = (int)(- q->qrb*cos(qtfrad));

    if (q->qrb < 600){
        q->weight = q->qrb;
    }else{
        q->weight= 1800 - 2 * q->qrb;
        if (q->weight < 0) q->weight = 0;
    }

    if (aband){
        int bi = toupper(aband->bandchar) - 'A';
        if (q->wkd[bi] > 1) q->weight *= q->wkd[bi];
        if (!(q->bands_qrv && (1 << bi))) q->weight = 0;
    }
    /*dbg("compute_qrbqtf(%s) %s->%s qrb=%f kx=%d ky=%d\n", 
            q->call, ctest->pwwlo, q->wwl, q->qrb, q->kx, q->ky); */
}


void qrv_recalc_qrbqtf(struct qrvdb *qrvdb){
    int i;

    for (i = 0; i < qrvdb->qrvs->len; i++){
        struct qrv_item *qi = (struct qrv_item *)g_index_array_index(qrvdb->qrvs, i);
        qrv_compute_qrbqtf(qi);
    }
}

void qrv_recalc_gst(struct qrvdb *qrvdb){
    int i, j;
    int qtf2;
	double max;

    if (!qrvdb) return;
                    
    qrvdb->beamwidth = 10;
    for (i=0;i<=180;i++){
        qrvdb->antchar[i]=cos(MY_PI/180.0*(double)i*120.0/qrvdb->beamwidth);
       /* dbg("i=%3d %5.3f\n", i, antchar[i]);*/
        if (qrvdb->antchar[i]<=0) break;
    }
    for (;i<=180;i++) qrvdb->antchar[i]=0;
    
    for (i=0; i<360; i++) qrvdb->gst[i]=0.0;

    for (i = 0; i < qrvdb->qrvs->len; i++){
        struct qrv_item *qi = (struct qrv_item *)g_index_array_index(qrvdb->qrvs, i);
        if (qi->qtf<=0 || qi->qtf>=360) continue;
        
        if (aband && qrv_skip(qi, aband->bandchar-'a')) continue;
        
        for (j=0;j<=180;j++){
            qtf2 = qi->qtf + j;
            if (qrvdb->antchar[j] == 0) break;
            if (qtf2<0) qtf2 += 360;
            qrvdb->gst[qtf2%360] += qi->weight * qrvdb->antchar[j];
        }
        for (j=1;j<180;j++){
            qtf2 = qi->qtf - j;
            if (qrvdb->antchar[j] == 0) break;
            if (qtf2<0) qtf2 += 360;
            qrvdb->gst[qtf2%360] += qi->weight * qrvdb->antchar[j];
        }
    }
    max = 0;
    for (i=0; i<360; i++) if (qrvdb->gst[i]>max) max = qrvdb->gst[i];
    if (!max) return;
    for (i=0; i<360; i++) qrvdb->gst[i] = qrvdb->gst[i] / max; 
} 


void qrv_set_wkd(struct qrvdb *qrvdb, struct qso *qso){
    char raw[20];
    struct qrv_item *qi;
	int bi;

    if (!qso) return;
    if (!qso->band) return;
            
    get_raw_call(raw, qso->callsign);
    qi = (struct qrv_item *)g_hash_table_lookup(qrvdb->hash, raw);

    bi = qso->band->bandchar - 'a';
    if (!qi) return;
    qi->bands_wkd |= 1 << bi;
}

void qrv_recalc_wkd(struct qrvdb *qrvdb){
    struct qrv_item *qi;
    struct band *b;
    struct qso *qso;
    int i, j;
    char raw[20];


    for (i = 0; i < qrvdb->qrvs->len; i++){
        struct qrv_item *qi = (struct qrv_item *)g_index_array_index(qrvdb->qrvs, i);
        qi->bands_wkd = 0;
    }

    if (!ctest) return;
    for (i=0; i<ctest->bands->len; i++){
        b = (struct band *)g_index_array_index(ctest->bands, i);
        for (j=0; j<b->qsos->len; j++){
            qso = (struct qso *)g_index_array_index(b->qsos, j);
            if (qso->dupe) continue;
            if (qso->error) continue;
            get_raw_call(raw, qso->callsign);
            qi = (struct qrv_item *)g_hash_table_lookup(qrvdb->hash, raw);
            //dbg("recalc_qrv_wkd(%s) qi=%p\n", raw, qi);
            if (!qi) continue;
            qi->bands_wkd |= 1 << (b->bandchar - 'a');
        }
    }


}

struct qrv_item *qrv_get(struct qrvdb *qrvdb, char *call){
    char raw[20];
	struct qrv_item *qi;

    get_raw_call(raw, call);
    qi = (struct qrv_item *)g_hash_table_lookup(qrvdb->hash, raw);
    return qi;
}

void qrv_read_line(char *str){
	char *c1=NULL, *c2=NULL;
    struct qrv_item *qi;
   
	if (regmatch(str, "^[0-9]{4}Z ([^ ]+)", &c1, &c2, NULL)==0){
//		log_addf("KST '%s'", c2);
    
        qi = qrv_get(qrvdb, c2);
        if (qi){
            qi->kst_time = time(NULL);
        }
        redraw_later();
    }

	if (c1) mem_free(c1);
	if (c2) mem_free(c2);
}


/* -------------- dialog edit qrv ------------------ */

char qrv_call[EQSO_LEN];
char qrv_wwl[EQSO_LEN];
char qrv_qrv[32];
char qrv_wkd[EQSO_LEN];
char qrv_text[MAX_STR_LEN];

void refresh_edit_qrv(cba_t cba){
    struct qrv_item *qi = (struct qrv_item*)GETCBA(cba, qrv_item);
    char *c;
    char raw[20];

    int bi = toupper(aband->bandchar) - 'A';
    
    if (!qi){
        int wkd[32];
        memset(wkd, 0, sizeof(wkd));
        qi = qrv_add(uc(qrv_call), "", 0, wkd, "", (time_t)0);
    }else{
        get_raw_call(raw, qi->call);
        g_hash_table_remove(qrvdb->hash, qi->call);
        CONDGFREE(qi->call);
        qi->call = g_strdup(qrv_call);
        uc(qi->call); 
        get_raw_call(raw, qi->call);
        g_hash_table_insert(qrvdb->hash, g_strdup(raw), qi); 
    }

    CONDGFREE(qi->wwl);
    qi->wwl = g_strdup(qrv_wwl);
    uc(qi->wwl);

    qi->bands_qrv = 0;
    for (c = qrv_qrv; *c != '\0'; c++){
        *c = toupper(*c);
        if (*c < 'A' || *c > 'Z') continue;
        qi->bands_qrv |= (1 << (*c - 'A'));
    }

    qi->wkd[bi] = atoi(qrv_wkd);
    
    CONDGFREE(qi->text);
    qi->text = g_strdup(qrv_text);

    qrv_compute_qrbqtf(qi);
    sw_qrv_sort(qrvdb);
}

void edit_qrv(cba_t cba){
    struct dialog *d;
    int i;
    struct qrv_item *qi;
    char x[2];


    if (!aband) return;
	qi = (struct qrv_item*)GETCBA(cba, qrv_item);
    if (qi){
		int bi;
        safe_strncpy0(qrv_call, qi->call, EQSO_LEN);
        safe_strncpy0(qrv_wwl, qi->wwl, EQSO_LEN);
        strcpy(qrv_qrv, "");
        x[1] = '\0';
        bi = toupper(aband->bandchar) - 'A';
        for (x[0] = 'A'; x[0] <= 'Z'; x[0]++){
            if (qi->bands_qrv & (1 << (x[0] - 'A'))) strcat(qrv_qrv, x);
        }
        g_snprintf(qrv_wkd, EQSO_LEN, "%d", qi->wkd[bi]);
        safe_strncpy0(qrv_text, qi->text, MAX_STR_LEN);
    }else{
        strcpy(qrv_call, "");
        strcpy(qrv_wwl, "");
        strcpy(qrv_qrv, "");
        if (aband) sprintf(qrv_qrv, "%c", toupper(aband->bandchar));
        strcpy(qrv_wkd, "");
        strcpy(qrv_text, "@OP");
    }
    
    if (!(d = mem_alloc(sizeof(struct dialog) + 20 * sizeof(struct dialog_item)))) return;
    memset(d, 0, sizeof(struct dialog) + 20 * sizeof(struct dialog_item));
    d->title = VTEXT(T_QRV); 
    d->fn = dlg_pf_fn;
    d->refresh = (void (*)(void *))refresh_edit_qrv;
    d->refresh_data = qi;
    d->y0 = 1;
    
    d->items[i=0].type = D_FIELD;
    d->items[i].dlen = EQSO_LEN;
    d->items[i].data = qrv_call;
    d->items[i].msg  = CTEXT(T_CALLSIGN2);
    d->items[i].wrap = 1;

    d->items[++i].type = D_FIELD;
    d->items[i].dlen = EQSO_LEN;
    d->items[i].data = qrv_wwl;
    d->items[i].msg  = CTEXT(T_WWL);
    d->items[i].wrap = 1;

    d->items[++i].type = D_FIELD;
    d->items[i].dlen = EQSO_LEN;
    d->items[i].data = qrv_qrv;
    d->items[i].msg  = CTEXT(T_QRV2);
    d->items[i].wrap = 1;

    d->items[++i].type = D_FIELD;
    d->items[i].dlen = EQSO_LEN;
    d->items[i].data = qrv_wkd;
    d->items[i].msg  = CTEXT(T_WKD);
    d->items[i].wrap = 1;

    d->items[++i].type = D_FIELD;
    d->items[i].dlen = MAX_STR_LEN;
    d->items[i].maxl = EQSO_LEN;
    d->items[i].data = qrv_text;
    d->items[i].msg  = CTEXT(T_REMARK);
    d->items[i].wrap = 2;


    d->items[++i].type = D_BUTTON;   
    d->items[i].gid = B_ENTER;
    d->items[i].fn = ok_dialog;
    d->items[i].text = VTEXT(T_OK);
    
    d->items[++i].type = D_BUTTON;
    d->items[i].gid = B_ESC;
    d->items[i].fn = cancel_dialog;
    d->items[i].text = VTEXT(T_CANCEL);
    d->items[i].align = AL_BUTTONS;
    d->items[i].wrap = 1;
    d->items[++i].type = D_END;
    do_dialog(d, getml(d, NULL));
    
}
