/*
 *  Copyright 1994-2016 Olivier Girondel
 *
 *  This file is part of lebiniou.
 *
 *  lebiniou is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation, either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  lebiniou is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with lebiniou. If not, see <http://www.gnu.org/licenses/>.
 */

#include "main.h"
#include "webcam.h"


#ifdef WITH_WEBCAM
extern int hflip, vflip, webcams;
extern char *video_base;
#endif


static void
usage()
{
  printf("Usage: %s [options]\n\n"
	 "\t-b, --basedir <path>\tSet base directory\n"
	 "\t-d, --datadir <path>\tSet data directory\n"
	 "\t-z, --schemes <file>\tSet the schemes file\n"
	 "\t-f, --fullscreen\tSet fullscreen\n"
	 "\t-h, --help\t\tDisplay this help\n"
	 "\t-i, --input <plugin>\tSet input plugin\n"
	 "\t-m, --maxfps <fps>\tSet maximum framerate\n"
	 "\t-o, --output <plugin>\tSet output plugin\n"
	 "\t-r, --random <mode>\tSet auto-random mode\n"
	 "\t-s, --screen <mode>\tSet On Screen Display mode\n"
	 "\t-v, --version\t\tDisplay the version and exit\n"
	 "\t-q, --quiet\t\tSuppress messages\n"
#ifndef FIXED
	 "\t-x, --width <width>\tSet width\n"
	 "\t-y, --height <height>\tSet height\n"
#endif
	 "\t-p, --pidfile <file>\tSet the PID file\n"
	 "\t-t, --themes <themes>\tComma-separated list of themes to use\n"
#ifdef WITH_WEBCAM
	 "\t    --webcams <webcams>\tNumber of webcams (default: %d)\n"
	 "\t    --video <device>\tWebcam base (default: %s)\n"
	 "\t-c, --camflip <h|v>\tFlip webcam image horizontally/vertically\n"
#endif
	 "\n"
	 "This version of %s was compiled with:\n"
	 "\tInput plugins: %s [default: %s]\n"
	 "\tOutput plugins: %s [default: %s]\n"
	 "\n",
	 PACKAGE_NAME,
#ifdef WITH_WEBCAM
	 webcams, DEFAULT_VIDEO_DEVICE,
#endif
	 PACKAGE_STRING,
	 INPUT_PLUGINS, DEFAULT_INPUT_PLUGIN,
	 OUTPUT_PLUGINS, DEFAULT_OUTPUT_PLUGIN);

  /* TODO print default values for all options */
  exit(0);
}


void
getargs(int argc, char **argv)
{
  int ch;
#ifndef FIXED
  int w, h;
#endif

#if HAVE_GETOPT_LONG
  static struct option long_opt[] =
	  {
		  {"basedir", required_argument, NULL, 'b'},
		  {"datadir", required_argument, NULL, 'd'},
		  {"schemes", required_argument, NULL, 'z'},
		  {"fullscreen", no_argument, NULL, 'f'},
		  {"help", no_argument, NULL, 'h'},
		  {"input", required_argument, NULL, 'i'},
		  {"maxfps", required_argument, NULL, 'm'},
		  {"output", required_argument, NULL, 'o'},
		  {"random", required_argument, NULL, 'r'},
		  {"screen", required_argument, NULL, 's'},
		  {"quiet", no_argument, NULL, 'q'},
#ifndef FIXED
		  {"width", required_argument, NULL, 'x'},
		  {"height", required_argument, NULL, 'y'},
#endif
		  {"version", no_argument, NULL, 'v'},
		  {"pidfile", required_argument, NULL, 'p'},
		  {"themes", required_argument, NULL, 't'},
#ifdef WITH_WEBCAM
		  {"camflip", required_argument, NULL, 'c'},
		  {"webcams", required_argument, NULL, 'W'},
		  {"device", required_argument, NULL, 'D'},
#endif
		  {0, 0, 0, 0}
	  };

  
  /* Get command line arguments */
  while ((ch = getopt_long(argc, argv, "b:c:d:fhi:m:o:p:qr:s:t:x:y:v", long_opt, NULL)) != -1)
#else
  while ((ch = getopt(argc, argv, "b:c:d:fhi:m:o:p:qr:s:t:x:y:v")) != -1)
#endif
    switch (ch) {
    case 'b':
      if (base_dir == NULL)
	base_dir = optarg;
      break;

#ifdef WITH_WEBCAM
    case 'c':
      if (*optarg == 'h') hflip=!hflip;
      if (*optarg == 'v') vflip=!vflip;
      break;
#endif

    case 'd':
      if (data_dir == NULL)
	data_dir = optarg;
      break;
				
    case 'z':
      if (schemes_file == NULL)
	schemes_file = optarg;
      break;
				
    case 'f':
      fullscreen = 1;
      break;
			
    case 'h':
      usage();
      break;

    case 'i':
      if (NULL != input_plugin)
	xfree(input_plugin);
      input_plugin = strdup(optarg);
      VERBOSE(printf("[c] Setting input plugin: %s\n", input_plugin));
      break;

    case 'o':
      if (NULL != output_plugin)
	xfree(output_plugin);
      output_plugin = strdup(optarg);
      VERBOSE(printf("[c] Setting output plugin: %s\n", output_plugin));
      break;

    case 'm':
      max_fps = xatol(optarg);
      if (max_fps > 0) {
	VERBOSE(printf("[c] Maximum fps set to %li\n", max_fps));
      } else
	xerror("Invalid max_fps (%li)\n", max_fps);
      break;

    case 'r':
      random_mode = (enum RandomMode)xatol(optarg);
      if (random_mode > BR_BOTH)
	xerror("Invalid random_mode (%d)\n", random_mode);
      else {
	VERBOSE(printf("[c] Random mode set to %d\n", random_mode));
      }
      break;

    case 's':
      osd_mode = (enum OSDMode)xatol(optarg);
      if (osd_mode > OSD_FULL_BG)
	xerror("Invalid osd_mode (%d)\n", osd_mode);
      else {
	VERBOSE(printf("[c] OSD mode set to %d\n", osd_mode));
      }
      break;

    case 't':
      if (NULL != themes)
	xfree(themes);
      themes = strdup(optarg);
      VERBOSE(printf("[c] Using themes: %s\n", themes));
      break;

    case 'v':
      printf("Le Biniou version %s\n", PACKAGE_VERSION);
      exit(0);
      break;

    case 'q':
      libbiniou_verbose = 0;
      break;

    case 'x':
#ifndef FIXED
      w = xatol(optarg);
      if (w > 0) {
	width = w;
	VERBOSE(printf("[c] Width set to %i\n", width));
      } else
	xerror("Invalid width: %s\n", w);
#else
      VERBOSE(fprintf(stderr, "[!] Compiled with fixed buffers, ignoring width= %li\n", xatol(optarg)));
#endif
      break;

    case 'y':
#ifndef FIXED
      h = xatol(optarg);
      if (h > 0) {
	height = h;
	VERBOSE(printf("[c] Height set to %i\n", height));
      } else
	xerror("Invalid height: %s\n", h);
#else
      VERBOSE(fprintf(stderr, "[!] Compiled with fixed buffers, ignoring height= %li\n", xatol(optarg)));
#endif
      break;
		  
    case 'p':
      if (pid_file == NULL)
	pid_file = optarg;
      break;

#ifdef WITH_WEBCAM
    case 'W': /* webcams */
      webcams = xatol(optarg);
      if ((webcams >= 0) && (webcams <= MAX_CAMS)) {
	VERBOSE(printf("[c] webcam: grabbing %d device%s\n", webcams, (webcams == 1 ? "": "s")));
      } else
	webcams = 1;
      break;

    case 'D': /* video_base */
      if (NULL != video_base)
	xfree(video_base);
      video_base = strdup(optarg);
      VERBOSE(printf("[c] webcam: first device is %s\n", video_base));
      break;
#endif

    default:
      usage();
      break;
    }

  if (NULL == base_dir)
    base_dir = DEFAULT_PLUGINSDIR;

  if (NULL == data_dir)
    data_dir = DEFAULT_DATADIR;
  
  if (NULL == schemes_file)
    schemes_file = DEFAULT_SCHEMES_FILE;
  
  if (NULL == input_plugin)
    input_plugin = strdup(DEFAULT_INPUT_PLUGIN);
  else if (!strcmp(input_plugin, "NULL"))
    input_plugin = NULL;
	
  if (NULL == output_plugin)
    output_plugin = strdup(DEFAULT_OUTPUT_PLUGIN);
  else if (!strcmp(output_plugin, "NULL"))
    output_plugin = NULL;

  if (NULL == themes)
    themes = strdup("biniou");

#ifdef WITH_WEBCAM
  if (NULL == video_base)
    video_base = strdup(DEFAULT_VIDEO_DEVICE);
#endif
} 
