/**
 * Copyright (c) Members of the EGEE Collaboration. 2004-2010.
 * See http://www.eu-egee.org/partners/ for details on the copyright
 * holders.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 *
 *  Authors:
 *  2009-
 *     Oscar Koeroo <okoeroo@nikhef.nl>
 *     Mischa Sall\'e <msalle@nikhef.nl>
 *     David Groep <davidg@nikhef.nl>
 *     NIKHEF Amsterdam, the Netherlands
 *     <grid-mw-security@nikhef.nl>
 *
 *  2007-2009
 *     Oscar Koeroo <okoeroo@nikhef.nl>
 *     David Groep <davidg@nikhef.nl>
 *     NIKHEF Amsterdam, the Netherlands
 *
 *  2003-2007
 *     Martijn Steenbakkers <martijn@nikhef.nl>
 *     Gerben Venekamp <venekamp@nikhef.nl>
 *     Oscar Koeroo <okoeroo@nikhef.nl>
 *     David Groep <davidg@nikhef.nl>
 *     NIKHEF Amsterdam, the Netherlands
 *
 */


/*!
    \file   lcmaps_verify_account_from_pem.c
    \brief  This interface returns the uid, gids and poolindex, i.e. leaseid,
            using LCMAPS and takes the user credential in PEM format as input
    \author Martijn Steenbakkers for the EU DataGrid.

    This interface returns the uid, gids and poolindex, i.e. leaseid, using LCMAPS.
    As input it requires the user credential in PEM format
    -# lcmaps_verify_account_from_pem: interface function
*/


/******************************************************************************
                             Include header files
******************************************************************************/
#include "lcmaps_config.h"

/* Needed for NULL */
#include <stdio.h>
#include <stdlib.h>
#include <stdarg.h>

/* LCMAPS includes */
#include "lcmaps.h"
#include "lcmaps_log.h"
#include "lcmaps_verify_account_from_pem.h"

#include "pluginmanager/_lcmaps_log.h"
#include "pluginmanager/_lcmaps_utils.h"

/******************************************************************************
                             Module specific defines
******************************************************************************/
/* Default settings */
#define LCMAPS_MAX_POLICIES 10

/******************************************************************************
                          Module specific prototypes
******************************************************************************/
int lcmaps_verify_account_from_pem_va (int narg, ...);



/******************************************************************************
Function:   lcmaps_verify_account_from_pem_va
Description:
    Verify the account mapping from the supplied user credential.

Parameters:
    narg:  the number of arguments that follow
    ap:    vararg list, that consists of
        - input:  The PEM-encoded string containing the user proxy
        - input:  a structure that contains account information:
                  uid, gids (primary and secondary) and the poolindex
Returns:
    0: success
    1: failure
******************************************************************************/
int lcmaps_verify_account_from_pem_va(
    int        narg,
    ...
)
{
    va_list                             ap;
    char *                              pem_string = NULL;        /* input variable */
    lcmaps_account_info_t               lcmaps_account;           /* input variable */
    char *                              lcmaps_policy_string = NULL;
    char *                              lcmaps_policies[LCMAPS_MAX_POLICIES];
    int                                 lcmaps_npols = LCMAPS_MAX_POLICIES;
    int                                 rc = -1;
    int                                 jj = 0;

    /* First initialize LCMAPS, initialize without file pointer or name which
     * will result in getting env var LCMAPS_LOG_FILE. Specify DO_USRLOG to try
     * that first, if that fails we will go automatically to syslog */
    if (lcmaps_init_and_logfile(NULL, NULL, (unsigned short)(DO_USRLOG)))
    {
        lcmaps_log(LOG_ERR, "%s: LCMAPS initialization failure\n", __func__);
        goto verify_account_from_pem_error;
    }


    /* Parse arguments from va_list */
    va_start(ap, narg);
    if (narg == 2)
    {
        pem_string = va_arg(ap, char *);
        lcmaps_account = va_arg(ap, lcmaps_account_info_t);
    }
    else
    {
        lcmaps_log(LOG_ERR, "%s: The number of arguments (%d) should be 2\n", __func__, narg);
        return 1;
    }
    va_end(ap);

    /* Parse policy environment variable */
    for (jj=0; jj < LCMAPS_MAX_POLICIES; jj++) lcmaps_policies[jj] = 0;
    lcmaps_policy_string = getenv("LCMAPS_POLICY_NAME");
    if ((rc = lcmaps_tokenize(lcmaps_policy_string, lcmaps_policies, &lcmaps_npols, ":")) != 0)
    {
        lcmaps_log(LOG_ERR, "%s: Cannot parse LCMAPS_POLICY_NAME environment variable, because\n", __func__);
        switch (rc)
        {
            case -1:
                lcmaps_log(LOG_ERR, "%s: of a malloc error\n", __func__);
                break;
            case -2:
                lcmaps_log(LOG_ERR, "%s: the policy list is too large (max = %d)\n",
                    __func__, LCMAPS_MAX_POLICIES);
                break;
            case -3:
                lcmaps_log(LOG_ERR, "%s: of a non-matching quote\n", __func__);
                break;
            case -4:
                lcmaps_log(LOG_ERR, "%s: invalid input\n", __func__);
                break;
            default:
                lcmaps_log(LOG_ERR, "%s: of an unknown error\n", __func__);
                break;
        }
        goto verify_account_from_pem_error;
    }

    /*
     * Now that we have the credential let's run (and terminate) LCMAPS !
     */
    rc = lcmaps_run_and_verify_account_from_pem(
        NULL,
        pem_string,
        lcmaps_account.uid,
        lcmaps_account.pgid_list,
        lcmaps_account.npgid,
        lcmaps_account.sgid_list,
        lcmaps_account.nsgid,
        lcmaps_account.poolindex,
        NULL,
        lcmaps_npols,
        lcmaps_policies
    );
    if (rc != 0)
    {
        lcmaps_log(LOG_NOTICE, "LCMAPS failed to verify the account mapping\n");
        if (lcmaps_term())
        {
            lcmaps_log(LOG_ERR, "LCMAPS termination failure\n");
            goto verify_account_from_pem_error;
        }
        goto verify_account_from_pem_error;
    }

    rc = lcmaps_term();
    if (rc)
    {
        lcmaps_log(LOG_ERR, "LCMAPS termination failure\n");
        goto verify_account_from_pem_error;
    }


    /* Clean policies */
    for (jj=0; jj < LCMAPS_MAX_POLICIES; jj++)
    {
        if ((lcmaps_policies[jj]) != NULL)
        {
            free(lcmaps_policies[jj]);
            lcmaps_policies[jj] = NULL;
        }
    }

    return 0;

 verify_account_from_pem_error:
    /* Clean policies */
    for (jj=0; jj < LCMAPS_MAX_POLICIES; jj++)
    {
        if ((lcmaps_policies[jj]) != NULL)
        {
            free(lcmaps_policies[jj]);
            lcmaps_policies[jj] = NULL;
        }
    }
    return 1;
}

/******************************************************************************
Function:   lcmaps_verify_account_from_pem
Description:
    Verify the account mapping from the supplied user credential.

Parameters:
    pem_string:    :  The PEM-encoded string containing the user proxy
    lcmaps_account :  A structure that contains account information: (input)
                      uid, gids (primary and secondary).
                      Please use lcmaps_account_info_clean() to clean this
                      structure after use.
Returns:
    0: success
    1: failure
******************************************************************************/
int lcmaps_verify_account_from_pem(
        char * pem_string,
        lcmaps_account_info_t lcmaps_account
)
{
    return (lcmaps_verify_account_from_pem_va(2, pem_string, lcmaps_account));
}

/******************************************************************************
CVS Information:
    $Source: /srv/home/dennisvd/svn/mw-security/lcmaps/src/lcmaps_verify_account_from_pem.c,v $
    $Date: 2014-03-04 16:17:33 +0100 (Tue, 04 Mar 2014) $
    $Revision: 17597 $
    $Author: msalle $
******************************************************************************/
