// Licensed to the .NET Foundation under one or more agreements.
// The .NET Foundation licenses this file to you under the MIT license.

using System.Collections.Generic;
using System.IO;
using System.Threading;
using System.Threading.Tasks;
using Xunit;

namespace System.Security.Cryptography.Tests
{
    public class Sha3_256Tests : HashAlgorithmTestDriver<Sha3_256Tests.Traits>
    {
        public sealed class Traits : IHashTrait
        {
            public static bool IsSupported => SHA3_256.IsSupported;
            public static int HashSizeInBytes => SHA3_256.HashSizeInBytes;
            public static HashAlgorithm Create() => SHA3_256.Create();
        }

        protected override HashAlgorithmName HashAlgorithm => HashAlgorithmName.SHA3_256;

        protected override bool TryHashData(ReadOnlySpan<byte> source, Span<byte> destination, out int bytesWritten)
        {
            return SHA3_256.TryHashData(source, destination, out bytesWritten);
        }

        protected override byte[] HashData(byte[] source) => SHA3_256.HashData(source);

        protected override byte[] HashData(ReadOnlySpan<byte> source) => SHA3_256.HashData(source);

        protected override int HashData(ReadOnlySpan<byte> source, Span<byte> destination) =>
            SHA3_256.HashData(source, destination);

        protected override int HashData(Stream source, Span<byte> destination) =>
            SHA3_256.HashData(source, destination);

        protected override byte[] HashData(Stream source) => SHA3_256.HashData(source);

        protected override ValueTask<int> HashDataAsync(Stream source, Memory<byte> destination, CancellationToken cancellationToken) =>
            SHA3_256.HashDataAsync(source, destination, cancellationToken);

        protected override ValueTask<byte[]> HashDataAsync(Stream source, CancellationToken cancellationToken) =>
            SHA3_256.HashDataAsync(source, cancellationToken);

        [ConditionalFact(nameof(IsSupported))]
        public void SHA3_256_Kats()
        {
            foreach ((string Msg, string MD) kat in Fips202Kats)
            {
                Verify(Convert.FromHexString(kat.Msg), kat.MD);
            }
        }

        [ConditionalFact(nameof(IsSupported))]
        public void SHA3_256_Empty_Stream()
        {
            VerifyRepeating("", 0, "a7ffc6f8bf1ed76651c14756a061d662f580ff4de43b49fa82d80a4b80f8434a");
        }

        [ConditionalFact(nameof(IsSupported))]
        public async Task SHA3_256_Empty_Stream_Async()
        {
            await VerifyRepeatingAsync("", 0, "a7ffc6f8bf1ed76651c14756a061d662f580ff4de43b49fa82d80a4b80f8434a");
        }

        [ConditionalFact(nameof(IsSupported))]
        public void SHA3_256_VerifyLargeStream_MultipleOf4096()
        {
            // Verfied with:
            // for _ in {1..1024}; do echo -n "0102030405060708"; done | openssl dgst -sha3-256
            VerifyRepeating(
                "0102030405060708",
                1024,
                "5e80dd4330d9124adce40a043f166d7e0f6853050fd99919c7b1436ee0a538e9");
        }

        [ConditionalFact(nameof(IsSupported))]
        public void SHA3_256_VerifyLargeStream_NotMultipleOf4096()
        {
            // Verfied with:
            // for _ in {1..1025}; do echo -n "0102030405060708"; done | openssl dgst -sha3-256
            VerifyRepeating(
                "0102030405060708",
                1025,
                "5dbbd15ba5745412a79835cc4bec1bede925da06eca7a5bbf50c38a6ec1c49bc");
        }

        [ConditionalFact(nameof(IsSupported))]
        public async Task SHA3_256_VerifyLargeStream_NotMultipleOf4096_Async()
        {
            // Verfied with:
            // for _ in {1..1025}; do echo -n "0102030405060708"; done | openssl dgst -sha3-256
            await VerifyRepeatingAsync(
                "0102030405060708",
                1025,
                "5dbbd15ba5745412a79835cc4bec1bede925da06eca7a5bbf50c38a6ec1c49bc");
        }

        [ConditionalFact(nameof(IsSupported))]
        public async Task SHA3_256_VerifyLargeStream_MultipleOf4096_Async()
        {
            // Verfied with:
            // for _ in {1..1024}; do echo -n "0102030405060708"; done | openssl dgst -sha3-256
            await VerifyRepeatingAsync(
                "0102030405060708",
                1024,
                "5e80dd4330d9124adce40a043f166d7e0f6853050fd99919c7b1436ee0a538e9");
        }

        [Fact]
        public void SHA3_256_HashSizes()
        {
            Assert.Equal(256, SHA3_256.HashSizeInBits);
            Assert.Equal(32, SHA3_256.HashSizeInBytes);
        }

        [Fact]
        public void SHA3_256_IsSupported_AgreesWithPlatformVersion()
        {
            Assert.Equal(PlatformDetection.SupportsSha3, SHA3_256.IsSupported);
        }

        private static IEnumerable<(string Msg, string MD)> Fips202Kats
        {
            get
            {
                // Generated from SHA3_256ShortMsg.rsp
                yield return (Msg: "", MD: "a7ffc6f8bf1ed76651c14756a061d662f580ff4de43b49fa82d80a4b80f8434a");
                yield return (Msg: "e9", MD: "f0d04dd1e6cfc29a4460d521796852f25d9ef8d28b44ee91ff5b759d72c1e6d6");
                yield return (Msg: "d477", MD: "94279e8f5ccdf6e17f292b59698ab4e614dfe696a46c46da78305fc6a3146ab7");
                yield return (Msg: "b053fa", MD: "9d0ff086cd0ec06a682c51c094dc73abdc492004292344bd41b82a60498ccfdb");
                yield return (Msg: "e7372105", MD: "3a42b68ab079f28c4ca3c752296f279006c4fe78b1eb79d989777f051e4046ae");
                yield return (Msg: "0296f2c40a", MD: "53a018937221081d09ed0497377e32a1fa724025dfdc1871fa503d545df4b40d");
                yield return (Msg: "e6fd42037f80", MD: "2294f8d3834f24aa9037c431f8c233a66a57b23fa3de10530bbb6911f6e1850f");
                yield return (Msg: "37b442385e0538", MD: "cfa55031e716bbd7a83f2157513099e229a88891bb899d9ccd317191819998f8");
                yield return (Msg: "8bca931c8a132d2f", MD: "dbb8be5dec1d715bd117b24566dc3f24f2cc0c799795d0638d9537481ef1e03e");
                yield return (Msg: "fb8dfa3a132f9813ac", MD: "fd09b3501888445ffc8c3bb95d106440ceee469415fce1474743273094306e2e");
                yield return (Msg: "71fbacdbf8541779c24a", MD: "cc4e5a216b01f987f24ab9cad5eb196e89d32ed4aac85acb727e18e40ceef00e");
                yield return (Msg: "7e8f1fd1882e4a7c49e674", MD: "79bef78c78aa71e11a3375394c2562037cd0f82a033b48a6cc932cc43358fd9e");
                yield return (Msg: "5c56a6b18c39e66e1b7a993a", MD: "b697556cb30d6df448ee38b973cb6942559de4c2567b1556240188c55ec0841c");
                yield return (Msg: "9c76ca5b6f8d1212d8e6896ad8", MD: "69dfc3a25865f3535f18b4a7bd9c0c69d78455f1fc1f4bf4e29fc82bf32818ec");
                yield return (Msg: "687ff7485b7eb51fe208f6ff9a1b", MD: "fe7e68ae3e1a91944e4d1d2146d9360e5333c099a256f3711edc372bc6eeb226");
                yield return (Msg: "4149f41be1d265e668c536b85dde41", MD: "229a7702448c640f55dafed08a52aa0b1139657ba9fc4c5eb8587e174ecd9b92");
                yield return (Msg: "d83c721ee51b060c5a41438a8221e040", MD: "b87d9e4722edd3918729ded9a6d03af8256998ee088a1ae662ef4bcaff142a96");
                yield return (Msg: "266e8cbd3e73d80df2a49cfdaf0dc39cd1", MD: "6c2de3c95900a1bcec6bd4ca780056af4acf3aa36ee640474b6e870187f59361");
                yield return (Msg: "a1d7ce5104eb25d6131bb8f66e1fb13f3523", MD: "ee9062f39720b821b88be5e64621d7e0ca026a9fe7248d78150b14bdbaa40bed");
                yield return (Msg: "d751ccd2cd65f27db539176920a70057a08a6b", MD: "7aaca80dbeb8dc3677d18b84795985463650d72f2543e0ec709c9e70b8cd7b79");
                yield return (Msg: "b32dec58865ab74614ea982efb93c08d9acb1bb0", MD: "6a12e535dbfddab6d374058d92338e760b1a211451a6c09be9b61ee22f3bb467");
                yield return (Msg: "4e0cc4f5c6dcf0e2efca1f9f129372e2dcbca57ea6", MD: "d2b7717864e9438dd02a4f8bb0203b77e2d3cd8f8ffcf9dc684e63de5ef39f0d");
                yield return (Msg: "d16d978dfbaecf2c8a04090f6eebdb421a5a711137a6", MD: "7f497913318defdc60c924b3704b65ada7ca3ba203f23fb918c6fb03d4b0c0da");
                yield return (Msg: "47249c7cb85d8f0242ab240efd164b9c8b0bd3104bba3b", MD: "435e276f06ae73aa5d5d6018f58e0f009be351eada47b677c2f7c06455f384e7");
                yield return (Msg: "cf549a383c0ac31eae870c40867eeb94fa1b6f3cac4473f2", MD: "cdfd1afa793e48fd0ee5b34dfc53fbcee43e9d2ac21515e4746475453ab3831f");
                yield return (Msg: "9b3fdf8d448680840d6284f2997d3af55ffd85f6f4b33d7f8d", MD: "25005d10e84ff97c74a589013be42fb37f68db64bdfc7626efc0dd628077493a");
                yield return (Msg: "6b22fe94be2d0b2528d9847e127eb6c7d6967e7ec8b9660e77cc", MD: "157a52b0477639b3bc179667b35c1cdfbb3eef845e4486f0f84a526e940b518c");
                yield return (Msg: "d8decafdad377904a2789551135e782e302aed8450a42cfb89600c", MD: "3ddecf5bba51643cd77ebde2141c8545f862067b209990d4cb65bfa65f4fa0c0");
                yield return (Msg: "938fe6afdbf14d1229e03576e532f078898769e20620ae2164f5abfa", MD: "9511abd13c756772b852114578ef9b96f9dc7d0f2b8dcde6ea7d1bd14c518890");
                yield return (Msg: "66eb5e7396f5b451a02f39699da4dbc50538fb10678ec39a5e28baa3c0", MD: "540acf81810a199996a612e885781308802fe460e9c638cc022e17076be8597a");
                yield return (Msg: "de98968c8bd9408bd562ac6efbca2b10f5769aacaa01365763e1b2ce8048", MD: "6b2f2547781449d4fa158180a178ef68d7056121bf8a2f2f49891afc24978521");
                yield return (Msg: "94464e8fafd82f630e6aab9aa339d981db0a372dc5c1efb177305995ae2dc0", MD: "ea7952ad759653cd47a18004ac2dbb9cf4a1e7bba8a530cf070570c711a634ea");
                yield return (Msg: "c178ce0f720a6d73c6cf1caa905ee724d5ba941c2e2628136e3aad7d853733ba", MD: "64537b87892835ff0963ef9ad5145ab4cfce5d303a0cb0415b3b03f9d16e7d6b");
                yield return (Msg: "6ef70a3a21f9f7dc41c553c9b7ef70db82ca6994ac89b3627da4f521f07e1ae263", MD: "0afe03b175a1c9489663d8a6f66d1b24aba5139b996400b8bd3d0e1a79580e4d");
                yield return (Msg: "0c4a931ff7eace5ea7cd8d2a6761940838f30e43c5d1253299abd1bd903fed1e8b36", MD: "dc5bebe05c499496a7ebfe04309cae515e3ea57c5d2a5fe2e6801243dd52c93b");
                yield return (Msg: "210f7b00bf8b4337b42450c721c3f781256359d208733846b97c0a4b7b044c38dbb219", MD: "3305c9d28e05288a2d13994d64c88d3506399cd62b2b544213cf3539a8e92e2e");
                yield return (Msg: "3cb8992759e2dc60ebb022bd8ee27f0f98039e6a9fe360373b48c7850ce113a0ff7b2ae5", MD: "3c00bf3e12ade9d2de2756506f809f147c8d6adc22e7bb666e0b1d26469e65a5");
                yield return (Msg: "22634f6ba7b4fccaa3ba4040b664dbe5a72bf394fb534e49c76ec4cdc223f4969e2d37e899", MD: "a87e5c78837d7be0060d8f5eda975489ec961b28d7088f42a70f92414ae17793");
                yield return (Msg: "6e1dcd796b2015ee6760f98fdb40e668b2cf38b05c91f6a91e83bcc8ac59f816f90a59d64e8e", MD: "746bf845c08aa186b5fe1ca35528232c4a491a3a2a32cd23e990bc603f3268ae");
                yield return (Msg: "ee0be20320f9d44073281265a6e9fa6b9d252495624b8d016b8ef57e1b4e859d8ad3b50b89416d", MD: "a3257baf14ca16e1137dc5158703f3b02ebc74fc7677165fe86d4be1f38e2f7c");
                yield return (Msg: "8ae2da242635b6568289bf6bec8a438dbac1f5b4d50a90bb7449bdb92a59378e23452dbcabbbe879", MD: "e25c44802c5cf2e9f633e683d37aa8c8db8a0e21c367808121d14d96c8a400b5");
                yield return (Msg: "bdd0252dec5b798ef20e51791a18e8ca234d9bfde632a9e5395337a112dd97cdf068c9f57615424f59", MD: "e02c1b197979c44a5a50d05ea4882c16d8205c2e3344265f8fe0e80aed06c065");
                yield return (Msg: "c4c7b6315cb60b0e6cd01ef0b65f6486fdae4b94c6be21465c3a31c416ad2f06dcf3d6eae8eecf84ca7a", MD: "2da21867cd6b5402d3caff92a05fddfca90199fd51a94a066af164ce3d36c949");
                yield return (Msg: "b17977aced3a1184b14b0e41a04dd8b513c925ca19211e1abdc6c1b987ac845545fb3b820a083b4f7883c0", MD: "f91b016d013ede8d6a2e1efd4c0dd99417da8b0222d787867ca02b0ea2e80e45");
                yield return (Msg: "f65c3aa1d9981a84e49fc86d938f3f756f60e3858d5e1f6957dd4d268e28d68e90ba9a11d7b192d6c37fb30b", MD: "3acbebf8eda9d3c99a6b6b666366c391e8200d55fd33ad8680734def1dc7ae85");
                yield return (Msg: "49abba1fa98f3c4470d5dd4ed36924af4a7ad62f4c2dd13e599238883ed7d0cb95bbaae58b460332e6b7681446", MD: "02bcd9ea4f1aa5276f38e30351a14a072bc5d53a52d04d559a65ca46f1bcb56e");
                yield return (Msg: "275645b5a2514fe65a82efac57e406f224e0259677674f1d133f00a5ee9a6d1a8fed0eadbbff5a825041d2a9715d", MD: "c70a874d786cd0f3f09fa4dc1bb8f551d45f26d77ad63de1a9fdfb3b7c09c041");
                yield return (Msg: "cd02b32107b9a640fc1bf439ac81a5c27d037c6076e1cfe6ad229638037ac1550e71cf9557c29c2fc6017afd5a8184", MD: "36c73d11d450784eb99af068cd4e1cbc5768c8a2118010aceec6d852dda80d95");
                yield return (Msg: "5a72e0e1aec82a6541f04883bb463b0c39c22b59431cfb8bfd332117a1afb5832ce5c76a58fcf6c6cb4e3e6f8e1112de", MD: "90fc3193552ec71d3315ebbb807913afd4cd2f0833a65e40d011d64de5e66513");
                yield return (Msg: "43402165911890719f9179f883bbbc2a3be77682e60dd24b356a22621c6d2e3dcdd4cb2ce613b0dfe9f58629ee853e0394", MD: "5c4b6ceac9441defa99b10b805a725d4018b74b3e1f24ad8934fc89b41b8fd9e");
                yield return (Msg: "fc56ca9a93982a4669ccaba6e3d184a19de4ce800bb643a360c14572aedb22974f0c966b859d91ad5d713b7ad99935794d22", MD: "e21806ce766bbce8b8d1b99bcf162fd154f54692351aec8e6914e1a694bda9ee");
                yield return (Msg: "ace6297e50d50a11388118efc88ef97209b11e9dfcb7ad482fc9bf7d8deecc237ad163d920c51f250306d6cedc411386a457c7", MD: "f5581403a082bbf5ad7e09bdfccc43bf9683ebc88291d71d9ce885a37e952bd6");
                yield return (Msg: "3bad18046e9424de24e12944cd992cfba4556f0b2ae88b7bd342be5cff9586092bb66fac69c529040d10dd66aa35c1023d87eb68", MD: "faed76ff5a1cd99183b311e502c54e516d70a87050cf8961c8cd46f65c1358cd");
                yield return (Msg: "e564c9a1f1aaf8545a259f52c3fd1821ed03c22fd7424a0b2ad629d5d3026ef4f27cbe06f30b991dfa54de2885f192af4dc4ddc46d", MD: "811529c600c9d780f796a29a6b3e89f8a12b3f29c36f72b06cca7edc36f48dc0");
                yield return (Msg: "6043fa6465d69cab45520af5f0fd46c81dbf677531799802629863681cea30ffa3b00836fbf49f87051d92aaeac0ed09bcb9f0755b7b", MD: "b0fceecdaef6c76d5fc3835b523ce2416f4a9b9bd1f90234445df0f2b689f2f5");
                yield return (Msg: "2040c538c79237e6f2b8188c6375ec2f610ac2301607b9c23660c3a1e1c3a902cb2950c59aac3af28f984f6369c4debe8623dfa74c967b", MD: "e33dbdc0acc23fcfad3c759c4333410bd3a40efb1366ade157d2c81d65a0a6c7");
                yield return (Msg: "00ff6c96b7aa3cf27d036cf20af7031434113252574bda9cf9244d85aef2593d3a7a83bff6be904b75164a1766828042bc3f4f090d98a03d", MD: "d000eafca34815783bed9b050c6901c97f2e77d4771a0ed724dd8f6ff1448791");
                yield return (Msg: "e8df14936cce118139e690f1662f88cfbc9c333b6dea658c02cb1d959644592842542fd9d8d61a04d4a892128f0ddff7b6502efffbabe5cb0a", MD: "3479a9617a3adca35854c08fe987c2fe7ff2b01b04f2d952c107b3f066420551");
                yield return (Msg: "4ed981a31f70dd6b70c161be1f01fc1bba54d06d9494e7eb194e213d5e0e71e0fddd49cb1f075353da22624cbe4ba871aab32906e45b6fbb691b", MD: "9c824a00e068d2fda73f9c2e7798e8d9394f57f94df0edeb132e78e8a379a0cf");
                yield return (Msg: "7802b70c6158bc26d5f157671c3f3d81ab399db552b9f851b72333770348eb1fdb8a085f924095eb9d5ccfd8474b7ba5a61c7d7bcde5a7b44362cf", MD: "fa9726ccb068c0adb5d20079c35a318b3d951eb43b196c509ab790b7e9202207");
                yield return (Msg: "ff83dcd7c1a488e5a128d5b746284552f1f2c091615d9519f459bc9010ca5e0ac19796c4a3fd7a15032a55a1410737d07855b07f61fbd8f5759e9218", MD: "8bd8d494a41acda4b7cd2994badaecff0f46ba2743458f6c3fdc0226f9492ede");
                yield return (Msg: "afd4764cc7d5de16a3cf80c51d0c0d919f18700c7dc9bc4e887d634fe0a3aa94097d590e4123b73f11ccb59e23496a3d53d2bfa908056c11c52c23abfb", MD: "e9e3b3da648cf230f1973f3814eb81316d2a496826ea39adf4674576f97e1167");
                yield return (Msg: "6fa6de509719ffbf17759f051453c0ac3cbe13346546bbc17050541074b034af197af06e41142211ee906a476039b3e07d6cb83a76aac6fca8eac307c034", MD: "766630993fbb651fd8d3603e3eebc81931fb1302a46791df259a6e13ca2cba9f");
                yield return (Msg: "93cbb7e47c8859bef939155bea488090283ecf5023d99767c960d86baa333af05aa696fc170fb8bbac1e6473956d96b964580ee6640f0cc57be9598e55fc86", MD: "d3212abca1100eb7658c0f916daf2692c57a47b772ee031c4ec6ad28a4a46de9");
                yield return (Msg: "67e384d209f1bc449fa67da6ce5fbbe84f4610129f2f0b40f7c0caea7ed5cb69be22ffb7541b2077ec1045356d9db4ee7141f7d3f84d324a5d00b33689f0cb78", MD: "9c9160268608ef09fe0bd3927d3dffa0c73499c528943e837be467b50e5c1f1e");
                yield return (Msg: "4bef1a43faacc3e38412c875360606a8115d9197d59f61a85e0b48b433db27695dc962ed75d191c4013979f401cf3a67c472c99000d3a152227db61de313ab5a1c", MD: "8703a1f7424c3535f1d4f88c9b03d194893499478969fbb0a5dc2808a069ab8f");
                yield return (Msg: "f0be5e961bb55b3a9452a536504f612a3e66aec8160a882e5156eb7278433b7ea21de31e39383d57fcdfb2fb4a8d227a9d6085fb55cad3abb78a225535da0e34efea", MD: "2fa180209bf6b4ad13c357d917fabb3e52c101a0cdb3f2299fa0f7f81dfb848e");
                yield return (Msg: "206f1c36ba25aea73398fffc9b65c4637cc1f05a6bbee014dccbd61e3b7aa9423887bbac62152a4bf73a4b7afabe54e08720589464da7985d8e6591ac081d115df2fe6", MD: "558ea7c800b687380cce7e06006e1ebe0b89973f788c4caac5780f22dbf382e8");
                yield return (Msg: "8cd71434c00663f3bda0205508a4a266548dc69e00ca91fde06d165b40279af92674f75bd8133e5a9eb9a075c9068f68f4b820008a1fb42d89d1d759859e68f8efc6fb60", MD: "085b343b08516f320a9b90fe50440a8bc51ae0850fa38d88724a4d6bd3df1ad4");
                yield return (Msg: "4cf5bbd91cac61c21102052634e99faedd6cdddcd4426b42b6a372f29a5a5f35f51ce580bb1845a3c7cfcd447d269e8caeb9b320bb731f53fe5c969a65b12f40603a685afe", MD: "f9dbb88c5bb4415e17dee9222174538eeab371b12d8d572cfdf55b806e3158e4");
                yield return (Msg: "e00e46c96dec5cb36cf4732048376657bcd1eff08ccc05df734168ae5cc07a0ad5f25081c07d098a4b285ec623407b85e53a0d8cd6999d16d3131c188befbfc9ebb10d62daf9", MD: "3571326a1577c400b967ac1c26df2a0dcf5db7070eac262a8071da16afa7c419");
                yield return (Msg: "981f41a83d8f17f71fc03f915a30cd8ac91d99aa1b49ef5c29fb88c68646b93a588debcd67474b457400c339cca028731df0b599875ab80df6f18b11b0b1c62f2a07b3d8209402", MD: "62aea8760759a996f4d855e99bcd79e9a57ea362522d9b42fd82c12c9294a217");
                yield return (Msg: "5c589fc54fefc4d6e2249a36583e1992fc6b8a9c070e8e00c45a639af22063e66ae5cdb80238c82db043a5e1f39f65626e6d7be5d6a2d3380fa212f89211200412e5e4315fc04e40", MD: "18deba74e9d93ae7df93c6c316ef201bf5e3a661e68868e14d4f56264f5d858c");
                yield return (Msg: "7c8691e7b2560fe87fcc5e2877f7e3c84d9101eca4818f6322a58986c6cf05627c0d6919ef2edc859f81fa1f33e0cc1f10edf7e52a9c33981af2ff0d720c94ea4d62170b2a4d1224fa", MD: "5a5a438b57c1b3ce8756094252362afeaa9fc91cd45b385d16994ec8af49aa6b");
                yield return (Msg: "97359b564b2bc20800ed1e5151b4d2581a0427ce9539d324c3637cfb0e5378dc2cf6d72946e2a3535a2f664ede88ed42a6814c84072b22c43de71e880a77c2d9a05b673bc15a82e3255f", MD: "be54f2e435f760d5b77c0ae61ef0aa7f5f3366f47819f350dc8a39aff8c73a8f");
                yield return (Msg: "a0dfaecd3e307c5ddf9a93603f7e19725a779218734904525b14586ff0ce0425e4efe7e1c06e745c28ed136f6031c4280fd4061d433ef700b6d1bc745064231fecf387015f94f504b6ad8c", MD: "60d80f1c703dad5da93db222fb45fb7fa768c8aa2787f4b81f1e00365b8f49e2");
                yield return (Msg: "568d66d061306c3419a1928ce7edc8e3400c30998f09bdac6f63ff351eb23d362e8dc5927eac805d694ac9563dcd7fb2efa9591c0d827af9f39146f0424873aa8e3963d65734b1713baf0a44", MD: "7a4fe37f296991121792dd7c2c30390725a1eebbf20b766a5a1c3c6c3646d996");
                yield return (Msg: "d65b9f881d1fc7f17d6dd429faca8404e6ce60fba7d89b7fba003c8ef84d8083182979327611fc341291ba80dc70ad3b2f28b6d29b988445e7fdb7c6561f45822ac81dbf677a0b27d961dc6358", MD: "51cc71b6934afcf28fa49942b76323f36cd6a0aecc5a0e49c10994ddcabdbb80");
                yield return (Msg: "711c88adf13e7a0e694652f2b9a397543f4937fafb4ccca7f1ad1d93cf74e818d0fedfaee099f019014ec9e1edfe9c03fdb11fe6492ad89011bf971a5c674461de15daff1f44b47adad308baa314", MD: "1780e52e306858478290c46b04d8068f078a7f6ad8e3790a68fc40dccfbdadc9");
                yield return (Msg: "f714a27cd2d1bc754f5e4972ab940d366a754e029b6536655d977956a2c53880332424ddf597e6866a22bfca7aa26b7d74bc4c925014c4ed37bfe37245fa42628d1c2ee75dc909edc469ee3452d894", MD: "f4afa72f3e489ad473dc247aae353da99fb005b490e2c4e1f5bd16a99732b100");
                yield return (Msg: "fe0c3280422c4ef6c82116e947da89f344d6ff997bf1aec6807e7379a695d0ba20ae31d2666f73bbdbc3a6d6ac2c12dcfb5a79173dfc9cd2e0d6000e3114f2767edec995772c6b47dadc136d500251e5", MD: "89198e2363efd4e0ba7a8a45f690f02712e6f856668517bae118d11e9a9dc7cc");
                yield return (Msg: "02e238461d0a99d49c4cd16f442edf682c39b93114fc3d79f8546a99e5ead02f0cfc45081561da44b5c70eb48340418707fd6b2614580d5c581868ba32f1ee3ac34bf6224845b32ba7f867e34700d45025", MD: "abef81b33591eedcac0cf32fb5a91c931f2d719c37801409133552170ce50dbf");
                yield return (Msg: "fb7c8cd4031007f8159d5c4c6120dee6777a3ace0a245b56f31e8aae7828dab3cf35c308de1d0d684592ef3a9e55796603a92f68d109f7a3ac1635f7c4d334955614c812753431bb0a0743291a0fc41547f3", MD: "5a67284d39e4f37caa64ca1a54593c35f6d8f3a3ec20d460393a39f6f57c4486");
                yield return (Msg: "6b2e868c7d0ee1c240d3a67e2fdf36e8e23817c02644a54453d10454da5859d41e833a5285ec63e8ce28aa64a50435a7740eea4b7d5827892678b35993d3f5da7a1c64f533173f3d0fa37e1aebf70827052c26", MD: "aecf5dab6fea9ffd1bce2cdfeec0bee9d214a669e8306d5b6688afa8957fc91f");
                yield return (Msg: "e5f3ba000c43bb6aca4e0a711a75912a48241cffa5b4b0b17f901f9e5097d94036c205f7a307d008567d05e58ac0dfaf6d971bf9d3d450cf2c7c83f6b328f676e9ab425642f5a5a71e389dc4fa49b6d7e848a09f", MD: "182d6e4316f4bc18d7163b1b21462d99f99c6f34d2c00ee771ce54fd6c5018b9");
                yield return (Msg: "939c61e68af5e2fdb75a2eebb159a85b0c87a126ce22701622f5c5ef517c3ab0ed492b1650a6c862457c685c04732198645b95f84ccb0e726a07ce132827a044dc76b34d3f19a81721f1ea365bc23e2604949bd5e8", MD: "121057b0b9a627be07dc54e7d1b719f0a3df9d20d29a03a38b5df0a51503df93");
                yield return (Msg: "9eadaf4811a604c65eaa7b1c6e89f2c0ab96bebec25a950ba78aac16d9371ca1e7458acf331e077ef6a735d68474ab22d2389bdf357fb2136c9f40e1e1eb99592c2bbb95d94931016b4d37faa08b1e9bf71bf2d3708a", MD: "c237194b902e48dca5bd096cb51562079d0cdccb2af8088197676c17b0896be2");
                yield return (Msg: "71dcca239dced2ac5cc49a9bf9ea69a99be22ba62216716b524db80f337dee5eb7e032869e4adc1497babd1fa82fa8c3cfbd30d2eadfb4c5d40f99f9d194d7182c9cb7d41e8adbdcf2917e086782fdd756e2961c944070", MD: "377d1cffb626735810b613fd31ef9bbb4577cd752521abe3a41afa921e623da0");
                yield return (Msg: "ea130d3236bca7dffb4b9e50e805309a503e7347227aeb9f1bd15c263a98dd65753d2eedaa734b9ad88f41158f32419ca529f3062b910c019f3f239f635fc1116e5ab7b242feb4471ed9168474e501d39d6bae52cc21061a", MD: "85c7a52d53f7b41162ea9f1ef0d07c3fb8f0ec621617f88cb3828ebe5388ab3d");
                yield return (Msg: "28f1be1156792af95c6f72e971bf1b64e0127b7653ff1e8c527f698907a27d1544815e38c7745529bc859260832416f2b41cd01e60c506239a7bf7553650bf70d1fe7a2c1220ac122ea1e18db27490447d8545a70bf0ffc8fa", MD: "b2eb3762a743d252567796692863b55636cb088e75527efd7306a2f6e3a48a85");
                yield return (Msg: "c8400ef09c13e8acc8a72258f5d1d20302c6e43b53250c2f6c38ff15be77e3cac04d04b8421fc8fdff8be5ca71edd108e9287b42dea338bf859100eea376da08a0e695f0dc90b95e467cbd3c2a917a504a5ae01c310ae802c4bd", MD: "69966e89b7bc7f39cd85791b92180ff3fed658d8240e393e1e6d7c24b8d0ac95");
                yield return (Msg: "a48950c961438e09f4d054ac66a498e5f1a4f6eabfde9b4bf5776182f0e43bcbce5dd436318f73fa3f92220cee1a0ff07ef132d047a530cbb47e808f90b2cc2a80dc9a1dd1ab2bb274d7a390475a6b8d97dcd4c3e26ffde6e17cf6", MD: "44c00cf622beca0fad08539ea466dcbe4476aef6b277c450ce8282fbc9a49111");
                yield return (Msg: "e543edcff8c094c0b329c8190b31c03fa86f06ace957918728692d783fa824ba4a4e1772afbe2d3f5cba701250d673405d2c38d52c52522c818947bcc0373835b198c4cc80b029d20884ac8c50893c3f565d528a0cb51bf8a197d9d6", MD: "6d5260384f3cefd3758fb900dcba3730d2b23cee03d197abeff01369dc73c180");
                yield return (Msg: "4e10ab631718aa5f6e69ee2c7e17908ec82cb81667e508f6981f3814790cfd5d112a305c91762c0bd9dd78e93ef3a64c8be77af945b74ff234a0b78f1ed962d0d68041f276d5ea40e8a63f2cab0a4a9ed3526c8c523db7cb776b9825b4", MD: "d88e5f3b2d0a698fd943233760a3000a3360d9040e7374b22e39ea58d868102d");
                yield return (Msg: "604d8842855354811cd736d95c7f46d043a194048b64bf6cda22c3e0391113dcc723e881ae2ad8dc5740aa6bda6669ddb96bb71acd10648380693f7b3d862c262553777004bd6852831618519fbb824759f4dd65af1b2a79cc01096d7c8d", MD: "8a8ab6cf5c02b9ae8f4c170740eff1592f3eda11d3420ac8b421d93cfbb35db8");
                yield return (Msg: "628180e14f41ebdfde3b4439de55ee9cd743d41040f3457ef2280370dd659619fa0ce69580c709725b275a6eda8bcb82a8447c20fdf68cba15412f83e2a10079fe9399a3e3fa61975ec0a64041c0ecde59e4844e9f8a608cb22d2576854182", MD: "8d154bf6f9cb72efc0d8b3927a8f690060d1d48bbe5cc72094d2c8b149a75132");
                yield return (Msg: "fc150b1619d5c344d615e86fca1a723f4eeb24fbe21b12facde3615a04744ef54d8a7191a4454357de35df878cb305692278648759681919d1af73c1fb0ff9783678aec838da933db0376e1629fcca3f32913f84bc2ff3ffc3f261d2312f591c", MD: "3f626c8bb20a132495bd3022b3fcd0ce0604b91a9d70132dab4099f73dde23d5");
                yield return (Msg: "6dadbecdd15e5646e3f37a6fe5b328e06113cce3c8cf07285939afba44d117321017902b3a9d2ff51f60d18e1b585dcdf34e49e170ee60fa4d1dc246548d2c1fc38e7983f42769c43d65a28016f3f4d479ebe1cd8fec5d1f886dd21aca5067d94f", MD: "9098ea34c40b541b153e80a8bd92da19432b18b7d329760b302f8a54c395dd06");
                yield return (Msg: "9cc5fd3035b72dc63b8c3c326fd013081e6b8716f526d3fe176b45256d4c37cc3dc8417dff49ada96c702b8fd715c65fc08a17a0a720b9cf1eedfd4922ccde6baba437f782ee33b95371056b0350dad743470c3b663299f16fcfd34f6fc459cd0ee4", MD: "b0c04f24bb6d3d4fcbfdf9222d0e886f1eb60a0566a478085f7623a025a5b981");
                yield return (Msg: "f3f063fbcf2d74aa5a02d240c962ed7bb119b3a212bdb41594e28428108e613152ed16e01e451fcf702b0e5a08f82eb12677652b93e05fdee00ae86cf2dc9a1fbf05b93952ec5b8515eacc324fb830e1ec236afd7d073d4b7f7ab1c2e048b99cbfa012", MD: "f930d79360b581b1bbfdeac57133a339444f5c44538c921631eabaf058277d32");
                yield return (Msg: "840739a3d6992c13ec63e6dbf46f9d6875b2bd87d8878a7b265c074e13ab17643c2de356ad4a7bfda6d3c0cc9ff381638963e46257de087bbdd5e8cc3763836b4e833a421781791dfcae9901be5805c0bbf99cca6daf574634ec2c61556f32e642730510", MD: "19795657e08cfbb247a17cf209a4905f46e4ddf58eea47feee0be9bb9f5c460f");
                yield return (Msg: "4a51b49393ab4d1b44fb6dc6628855a34e7c94d13b8b2142e5d5a7bf810e202cefdca50e3780844a33b9942f89e5c5b7dd6afb0a44541d44fb40687859780af5025fecc85e10cf8249429a3b0c6ff2d68c350c87c2fcbf936bd9de5701b2c48ce9a330c9ee", MD: "128fb4114e43eefd19277c708be9e6873e66d7fd59c58a1485b7b015facfa795");
                yield return (Msg: "afc309e6b7b74dfb0d368e3894266fc4a706c3325e21f5550d07a6560e3d9703c134ca6ad078e4a7b82ad6fa85b0bc1ddcab05d43f29d5c58d1da78ac80c37051b089ff31ce2c0c44e9ce3abea1da0f1df28008e178fdefafca493413bf1d256c729d0a9225e", MD: "03e782b01a4ba10f640470bb3cae487eb9cbbaab8c9941978b194f6a312cf79e");
                yield return (Msg: "c5ae750f2230642092397b84ad5526c46ae9480ada16892816e0f2db7690b751035653ea2f33da3cc4168b591b46a5548eff7d012f60ccfdbb854deec9f0880c472de8e127b5144c56147cccee4732fbac68fc59a48da74b33ed9e643644bbe279795c7c737eba", MD: "f64b7ab243ce6e6c04b483888ba8a655465c21d95eb60c7b8d6e566a3811bae2");
                yield return (Msg: "603e13f61499e12ec6b33b68847a281d314f54dc705c0f3fc428981ff5689c04b519fadf83cbc9fcd0409c326035045df480570e265bb080940037ce4076a36437aafdb371c1a62af9ad9b614dfef89708fbbb5ebef2cb9528cc399781e4c5b22f1aa4dba623809f", MD: "5f76962fd3d373e5db2953c0823a51fe81f874450bedf7e46876394b04d3ef66");
                yield return (Msg: "e03115cfa19efcd796da389063c4be6acce684d983f8edfb3da6887b0b94fbb5e89e3a1a8e64fdd68f0670b1a02c2c33384a660c5a2266b3ae8a3b4cd76faecf011a7467b9b2a818020278a5a57d1eb1c87f1224c2d67dd02e81f1553eb75841532c2b7cca8fe5e418", MD: "d107ee6ee4a58871a33c49657faa2573e475f11918c4a4e3801d0e17fb93c6e3");
                yield return (Msg: "0e6c1d58b1b9d3a2d399aafd60529e07d483a2755bb7e44c373b5355632d5fca76d6ff56c93af93ddcec5ed6f62753420c1b1758e48542df7b824b00a3a54dfaf0470b18d51e31e10b12dd8e324b5dc1bb8f3b7305cb762ec6ef137dadffd4a2466748861d9004f626b0", MD: "02ab2dbb02944354799051247b1a25c19f3696e1afcb502b859e83798b33fd77");
                yield return (Msg: "6db2a43a229b10c3629249fc5136468b4d84df7b89ec90ebf7aa7a036c53aa2dffae9e81b2c60580543dc706a5e3457abc87e248a60ec29150c2d221a6ec08a1fda4ec0daee8576904ec7ab059b1230e7bd93c4e55ba9496cbb1e352e5b8086e303b94c861288ce53c466b", MD: "8cc4d39b2f5ba0bc9d2ee2a8777cf08533e60cc69b65a7b31c5c2121193aa31e");
                yield return (Msg: "31d995f7ff8b6de70829a8336c610f10df2c866107a4922b25151849f8566861df5a79163d02767f21357ad82733997899261f03dafb1ce1056f20efd16d4374b89768565823c38e19e899d910b847b023f1867b6e4fed02e604b8243c0bc7cb05b9ea1f17955bfa36698c9c", MD: "c99c7191b34c9ad3f941d4ad442cc865205cbb4c2a6927c592e831cbc4d36fcf");
                yield return (Msg: "cb0b8cb7de621c8e0a0fc6be2fc18d0e8818a2c2dd0b3219fa87831a61583f903c4d105495976ccac973b3ae3a09771145931a9e74c19f22f45cba4c492b29b1401347122581dfe2370d3e0359578cd10a355c619711810a8f8c232578671312c0a45c7cf7e81bdd3b249044f3", MD: "6d2f57a7e42b35369cf2cd60caf9e65aca7d9aa019e6824bb806348f1acf3c7c");
                yield return (Msg: "48dff78aed5f6e823054924a78dc1b8e51a117f1610181529f6d164ebf0f6406f0b02422cad8c916823759a361437ca17423d3fd84cc8afe486a31ccda01c732685418a32c064a7b9effb288e811ecc99adb2a759feecc3f702f31d9877dcdb717937c15fa2f163bea744400f58c", MD: "14b631f0f00a3024ad1810dabf02711e28449668abe27f69380942268968d4f6");
                yield return (Msg: "06cc9fa542ceb35c88fb6ab82c29d5dcd530f807d3f1c3bcb3974421101d1aa6ac112de6bf979cd28eb0f70c40bcaf91ed3eca9bf9e0dbc6a0b73271d1c7506740ca9ebfb72d5e00ac5ce189193ffa308804b42a6d20402bb99031cdac65ec36eb7f59f5d299df2e0b8690f760b9a0", MD: "574fd82a9fceb8f7bbbf244d16e0412cbda8153b720846c32b8f10fe5779a881");
                yield return (Msg: "8d93627c0b7cbf61a7fe70e78c2c8ed23b1344b4cfed31bd85980dd37b4690e5b8758f7d6d2269957a39a1ac3451cc196696ae9e9606a04089e13456095a1ce1e593481b3ac84f53f1cb10f789b099f316c948398ad52fa13474bdf486de9b431bd5d57ef9d83a42139a05f112b2bd08", MD: "344ec86642eabb206b2fd930e4c5dde78aa878577d6c271cb0069d4999495652");
                yield return (Msg: "d0af484b8be6b41c1971ae9d90650a1e894356c9191d6be303fa424f2b7c09544ec076a0f1865c8c97927ca137529d5bedc0df2ef08a4cc7c470b094b1eeaa86731c041633d24086b60f7369d59c57652dec9b3817477df9db289ba020e306c9a78a99b539128992deb23cfc508c5fc3af", MD: "b7ba998726477c32792e9c3eddc1cb6feb7c3933e49f2e7590d8ce7a2113e6f8");
                yield return (Msg: "b212f7ef04ffcdcf72c39a6309486c0eeb390ff8f218d6bd978b976612f7f898c350e90bd130723e1126af69295019b4f52c06a629ab74e03887020b75d73f0f78e12785c42feb70a7e5f12761511c9688c44da6aaa02afa35b31edc94c3a0779b6ab9462525c0ccfba76986f873fe1e6ba9", MD: "2f26b96c1fa3f3dee728f17584e733b4189821c659b8885a5fb1d12d60d2aaa9");
                yield return (Msg: "86591ada83fba8175a0fe91d264e7f9b2df97ee4c32570e76b579d6140508951932abdadd6a4ca53b8bb8c42927aac0a02126881d52d97b82b80e72dd59f6a42021651ee1bb5f7b3eb2b21d003d784b75dda87c13f714b216282e8175474fa661b445d071bd5341f3a88302f410d0f8a857962", MD: "e3edbc8c42ce5d2384dfb24fb1de5d4798b1bc3cc78c97033894040dfa6feb6c");
                yield return (Msg: "92b5a8e84b6a2ac4d5b1e61d63804abd641dd630058ec6d5f752f135724ef1947a0a84c6611d32448de6307f7b7d857404e96b81df94f87768fcfdf09faa2fe37468847542afe012995ff1bd40b257a47a7309f8896bf4fb711de55bfeb3a8be0837729ef6067c578182f17ebb080a754f22773c", MD: "80ed0a702812297c2aa1b6b4b530c2b5ed17ecfba6d51791cf152d4303ced2e6");
                yield return (Msg: "d284a0a9a4de5d4c68cc23884c95ad7619aa39b20a2cf401deaeb3362c3ce356f79cc3fa82d3d1f565ec8137e1f435f171496afaa1152f722315dca5209f0031cce39b6c3d718e007dfb4fd8de5ce1408dda04476aa8a96817afa86a4f8fb5857ae091c67ebd7db5d783f434ead699aa96e56f610d", MD: "654eccefd0a4fdb2ac0ab56288c64399b37bc4d57ff4a9f1cce94362fc491bda");
                yield return (Msg: "f57f0f8795385b805246a0a2573afc274346a9eccf50c626b0455a50bfb09668578b5a5afe54fbbd486444bdf97dba586aa224ce2e2b4b52f418ff06afa65a26f5204983a5f84734cd166c88cb70a73fb2db48f9ef20c1ee2c53ade07460114e98e7e2ebd24ac84ea90422eb143c4a42e2991a565959", MD: "135ec8b144a667dceae8fadd287df81c10ef3ebef87ff2fb56e60ae708a88f3b");
                yield return (Msg: "2a41a52e6578873588a57f11f1be7c7eb398d01f3bfdec2c33fe6b65a68a534a6540978daa82e0c8fccb8c6c5242f7f97b8ffa75bdedb217bd8083439eea5cbb6d193c13bd62f5658ed4304774c6b1faf5b3dce432487840cabab415fb5d67640a739ca6e5414e760869708a9d7331e7e7ad7d55e035c7", MD: "a6a1b8a26f6f440f19f16dce1d3001477d73ee7f6c374bce2922167b81970d6a");
                yield return (Msg: "4d11aa5d3c6b6900f49ff90dd815744572be5648b64bde638b9db7a9877dd745fa8ea80e2f7f655cee85c71a4509e21d899e49b4973579815f947587a404ad83fd4a248020d9d2a65f46485373fc926d793161f63a196ae0af590923c5be2a0e5d2f69da97e0788550c9c1dee9574ddc4a61e533275d7729", MD: "fc5159f0ddd6d765c85fcc3fc3ac1dc0d317d8ea0b110e96ac9f7a398dc386c5");
                yield return (Msg: "05cd99bfe031d123ca7061d3de0956f4bbf164bad792db881713d6599ddab55ee24fcee804e360896152c8766424f8309f7a24641a07be0feb5da5e5076a9af45842f385101f93433ca5199f9c6b5872b2b808e4198aba8e18dd12db772930b4912d6f5cabeb529884f4bb142de55e021b3276047b22b64cc5", MD: "8aa07742e6f1f47ad020ed6684edc8dba4af36b782955f0f972be3ae980aea0e");
                yield return (Msg: "529684398d68bdc19e7a00ce32cc1a8c1315b97f07137474f61f0cb84a04f2879b1109c78c6dacf7f0abf362329e3298f36fc31ef4ec06653723a5f961301dfb63537ad15946611cb2cd54ea928e322e7423fd6d146ee0b98c2c71e3bdcd33edf0845fbebd9ae4192d07acd01b432135e05af0d22f3f0c5a3d62", MD: "a07049b6ebd7b355479a3d802fda436b83ae6747d741cf9626f7c62f47cbd563");
                yield return (Msg: "982fb5f4af498a4a75e33a033235ea3ddb70d9d236519f883ff5b388cbef30126b98d96e93a65a26fb00d17246d18cf4e2db14a52f0f6b10e35a93beadc14ff118b02e95b38fc4736f973ba848e40b5527cb0599076d96bc578c4aada09e8faf6820bc4f562d5199974f808b7f95edca74e6b3940894a7f66534e0", MD: "09c60fec5a089a23f5da3ed2492aa21fcf7aa36183850fafc15ae8c63f596db0");
                yield return (Msg: "ca88614828f8acdb5fcffab6bb2fb62d932b7808e4d9cc3139a835b0cef471d9f4d8ffc4b744dffebf4f997e74ce80db662538bceb5d768f0a77077e9700149ea0e6a46a088a62717216a14b60119dd19c31038ed870b4709161c6c339c5cc60945a582263f3be9a40cd1a04c921947900f6e266f2390f3c970f7b69", MD: "fe2d4183ccdaa816b4446a9b6c07d0ba4b42ac743599db5dc482b1941f443c71");
                yield return (Msg: "ab6b92daf83275cb9c1b76cfb59fbcc8ac53188e0b6980918e7ac0c07c836ca9372d19e11251cca664bbb3c3db2e13b412a9820b65e95612042f5db24643cf9340b9808597735a1f92670ba573a2fb2f088d81087d70565574344af7576d35b2ed98318e2ca0067d4fa8e63f28045b83b6887d4ffa0668a10712ed5759", MD: "744538e1ae1cd7357710b56c3bc6f1bd7a8564118a1e0f9acc30fcf0b5396eef");
                yield return (Msg: "bfd4c7c8e90858ccf9c8834abefd9c1846ca4a11966fdd139d6de24a6bebf4b19f58d5d51e52bddd0bc6f1c7f35998f44707cae7100aeb4adefe373101429da3fca1d15737329dbbf47c783a84de59bfbb2fcd75a1a148d26aebb8d3a9a76089c0f8e4d49b71a06f9e323e2cdb54888189887a44b1fa9cb32b7c8fb7c9e0", MD: "58b17843bc851a721c5a258eef57b3854d02190e732d9b8e7a9f926ac409c173");
                yield return (Msg: "c5019433c285da2bb93f119e58b4f36cd1e4d99dda35dbf4f8ae39c7fe65fa0ed03bd2b96dc649472d8f1a94477ed9f29592d97c9cd54da7c790ad1af3bb5cc030b7871bc64050db779d2caf0419895bf3b7b50b8e22fbe62fe30fe7bbd6ace86ddf7b00d5d9370f20cf0f97996f4bce70bb33f1ba022cdaba0f25d55fa031", MD: "f7c92a3fb7f180370d628be78de874d693f74ccc7a54c741634258d8c512fd7f");
                yield return (Msg: "84b60cb3720bf29748483cf7abd0d1f1d9380459dfa968460c86e5d1a54f0b19dac6a78bf9509460e29dd466bb8bdf04e5483b782eb74d6448166f897add43d295e946942ad9a814fab95b4aaede6ae4c8108c8edaeff971f58f7cf96566c9dc9b6812586b70d5bc78e2f829ec8e179a6cd81d224b161175fd3a33aacfb1483f", MD: "8814630a39dcb99792cc4e08cae5dd078973d15cd19f17bacf04deda9e62c45f");
                yield return (Msg: "14365d3301150d7c5ba6bb8c1fc26e9dab218fc5d01c9ed528b72482aadee9c27bef667907797d55514468f68791f053daa2df598d7db7d54beea493bdcbb0c75c7b36ad84b9996dca96354190bd96d9d7fbe8ff54ffaf77c55eb92985da50825ee3b4179f5ec88b6fa60bb361d0caf9493494fe4d28ef843f0f498a2a9331b82a", MD: "9b690531dee948a9c559a2e0efab2ec824151a9175f2730a030b748d07cbaa7f");
                yield return (Msg: "4a757db93f6d4c6529211d70d5f8491799c0f73ae7f24bbd2138db2eaf2c63a85063b9f7adaa03fc348f275323248334e3ffdf9798859f9cf6693d29566ff7d50976c505ecb58e543c459b39acdf4ce4b5e80a682eaa7c1f1ce5fe4acb864ff91eb6892b23165735ea49626898b40ceeb78161f5d0ea4a103cb404d937f9d1dc362b", MD: "1ac7cc7e2e8ea14fb1b90096f41265100712c5dd41519d78b2786cfb6355af72");
                yield return (Msg: "da11c39c77250f6264dda4b096341ff9c4cc2c900633b20ea1664bf32193f790a923112488f882450cf334819bbaca46ffb88eff0265aa803bc79ca42739e4347c6bff0bb9aa99780261ffe42be0d3b5135d03723338fb2776841a0b4bc26360f9ef769b34c2bec5ed2feb216e2fa30fa5c37430c0360ecbfba3af6fb6b8dedacbb95c", MD: "c163cd43de224ac5c262ae39db746cfcad66074ebaec4a6da23d86b310520f21");
                yield return (Msg: "3341ca020d4835838b0d6c8f93aaaebb7af60730d208c85283f6369f1ee27fd96d38f2674f316ef9c29c1b6b42dd59ec5236f65f5845a401adceaa4cf5bbd91cac61c21102052634e99faedd6cdddcd4426b42b6a372f29a5a5f35f51ce580bb1845a3c7cfcd447d269e8caeb9b320bb731f53fe5c969a65b12f40603a685afed86bfe53", MD: "6c3e93f2b49f493344cc3eb1e9454f79363032beee2f7ea65b3d994b5cae438f");
                yield return (Msg: "989fc49594afc73405bacee4dbbe7135804f800368de39e2ea3bbec04e59c6c52752927ee3aa233ba0d8aab5410240f4c109d770c8c570777c928fce9a0bec9bc5156c821e204f0f14a9ab547e0319d3e758ae9e28eb2dbc3d9f7acf51bd52f41bf23aeb6d97b5780a35ba08b94965989744edd3b1d6d67ad26c68099af85f98d0f0e4fff9", MD: "b10adeb6a9395a48788931d45a7b4e4f69300a76d8b716c40c614c3113a0f051");
                yield return (Msg: "e5022f4c7dfe2dbd207105e2f27aaedd5a765c27c0bc60de958b49609440501848ccf398cf66dfe8dd7d131e04f1432f32827a057b8904d218e68ba3b0398038d755bd13d5f168cfa8a11ab34c0540873940c2a62eace3552dcd6953c683fdb29983d4e417078f1988c560c9521e6f8c78997c32618fc510db282a985f868f2d973f82351d11", MD: "3293a4b9aeb8a65e1014d3847500ffc8241594e9c4564cbd7ce978bfa50767fe");
                yield return (Msg: "b1f6076509938432145bb15dbe1a7b2e007934be5f753908b50fd24333455970a7429f2ffbd28bd6fe1804c4688311f318fe3fcd9f6744410243e115bcb00d7e039a4fee4c326c2d119c42abd2e8f4155a44472643704cc0bc72403b8a8ab0fd4d68e04a059d6e5ed45033b906326abb4eb4147052779bad6a03b55ca5bd8b140e131bed2dfada", MD: "f82d9602b231d332d902cb6436b15aef89acc591cb8626233ced20c0a6e80d7a");
                yield return (Msg: "56ea14d7fcb0db748ff649aaa5d0afdc2357528a9aad6076d73b2805b53d89e73681abfad26bee6c0f3d20215295f354f538ae80990d2281be6de0f6919aa9eb048c26b524f4d91ca87b54c0c54aa9b54ad02171e8bf31e8d158a9f586e92ffce994ecce9a5185cc80364d50a6f7b94849a914242fcb73f33a86ecc83c3403630d20650ddb8cd9c4", MD: "4beae3515ba35ec8cbd1d94567e22b0d7809c466abfbafe9610349597ba15b45");

                // First 5 from SHA3_256LongMsg.rsp
                yield return (
                    Msg:
                        "b1caa396771a09a1db9bc20543e988e359d47c2a616417bbca1b62cb02796a888fc6eeff5c0b5c3d5062fcb4256f6ae1782f492c1cf03610b4a1fb7b814c057878e1190b9835425c7a4a0e182ad1f91535ed2a35033a5d8c670e21c575ff43c194a58a82d4a1a44881dd61f9f8161fc6b998860cbe4975780be93b6f87980bad0a99aa2cb7556b47" +
                        "8ca35d1f3746c33e2bb7c47af426641cc7bbb3425e2144820345e1d0ea5b7da2c3236a52906acdc3b4d34e474dd714c0c40bf006a3a1d889a632983814bbc4a14fe5f159aa89249e7c738b3b73666bac2a615a83fd21ae0a1ce7352ade7b278b587158fd2fabb217aa1fe31d0bda53272045598015a8ae4d8cec226fefa58daa05500906c4d85e75" +
                        "67",
                    MD: "cb5648a1d61c6c5bdacd96f81c9591debc3950dcf658145b8d996570ba881a05");

                yield return (
                    Msg:
                        "712b03d9ebe78d3a032a612939c518a6166ca9a161183a7596aa35b294d19d1f962da3ff64b57494cb5656e24adcf3b50e16f4e52135d2d9de76e94aa801cf49db10e384035329c54c9455bb3a9725fd9a44f44cb9078d18d3783d46ce372c31281aecef2f8b53d5702b863d71bc5786a33dd15d9256103b5ff7572f703d5cde6695e6c84f239acd" +
                        "1d6512ef581330590f4ab2a114ea064a693d5f8df5d908587bc7f998cde4a8b43d8821595566597dc8b3bf9ea78b154bd8907ee6c5d4d8a851f94be510962292b7ddda04d17b79fab4c022deb400e5489639dbc448f573d5cf72073a8001b36f73ac6677351b39d9bdb900e9a1121f488a7fa0aee60682e7dc7c531c85ec0154593ded3ae70e4121" +
                        "cae58445d8896b549cacf22d07cdace7625d57158721b44851d796d6511c38dac28dd37cbf2d7073b407fbc813149adc485e3dacee66755443c389d2d90dc70d8ff91816c0c5d7adbad7e30772a1f3ce76c72a6a2284ec7f174aefb6e9a895c118717999421b470a9665d2728c3c60c6d3e048d58b43c0d1b5b2f00be8b64bfe453d1e8fadf56993" +
                        "31f9",
                    MD: "095dcd0bc55206d2e1e715fb7173fc16a81979f278495dfc69a6d8f3174eba5a");

                yield return (
                    Msg:
                        "2a459282195123ebc6cf5782ab611a11b9487706f7795e236df3a476404f4b8c1e9904e2dc5ef29c5e06b179b8649707928c3913d1e53164747f1fa9bba6eeaf8fb759d71e32adc8c611d061345882f1cdeee3ab4cab3554adb2e43f4b01c37b4546994b25f4dcd6c497bc206865643930157cb5b2f4f25be235fa223688535907efcc253bcd0830" +
                        "21407ea09cb1c34684aa0c1849e7efe2d9af6938c46525af9e5afb4da6e5b83da4b61dc718672a8090549cbe5aadb44f5bc93a6b3fbdc2e6d32e2eaaae637465179ea17f23ad1e4f1ebc328e2c6dc90c302b74a1edbbb0676c136b269d70c41040a313af06ab291bf489d9700950b77f207c1fc41884799931b3bca8b93331a6e96b7a3f0a8bd24c" +
                        "db64964c377e0512f36444bb0643a4e3ecb328194cd5428fd89ede167472a14a9bf5730aff1e3b2c708de96eff1ebaaf63beb75f9c7d8034d6e5471e8f8a1f7efce37793a958e134619c19c54d3d42645f7a7263f25471fbaae8be3ea2fbd34ec6d7aacd7d5680948c3cd9a837c9c469a88f600d95829f4d1e4e4a5ef4ed4623c07815a1c33d9fb3" +
                        "b91333ff04eac92806a68a46cf2e9293f8bff466ce87fe66b46fbff7c238c7f9b2c92eb2fdc7d8084167f6f4e680d03301e5c33f78f1857d6863b1b8c36c7fce3e07d2a96a8979712079ae0023a1e3970165bfcf3a5463d2a4fdf1ca0e044f9a247528cd935734cb6d85ba53ceb95325c0eaf0ff5cd81ecb32e58917eb26bfc52dba3704bf5a927f" +
                        "ee3220",
                    MD: "cb1c691c87244c0caf733aacd427f83412cd48820b358c1b15dd9fadee54e5af");

                yield return (
                    Msg:
                        "32659902674c94473a283be00835eb86339d394a189a87da41dad500db27da6b6a4753b2bb219c961a227d88c6df466ba2fc1e9a2d4c982db4398778c76714d5e9940da48bc3808f3c9989131a07683b8c29d6af336e9aee1dfa57d83c48a86f17146edec07869bb06550689ebf4788159ed0a921048b4a6e3e3ec272413bec15d8e1f6a40897fa0" +
                        "e11d9df223ef9fc270106249ae220fdc6ebdef6d6611805421ccc850f53ee9c836baf657a94005883b5a85def344d218264f07b2ea8714afcc941096c6ded0bb6bf5b8bf652fd15a21931c58c9f526e27363ddff98c0a25bc7af9f469ab35bffea948b333f042cc18a82cec0177f33c3bdbf185b580353de79e51e675b03b31e195f19ba1f063d44" +
                        "def0441dc52820426c2c61cf12974ec249fd3502f017ffa06220075ced7e2d6b86a52677ba3916e8e8726062aec5bc8ea1c18b1e4137680b2c9d002191b423bee8691bd7e0f93c3b9959bc1c14d5c5cbe8f7c9c336aa16e9de9faa12f3f048c66d04cb441eb2bbc5e8a91e052c0f9000856896f9b7ba30c1e2eead36fc7ac30a7d3ddfc65caaba0e" +
                        "3b292d26dfba46b5e2dc9bc9acadde1c9f52b2969299bd1281ddff65822b629cfba2928613200e73661b803afdcc4a817d9361389e975e67dfadd22a797bdaf991ddf42db18711c079ecec55925f9978e478612609bacd900172011c27e24bad639ffc24a23877278318872153aef6893ccb5b68b94b33154df7334375aadd3edbb35272cc7b672d" +
                        "ec68faa62900873ded52f6049891b77f2d0311a84b19b73660e09d1f1998095c1da1edecfa9f741b5fd6db048dd68255085d43529279021d59ed853470d6863b7c8e07fcb0d1e6acfb1eb16f7f60bb1f46ce70493010e57930a3b4b8b87e065272f6f1dd31df057627f4214e58798b664e1e40960f2789d44ccacfb3dbd8b02a68a053976711f803" +
                        "4c1ed3a8",
                    MD: "5ac9275e02543410359a3f364b2ae3b85763321fd6d374d13fe54314e5561b01");

                yield return (
                    Msg:
                        "a65da8277a3b3738432bca9822d43b3d810cdad3b0ed2468d02bd269f1a416cd77392190c2dde8630eeb28a297bda786017abe9cf82f14751422ac9fff6322d5d9a33173db49792d3bc37fff501af667f7ca3dd335d028551e04039ef5a9d42a9443e1b80ea872fd945ad8999514ae4a29a35f60b0f7e971b67ae04d1ba1b53470c03847a3225c3d" +
                        "df593a57aed3599661ae2d2bb1cddd2fa62c4a94b8704c5c35c33e08e2debe54e567ae21e27e7eb36593ae1c807a8ef8b5c1495b15412108aaf3fce4130520aa6e2d3bdf7b3ea609fdf9ea1c64258435aae2e58a7b3abda198f979c17dbe0aa74253e979bf3a5800f388ea11a7f7454c4e36270a3083a790c77cbe89693205b32880c0d8f79b1c00" +
                        "0ee9b5e58f175ba7696616c17c45673cff25d1221f899836e95cc9e26a887a7115c4537e65ad4eacc319ba98a9a8860c089cbc76e7ea4c984d900b80622afbbbd1c0cdc670e3a4c523f81c77fed38b6aa988876b097da8411cc48e9b25a826460a862aa3fadfe75952aa4347c2effebdac9138ebcc6c34991e9f5b19fc2b847a87be72ff49c99ecf" +
                        "19d837ee3e23686cd760d9dd7adc78091bca79e42fdb9bc0120faec1a6ca52913e2a0156ba9850e1f39d712859f7fdf7daedf0e206dff67e7121e5d1590a8a068947a8657d753e83c7f009b6b2e54acc24afc9fdc9601a1d6d9d1f17aab0ce96c4d83405d1e3baba1dffa86ecccee7f1c1b80b1bbf859106ce2b647ae1e4a6a9b584ae1dfc0a4dee" +
                        "bb755638f1d95dcc79b1be263177e2a05c72bde545d09ba726f41d9547117e876af81bfc672e33c71442eb05675d9552df1b313d1f9934f9ddd08955fa21d6edf23000a277f6f149591299a0a96032861ecdc96bb76afa05a2bffb445d61dc891bc70c13695920b911cad0df3fa842a3e2318c57556974343f69794cb8fa18c1ad624835857e4781" +
                        "041198aa705c4d11f3ef82e941be2aee7a770e54521312fe6facbaf1138eee08fa90fae986a5d93719aeb30ac292a49c1d91bf4574d553a92a4a6c305ab09db6bbeffd84c7aa707f1c1628a0220d6ba4ee5e960566686228a6e766d8a30dddf30ed5aa637c949950c3d0e894a7560670b6879a7d70f3c7e5ab29aed236cc3527bdea076fec8add12" +
                        "d784fbcf9a",
                    MD: "68f62c418a6b97026cc70f6abf8419b671ee373709fa13074e37bd39f0a50fcb");
            }
        }
    }
}
