// Licensed to the .NET Foundation under one or more agreements.
// The .NET Foundation licenses this file to you under the MIT license.

using System.Collections.Generic;
using System.IO;
using System.Linq;
using System.Threading;
using System.Threading.Tasks;
using Xunit;

namespace System.Security.Cryptography.Tests
{
    public class KmacXof128Tests : KmacTestDriver<KmacXof128Tests.Traits, KmacXof128>
    {
        public class Traits : IKmacTrait<KmacXof128>
        {
            public static KmacXof128 Create(byte[] key, byte[] customizationString) =>
                new KmacXof128(key, customizationString);

            public static KmacXof128 Create(ReadOnlySpan<byte> key, ReadOnlySpan<byte> customizationString) =>
                new KmacXof128(key, customizationString);

            public static bool IsSupported => KmacXof128.IsSupported;

            public static void AppendData(KmacXof128 kmac, byte[] data) => kmac.AppendData(data);
            public static void AppendData(KmacXof128 kmac, ReadOnlySpan<byte> data) => kmac.AppendData(data);
            public static byte[] GetHashAndReset(KmacXof128 kmac, int outputLength) => kmac.GetHashAndReset(outputLength);
            public static void GetHashAndReset(KmacXof128 kmac, Span<byte> destination) => kmac.GetHashAndReset(destination);
            public static byte[] GetCurrentHash(KmacXof128 kmac, int outputLength) => kmac.GetCurrentHash(outputLength);
            public static void GetCurrentHash(KmacXof128 kmac, Span<byte> destination) => kmac.GetCurrentHash(destination);
            public static KmacXof128 Clone(KmacXof128 kmac) => kmac.Clone();

            public static byte[] HashData(byte[] key, byte[] source, int outputLength, byte[] customizationString) =>
                KmacXof128.HashData(key, source, outputLength, customizationString);

            public static byte[] HashData(ReadOnlySpan<byte> key, ReadOnlySpan<byte> source, int outputLength, ReadOnlySpan<byte> customizationString) =>
                KmacXof128.HashData(key, source, outputLength, customizationString);

            public static void HashData(ReadOnlySpan<byte> key, ReadOnlySpan<byte> source, Span<byte> destination, ReadOnlySpan<byte> customizationString) =>
                KmacXof128.HashData(key, source, destination, customizationString);

            public static byte[] HashData(byte[] key, Stream source, int outputLength, byte[] customizationString) =>
                KmacXof128.HashData(key, source, outputLength, customizationString);

            public static byte[] HashData(ReadOnlySpan<byte> key, Stream source, int outputLength, ReadOnlySpan<byte> customizationString) =>
                KmacXof128.HashData(key, source, outputLength, customizationString);

            public static void HashData(ReadOnlySpan<byte> key, Stream source, Span<byte> destination, ReadOnlySpan<byte> customizationString) =>
                KmacXof128.HashData(key, source, destination, customizationString);

            public static ValueTask HashDataAsync(
                ReadOnlyMemory<byte> key,
                Stream source,
                Memory<byte> destination,
                ReadOnlyMemory<byte> customizationString,
                CancellationToken cancellationToken)
            {
                return KmacXof128.HashDataAsync(key, source, destination, customizationString, cancellationToken);
            }

            public static ValueTask<byte[]> HashDataAsync(
                ReadOnlyMemory<byte> key,
                Stream source,
                int outputLength,
                ReadOnlyMemory<byte> customizationString,
                CancellationToken cancellationToken)
            {
                return KmacXof128.HashDataAsync(key, source, outputLength, customizationString, cancellationToken);
            }

            public static ValueTask<byte[]> HashDataAsync(
                byte[] key,
                Stream source,
                int outputLength,
                byte[] customizationString,
                CancellationToken cancellationToken)
            {
                return KmacXof128.HashDataAsync(key, source, outputLength, customizationString, cancellationToken);
            }
        }

        protected override IEnumerable<KmacTestVector> TestVectors
        {
            get
            {
                // Generated test vectors.
                // See KmacTestDriver for how they were generated.
                yield return new(Key: "01020304", Msg: "", Custom: "", Mac: "1e");
                yield return new(Key: "01020304", Msg: "68656c6c6f", Custom: "", Mac: "9c");
                yield return new(Key: "01020304", Msg: "676f6f64627965", Custom: "", Mac: "08");
                yield return new(Key: "01020304", Msg: "", Custom: "0102030405060708", Mac: "d1");
                yield return new(Key: "01020304", Msg: "68656c6c6f", Custom: "0102030405060708", Mac: "b6");
                yield return new(Key: "01020304", Msg: "676f6f64627965", Custom: "0102030405060708", Mac: "fb");
                yield return new(Key: "01020304", Msg: "", Custom: "", Mac: "1edf5ff7778a07ca");
                yield return new(Key: "01020304", Msg: "68656c6c6f", Custom: "", Mac: "9cbce8cc6b7d8277");
                yield return new(Key: "01020304", Msg: "676f6f64627965", Custom: "", Mac: "084eef7e8e172654");
                yield return new(Key: "01020304", Msg: "", Custom: "0102030405060708", Mac: "d1ee285714138a2c");
                yield return new(Key: "01020304", Msg: "68656c6c6f", Custom: "0102030405060708", Mac: "b6abf2e9f743fc0b");
                yield return new(Key: "01020304", Msg: "676f6f64627965", Custom: "0102030405060708", Mac: "fb62ef385a7ba6b0");
                yield return new(Key: "01020304", Msg: "", Custom: "", Mac: "1edf5ff7778a07ca599f5b27f73ecc2bd1f0ef9bd15a791d");
                yield return new(Key: "01020304", Msg: "68656c6c6f", Custom: "", Mac: "9cbce8cc6b7d8277fccb402159222bd6f492a3aa02aed565");
                yield return new(Key: "01020304", Msg: "676f6f64627965", Custom: "", Mac: "084eef7e8e172654bd90eba7c90adf965df0f11629011d3c");
                yield return new(Key: "01020304", Msg: "", Custom: "0102030405060708", Mac: "d1ee285714138a2c7a852777da9e26cd2b764f298a35579e");
                yield return new(Key: "01020304", Msg: "68656c6c6f", Custom: "0102030405060708", Mac: "b6abf2e9f743fc0b991860c66cd397205ffc345865e6641c");
                yield return new(Key: "01020304", Msg: "676f6f64627965", Custom: "0102030405060708", Mac: "fb62ef385a7ba6b0c266eac9ccf53605fb951890b9110b36");
                yield return new(Key: "01020304", Msg: "", Custom: "", Mac: "1edf5ff7778a07ca599f5b27f73ecc2bd1f0ef9bd15a791df1a7e847da574000a439724d132131bf");
                yield return new(Key: "01020304", Msg: "68656c6c6f", Custom: "", Mac: "9cbce8cc6b7d8277fccb402159222bd6f492a3aa02aed56568505a59e40b2839c38bafbfe0deef14");
                yield return new(Key: "01020304", Msg: "676f6f64627965", Custom: "", Mac: "084eef7e8e172654bd90eba7c90adf965df0f11629011d3c8dd7fcbfc8d9976c7bccad09730053c8");
                yield return new(Key: "01020304", Msg: "", Custom: "0102030405060708", Mac: "d1ee285714138a2c7a852777da9e26cd2b764f298a35579edab554e4fcf02e33d5043ecc883dd950");
                yield return new(Key: "01020304", Msg: "68656c6c6f", Custom: "0102030405060708", Mac: "b6abf2e9f743fc0b991860c66cd397205ffc345865e6641cafcef1b8a44df9cdacebc83765e30649");
                yield return new(Key: "01020304", Msg: "676f6f64627965", Custom: "0102030405060708", Mac: "fb62ef385a7ba6b0c266eac9ccf53605fb951890b9110b36577fbecf53113b1d077e7ab32abf07ba");
                yield return new(Key: "01020304", Msg: "", Custom: "", Mac: "1edf5ff7778a07ca599f5b27f73ecc2bd1f0ef9bd15a791df1a7e847da574000a439724d132131bfab8b63fcd8165be710b3b51816b53d94");
                yield return new(Key: "01020304", Msg: "68656c6c6f", Custom: "", Mac: "9cbce8cc6b7d8277fccb402159222bd6f492a3aa02aed56568505a59e40b2839c38bafbfe0deef1487ed6dba36acdd28d1df39262dc2dcdd");
                yield return new(Key: "01020304", Msg: "676f6f64627965", Custom: "", Mac: "084eef7e8e172654bd90eba7c90adf965df0f11629011d3c8dd7fcbfc8d9976c7bccad09730053c89c1b8f9b8f2f670d96f7ffe6dbfe19e8");
                yield return new(Key: "01020304", Msg: "", Custom: "0102030405060708", Mac: "d1ee285714138a2c7a852777da9e26cd2b764f298a35579edab554e4fcf02e33d5043ecc883dd9504c5ebd2a46e991adc2df5b7a1a989728");
                yield return new(Key: "01020304", Msg: "68656c6c6f", Custom: "0102030405060708", Mac: "b6abf2e9f743fc0b991860c66cd397205ffc345865e6641cafcef1b8a44df9cdacebc83765e3064976df2372cd5a6b71d863744d7b595a62");
                yield return new(Key: "01020304", Msg: "676f6f64627965", Custom: "0102030405060708", Mac: "fb62ef385a7ba6b0c266eac9ccf53605fb951890b9110b36577fbecf53113b1d077e7ab32abf07ba6e131612a2ea25105e2e4ad9e751e5a5");
                yield return new(Key: "01020304", Msg: "", Custom: "", Mac: "1edf5ff7778a07ca599f5b27f73ecc2bd1f0ef9bd15a791df1a7e847da574000a439724d132131bfab8b63fcd8165be710b3b51816b53d940b4ee6735c3fd40602d9206de8216cd7");
                yield return new(Key: "01020304", Msg: "68656c6c6f", Custom: "", Mac: "9cbce8cc6b7d8277fccb402159222bd6f492a3aa02aed56568505a59e40b2839c38bafbfe0deef1487ed6dba36acdd28d1df39262dc2dcddca7bb21a1574936a9f30738d01eede3d");
                yield return new(Key: "01020304", Msg: "676f6f64627965", Custom: "", Mac: "084eef7e8e172654bd90eba7c90adf965df0f11629011d3c8dd7fcbfc8d9976c7bccad09730053c89c1b8f9b8f2f670d96f7ffe6dbfe19e8e6931adcaa6845bd9752c7c1aa418ae8");
                yield return new(Key: "01020304", Msg: "", Custom: "0102030405060708", Mac: "d1ee285714138a2c7a852777da9e26cd2b764f298a35579edab554e4fcf02e33d5043ecc883dd9504c5ebd2a46e991adc2df5b7a1a9897286f05dbefd7e98a87b5c0cbbf0d233dbc");
                yield return new(Key: "01020304", Msg: "68656c6c6f", Custom: "0102030405060708", Mac: "b6abf2e9f743fc0b991860c66cd397205ffc345865e6641cafcef1b8a44df9cdacebc83765e3064976df2372cd5a6b71d863744d7b595a624f0ec6733e6ad1bc086ab5e1731501bb");
                yield return new(Key: "01020304", Msg: "676f6f64627965", Custom: "0102030405060708", Mac: "fb62ef385a7ba6b0c266eac9ccf53605fb951890b9110b36577fbecf53113b1d077e7ab32abf07ba6e131612a2ea25105e2e4ad9e751e5a593ebd517c8243301a128737e077a2feb");
                yield return new(Key: "01020304", Msg: "", Custom: "", Mac: "1edf5ff7778a07ca599f5b27f73ecc2bd1f0ef9bd15a791df1a7e847da574000a439724d132131bfab8b63fcd8165be710b3b51816b53d940b4ee6735c3fd40602d9206de8216cd7d8465911029a8a4a3c29bcd10e230d0f");
                yield return new(Key: "01020304", Msg: "68656c6c6f", Custom: "", Mac: "9cbce8cc6b7d8277fccb402159222bd6f492a3aa02aed56568505a59e40b2839c38bafbfe0deef1487ed6dba36acdd28d1df39262dc2dcddca7bb21a1574936a9f30738d01eede3dfd5825092b9c0366069ed975282d9d85");
                yield return new(Key: "01020304", Msg: "676f6f64627965", Custom: "", Mac: "084eef7e8e172654bd90eba7c90adf965df0f11629011d3c8dd7fcbfc8d9976c7bccad09730053c89c1b8f9b8f2f670d96f7ffe6dbfe19e8e6931adcaa6845bd9752c7c1aa418ae8646d552a05044ef1de8c48f9f813a06c");
                yield return new(Key: "01020304", Msg: "", Custom: "0102030405060708", Mac: "d1ee285714138a2c7a852777da9e26cd2b764f298a35579edab554e4fcf02e33d5043ecc883dd9504c5ebd2a46e991adc2df5b7a1a9897286f05dbefd7e98a87b5c0cbbf0d233dbc44f21f502f6cb103830f77cc67ffc90d");
                yield return new(Key: "01020304", Msg: "68656c6c6f", Custom: "0102030405060708", Mac: "b6abf2e9f743fc0b991860c66cd397205ffc345865e6641cafcef1b8a44df9cdacebc83765e3064976df2372cd5a6b71d863744d7b595a624f0ec6733e6ad1bc086ab5e1731501bb36697de81dd430d7c25968fd54a6255c");
                yield return new(Key: "01020304", Msg: "676f6f64627965", Custom: "0102030405060708", Mac: "fb62ef385a7ba6b0c266eac9ccf53605fb951890b9110b36577fbecf53113b1d077e7ab32abf07ba6e131612a2ea25105e2e4ad9e751e5a593ebd517c8243301a128737e077a2feba56795b1ba84017af250aa26c4e91f72");
                yield return new(Key: "01020304", Msg: "", Custom: "", Mac: "1edf5ff7778a07ca599f5b27f73ecc2bd1f0ef9bd15a791df1a7e847da574000a439724d132131bfab8b63fcd8165be710b3b51816b53d940b4ee6735c3fd40602d9206de8216cd7d8465911029a8a4a3c29bcd10e230d0f271dfb12f20ede83f8d80b6d23be9a70");
                yield return new(Key: "01020304", Msg: "68656c6c6f", Custom: "", Mac: "9cbce8cc6b7d8277fccb402159222bd6f492a3aa02aed56568505a59e40b2839c38bafbfe0deef1487ed6dba36acdd28d1df39262dc2dcddca7bb21a1574936a9f30738d01eede3dfd5825092b9c0366069ed975282d9d85c7355dbd56f0f0cdffea6221221032a1");
                yield return new(Key: "01020304", Msg: "676f6f64627965", Custom: "", Mac: "084eef7e8e172654bd90eba7c90adf965df0f11629011d3c8dd7fcbfc8d9976c7bccad09730053c89c1b8f9b8f2f670d96f7ffe6dbfe19e8e6931adcaa6845bd9752c7c1aa418ae8646d552a05044ef1de8c48f9f813a06c57336c9d5978d4a133a029b2e31ab7bb");
                yield return new(Key: "01020304", Msg: "", Custom: "0102030405060708", Mac: "d1ee285714138a2c7a852777da9e26cd2b764f298a35579edab554e4fcf02e33d5043ecc883dd9504c5ebd2a46e991adc2df5b7a1a9897286f05dbefd7e98a87b5c0cbbf0d233dbc44f21f502f6cb103830f77cc67ffc90d1604a8694492a57a9b5d24804bbe4e37");
                yield return new(Key: "01020304", Msg: "68656c6c6f", Custom: "0102030405060708", Mac: "b6abf2e9f743fc0b991860c66cd397205ffc345865e6641cafcef1b8a44df9cdacebc83765e3064976df2372cd5a6b71d863744d7b595a624f0ec6733e6ad1bc086ab5e1731501bb36697de81dd430d7c25968fd54a6255c191c50323707918c02217508dbd599fa");
                yield return new(Key: "01020304", Msg: "676f6f64627965", Custom: "0102030405060708", Mac: "fb62ef385a7ba6b0c266eac9ccf53605fb951890b9110b36577fbecf53113b1d077e7ab32abf07ba6e131612a2ea25105e2e4ad9e751e5a593ebd517c8243301a128737e077a2feba56795b1ba84017af250aa26c4e91f7262a86b95c618157ea3b07d58a680a02c");
                yield return new(Key: "01020304", Msg: "", Custom: "", Mac: "1edf5ff7778a07ca599f5b27f73ecc2bd1f0ef9bd15a791df1a7e847da574000a439724d132131bfab8b63fcd8165be710b3b51816b53d940b4ee6735c3fd40602d9206de8216cd7d8465911029a8a4a3c29bcd10e230d0f271dfb12f20ede83f8d80b6d23be9a7071bde1b4a0c1a297d35828d43a7a693a");
                yield return new(Key: "01020304", Msg: "68656c6c6f", Custom: "", Mac: "9cbce8cc6b7d8277fccb402159222bd6f492a3aa02aed56568505a59e40b2839c38bafbfe0deef1487ed6dba36acdd28d1df39262dc2dcddca7bb21a1574936a9f30738d01eede3dfd5825092b9c0366069ed975282d9d85c7355dbd56f0f0cdffea6221221032a1dcd779ad2a4a00a0cd69aba3d9581cec");
                yield return new(Key: "01020304", Msg: "676f6f64627965", Custom: "", Mac: "084eef7e8e172654bd90eba7c90adf965df0f11629011d3c8dd7fcbfc8d9976c7bccad09730053c89c1b8f9b8f2f670d96f7ffe6dbfe19e8e6931adcaa6845bd9752c7c1aa418ae8646d552a05044ef1de8c48f9f813a06c57336c9d5978d4a133a029b2e31ab7bbe38294e300f7c99f5a651fdc05d641d3");
                yield return new(Key: "01020304", Msg: "", Custom: "0102030405060708", Mac: "d1ee285714138a2c7a852777da9e26cd2b764f298a35579edab554e4fcf02e33d5043ecc883dd9504c5ebd2a46e991adc2df5b7a1a9897286f05dbefd7e98a87b5c0cbbf0d233dbc44f21f502f6cb103830f77cc67ffc90d1604a8694492a57a9b5d24804bbe4e372389db81ef9de862845631807fe2eee1");
                yield return new(Key: "01020304", Msg: "68656c6c6f", Custom: "0102030405060708", Mac: "b6abf2e9f743fc0b991860c66cd397205ffc345865e6641cafcef1b8a44df9cdacebc83765e3064976df2372cd5a6b71d863744d7b595a624f0ec6733e6ad1bc086ab5e1731501bb36697de81dd430d7c25968fd54a6255c191c50323707918c02217508dbd599fa892c8782cabb7127e069e101375b8078");
                yield return new(Key: "01020304", Msg: "676f6f64627965", Custom: "0102030405060708", Mac: "fb62ef385a7ba6b0c266eac9ccf53605fb951890b9110b36577fbecf53113b1d077e7ab32abf07ba6e131612a2ea25105e2e4ad9e751e5a593ebd517c8243301a128737e077a2feba56795b1ba84017af250aa26c4e91f7262a86b95c618157ea3b07d58a680a02c378205de5874eb5560830b499651b996");
                yield return new(Key: "01020304", Msg: "", Custom: "", Mac: "1edf5ff7778a07ca599f5b27f73ecc2bd1f0ef9bd15a791df1a7e847da574000a439724d132131bfab8b63fcd8165be710b3b51816b53d940b4ee6735c3fd40602d9206de8216cd7d8465911029a8a4a3c29bcd10e230d0f271dfb12f20ede83f8d80b6d23be9a7071bde1b4a0c1a297d35828d43a7a693a8a2da4fdd147c02cf9ff9a5416eda686");
                yield return new(Key: "01020304", Msg: "68656c6c6f", Custom: "", Mac: "9cbce8cc6b7d8277fccb402159222bd6f492a3aa02aed56568505a59e40b2839c38bafbfe0deef1487ed6dba36acdd28d1df39262dc2dcddca7bb21a1574936a9f30738d01eede3dfd5825092b9c0366069ed975282d9d85c7355dbd56f0f0cdffea6221221032a1dcd779ad2a4a00a0cd69aba3d9581cec6cace1ed107dbbcd22d14c05eef6c971");
                yield return new(Key: "01020304", Msg: "676f6f64627965", Custom: "", Mac: "084eef7e8e172654bd90eba7c90adf965df0f11629011d3c8dd7fcbfc8d9976c7bccad09730053c89c1b8f9b8f2f670d96f7ffe6dbfe19e8e6931adcaa6845bd9752c7c1aa418ae8646d552a05044ef1de8c48f9f813a06c57336c9d5978d4a133a029b2e31ab7bbe38294e300f7c99f5a651fdc05d641d3583c48dfab3723f29404e17ca84cec57");
                yield return new(Key: "01020304", Msg: "", Custom: "0102030405060708", Mac: "d1ee285714138a2c7a852777da9e26cd2b764f298a35579edab554e4fcf02e33d5043ecc883dd9504c5ebd2a46e991adc2df5b7a1a9897286f05dbefd7e98a87b5c0cbbf0d233dbc44f21f502f6cb103830f77cc67ffc90d1604a8694492a57a9b5d24804bbe4e372389db81ef9de862845631807fe2eee1b77ab885c5f7e3e05e4d7bbedbb7c826");
                yield return new(Key: "01020304", Msg: "68656c6c6f", Custom: "0102030405060708", Mac: "b6abf2e9f743fc0b991860c66cd397205ffc345865e6641cafcef1b8a44df9cdacebc83765e3064976df2372cd5a6b71d863744d7b595a624f0ec6733e6ad1bc086ab5e1731501bb36697de81dd430d7c25968fd54a6255c191c50323707918c02217508dbd599fa892c8782cabb7127e069e101375b8078dde824debbf2c73b9e13930a8aebdaf4");
                yield return new(Key: "01020304", Msg: "676f6f64627965", Custom: "0102030405060708", Mac: "fb62ef385a7ba6b0c266eac9ccf53605fb951890b9110b36577fbecf53113b1d077e7ab32abf07ba6e131612a2ea25105e2e4ad9e751e5a593ebd517c8243301a128737e077a2feba56795b1ba84017af250aa26c4e91f7262a86b95c618157ea3b07d58a680a02c378205de5874eb5560830b499651b99601f11d7c4a73641708e4a0a9290d7d18");
                yield return new(Key: "01020304", Msg: "", Custom: "", Mac: "1edf5ff7778a07ca599f5b27f73ecc2bd1f0ef9bd15a791df1a7e847da574000a439724d132131bfab8b63fcd8165be710b3b51816b53d940b4ee6735c3fd40602d9206de8216cd7d8465911029a8a4a3c29bcd10e230d0f271dfb12f20ede83f8d80b6d23be9a7071bde1b4a0c1a297d35828d43a7a693a8a2da4fdd147c02cf9ff9a5416eda686a3c8919ea6f7f0a0f24b236336add669");
                yield return new(Key: "01020304", Msg: "68656c6c6f", Custom: "", Mac: "9cbce8cc6b7d8277fccb402159222bd6f492a3aa02aed56568505a59e40b2839c38bafbfe0deef1487ed6dba36acdd28d1df39262dc2dcddca7bb21a1574936a9f30738d01eede3dfd5825092b9c0366069ed975282d9d85c7355dbd56f0f0cdffea6221221032a1dcd779ad2a4a00a0cd69aba3d9581cec6cace1ed107dbbcd22d14c05eef6c971e63f40787bc155004b55b0690f9b8dea");
                yield return new(Key: "01020304", Msg: "676f6f64627965", Custom: "", Mac: "084eef7e8e172654bd90eba7c90adf965df0f11629011d3c8dd7fcbfc8d9976c7bccad09730053c89c1b8f9b8f2f670d96f7ffe6dbfe19e8e6931adcaa6845bd9752c7c1aa418ae8646d552a05044ef1de8c48f9f813a06c57336c9d5978d4a133a029b2e31ab7bbe38294e300f7c99f5a651fdc05d641d3583c48dfab3723f29404e17ca84cec573ea183fb79455ee423238a2da0ca361f");
                yield return new(Key: "01020304", Msg: "", Custom: "0102030405060708", Mac: "d1ee285714138a2c7a852777da9e26cd2b764f298a35579edab554e4fcf02e33d5043ecc883dd9504c5ebd2a46e991adc2df5b7a1a9897286f05dbefd7e98a87b5c0cbbf0d233dbc44f21f502f6cb103830f77cc67ffc90d1604a8694492a57a9b5d24804bbe4e372389db81ef9de862845631807fe2eee1b77ab885c5f7e3e05e4d7bbedbb7c826afdab20663d9c7e1cce8ba532173c5e4");
                yield return new(Key: "01020304", Msg: "68656c6c6f", Custom: "0102030405060708", Mac: "b6abf2e9f743fc0b991860c66cd397205ffc345865e6641cafcef1b8a44df9cdacebc83765e3064976df2372cd5a6b71d863744d7b595a624f0ec6733e6ad1bc086ab5e1731501bb36697de81dd430d7c25968fd54a6255c191c50323707918c02217508dbd599fa892c8782cabb7127e069e101375b8078dde824debbf2c73b9e13930a8aebdaf4c6b2fd7a2e9a2deff121b87d805dbad2");
                yield return new(Key: "01020304", Msg: "676f6f64627965", Custom: "0102030405060708", Mac: "fb62ef385a7ba6b0c266eac9ccf53605fb951890b9110b36577fbecf53113b1d077e7ab32abf07ba6e131612a2ea25105e2e4ad9e751e5a593ebd517c8243301a128737e077a2feba56795b1ba84017af250aa26c4e91f7262a86b95c618157ea3b07d58a680a02c378205de5874eb5560830b499651b99601f11d7c4a73641708e4a0a9290d7d18b1f4bf9dd5544c06f77b912dfc480ba0");
                yield return new(Key: "01020304", Msg: "", Custom: "", Mac: "1edf5ff7778a07ca599f5b27f73ecc2bd1f0ef9bd15a791df1a7e847da574000a439724d132131bfab8b63fcd8165be710b3b51816b53d940b4ee6735c3fd40602d9206de8216cd7d8465911029a8a4a3c29bcd10e230d0f271dfb12f20ede83f8d80b6d23be9a7071bde1b4a0c1a297d35828d43a7a693a8a2da4fdd147c02cf9ff9a5416eda686a3c8919ea6f7f0a0f24b236336add66998ca36b5c174844a3532ee01afb9365e");
                yield return new(Key: "01020304", Msg: "68656c6c6f", Custom: "", Mac: "9cbce8cc6b7d8277fccb402159222bd6f492a3aa02aed56568505a59e40b2839c38bafbfe0deef1487ed6dba36acdd28d1df39262dc2dcddca7bb21a1574936a9f30738d01eede3dfd5825092b9c0366069ed975282d9d85c7355dbd56f0f0cdffea6221221032a1dcd779ad2a4a00a0cd69aba3d9581cec6cace1ed107dbbcd22d14c05eef6c971e63f40787bc155004b55b0690f9b8dea3c8412751c2e6b7df878b0f0d862e02f");
                yield return new(Key: "01020304", Msg: "676f6f64627965", Custom: "", Mac: "084eef7e8e172654bd90eba7c90adf965df0f11629011d3c8dd7fcbfc8d9976c7bccad09730053c89c1b8f9b8f2f670d96f7ffe6dbfe19e8e6931adcaa6845bd9752c7c1aa418ae8646d552a05044ef1de8c48f9f813a06c57336c9d5978d4a133a029b2e31ab7bbe38294e300f7c99f5a651fdc05d641d3583c48dfab3723f29404e17ca84cec573ea183fb79455ee423238a2da0ca361f56d716a2a1805835b33a628ed458c9d6");
                yield return new(Key: "01020304", Msg: "", Custom: "0102030405060708", Mac: "d1ee285714138a2c7a852777da9e26cd2b764f298a35579edab554e4fcf02e33d5043ecc883dd9504c5ebd2a46e991adc2df5b7a1a9897286f05dbefd7e98a87b5c0cbbf0d233dbc44f21f502f6cb103830f77cc67ffc90d1604a8694492a57a9b5d24804bbe4e372389db81ef9de862845631807fe2eee1b77ab885c5f7e3e05e4d7bbedbb7c826afdab20663d9c7e1cce8ba532173c5e4b34283ff3e62a61257d54f24583f073e");
                yield return new(Key: "01020304", Msg: "68656c6c6f", Custom: "0102030405060708", Mac: "b6abf2e9f743fc0b991860c66cd397205ffc345865e6641cafcef1b8a44df9cdacebc83765e3064976df2372cd5a6b71d863744d7b595a624f0ec6733e6ad1bc086ab5e1731501bb36697de81dd430d7c25968fd54a6255c191c50323707918c02217508dbd599fa892c8782cabb7127e069e101375b8078dde824debbf2c73b9e13930a8aebdaf4c6b2fd7a2e9a2deff121b87d805dbad2ae089312eff200f4cc881dc78082a8c9");
                yield return new(Key: "01020304", Msg: "676f6f64627965", Custom: "0102030405060708", Mac: "fb62ef385a7ba6b0c266eac9ccf53605fb951890b9110b36577fbecf53113b1d077e7ab32abf07ba6e131612a2ea25105e2e4ad9e751e5a593ebd517c8243301a128737e077a2feba56795b1ba84017af250aa26c4e91f7262a86b95c618157ea3b07d58a680a02c378205de5874eb5560830b499651b99601f11d7c4a73641708e4a0a9290d7d18b1f4bf9dd5544c06f77b912dfc480ba0f01429700bc8da286b17c56b6441630d");
                yield return new(Key: "01020304", Msg: "", Custom: "", Mac: "1edf5ff7778a07ca599f5b27f73ecc2bd1f0ef9bd15a791df1a7e847da574000a439724d132131bfab8b63fcd8165be710b3b51816b53d940b4ee6735c3fd40602d9206de8216cd7d8465911029a8a4a3c29bcd10e230d0f271dfb12f20ede83f8d80b6d23be9a7071bde1b4a0c1a297d35828d43a7a693a8a2da4fdd147c02cf9ff9a5416eda686a3c8919ea6f7f0a0f24b236336add66998ca36b5c174844a3532ee01afb9365e432f4f913c98298e28511e948b31d5da");
                yield return new(Key: "01020304", Msg: "68656c6c6f", Custom: "", Mac: "9cbce8cc6b7d8277fccb402159222bd6f492a3aa02aed56568505a59e40b2839c38bafbfe0deef1487ed6dba36acdd28d1df39262dc2dcddca7bb21a1574936a9f30738d01eede3dfd5825092b9c0366069ed975282d9d85c7355dbd56f0f0cdffea6221221032a1dcd779ad2a4a00a0cd69aba3d9581cec6cace1ed107dbbcd22d14c05eef6c971e63f40787bc155004b55b0690f9b8dea3c8412751c2e6b7df878b0f0d862e02f0a73b0ab1b4a47205e764e023d2ece38");
                yield return new(Key: "01020304", Msg: "676f6f64627965", Custom: "", Mac: "084eef7e8e172654bd90eba7c90adf965df0f11629011d3c8dd7fcbfc8d9976c7bccad09730053c89c1b8f9b8f2f670d96f7ffe6dbfe19e8e6931adcaa6845bd9752c7c1aa418ae8646d552a05044ef1de8c48f9f813a06c57336c9d5978d4a133a029b2e31ab7bbe38294e300f7c99f5a651fdc05d641d3583c48dfab3723f29404e17ca84cec573ea183fb79455ee423238a2da0ca361f56d716a2a1805835b33a628ed458c9d61e21393f1a454d0916cd9c7f41cbb7d0");
                yield return new(Key: "01020304", Msg: "", Custom: "0102030405060708", Mac: "d1ee285714138a2c7a852777da9e26cd2b764f298a35579edab554e4fcf02e33d5043ecc883dd9504c5ebd2a46e991adc2df5b7a1a9897286f05dbefd7e98a87b5c0cbbf0d233dbc44f21f502f6cb103830f77cc67ffc90d1604a8694492a57a9b5d24804bbe4e372389db81ef9de862845631807fe2eee1b77ab885c5f7e3e05e4d7bbedbb7c826afdab20663d9c7e1cce8ba532173c5e4b34283ff3e62a61257d54f24583f073efe635022ac9608e5ab8ee25b550a1cbb");
                yield return new(Key: "01020304", Msg: "68656c6c6f", Custom: "0102030405060708", Mac: "b6abf2e9f743fc0b991860c66cd397205ffc345865e6641cafcef1b8a44df9cdacebc83765e3064976df2372cd5a6b71d863744d7b595a624f0ec6733e6ad1bc086ab5e1731501bb36697de81dd430d7c25968fd54a6255c191c50323707918c02217508dbd599fa892c8782cabb7127e069e101375b8078dde824debbf2c73b9e13930a8aebdaf4c6b2fd7a2e9a2deff121b87d805dbad2ae089312eff200f4cc881dc78082a8c93b7d137b988e254e346358eeefe52fae");
                yield return new(Key: "01020304", Msg: "676f6f64627965", Custom: "0102030405060708", Mac: "fb62ef385a7ba6b0c266eac9ccf53605fb951890b9110b36577fbecf53113b1d077e7ab32abf07ba6e131612a2ea25105e2e4ad9e751e5a593ebd517c8243301a128737e077a2feba56795b1ba84017af250aa26c4e91f7262a86b95c618157ea3b07d58a680a02c378205de5874eb5560830b499651b99601f11d7c4a73641708e4a0a9290d7d18b1f4bf9dd5544c06f77b912dfc480ba0f01429700bc8da286b17c56b6441630db59d841d4826d76263c004489e4ea185");
                yield return new(Key: "01020304", Msg: "", Custom: "", Mac: "1edf5ff7778a07ca599f5b27f73ecc2bd1f0ef9bd15a791df1a7e847da574000a439724d132131bfab8b63fcd8165be710b3b51816b53d940b4ee6735c3fd40602d9206de8216cd7d8465911029a8a4a3c29bcd10e230d0f271dfb12f20ede83f8d80b6d23be9a7071bde1b4a0c1a297d35828d43a7a693a8a2da4fdd147c02cf9ff9a5416eda686a3c8919ea6f7f0a0f24b236336add66998ca36b5c174844a3532ee01afb9365e432f4f913c98298e28511e948b31d5da06fc2d343505439386eb643772574f9c");
                yield return new(Key: "01020304", Msg: "68656c6c6f", Custom: "", Mac: "9cbce8cc6b7d8277fccb402159222bd6f492a3aa02aed56568505a59e40b2839c38bafbfe0deef1487ed6dba36acdd28d1df39262dc2dcddca7bb21a1574936a9f30738d01eede3dfd5825092b9c0366069ed975282d9d85c7355dbd56f0f0cdffea6221221032a1dcd779ad2a4a00a0cd69aba3d9581cec6cace1ed107dbbcd22d14c05eef6c971e63f40787bc155004b55b0690f9b8dea3c8412751c2e6b7df878b0f0d862e02f0a73b0ab1b4a47205e764e023d2ece3873c6087d53628a7ed2feef4d4bdcb522");
                yield return new(Key: "01020304", Msg: "676f6f64627965", Custom: "", Mac: "084eef7e8e172654bd90eba7c90adf965df0f11629011d3c8dd7fcbfc8d9976c7bccad09730053c89c1b8f9b8f2f670d96f7ffe6dbfe19e8e6931adcaa6845bd9752c7c1aa418ae8646d552a05044ef1de8c48f9f813a06c57336c9d5978d4a133a029b2e31ab7bbe38294e300f7c99f5a651fdc05d641d3583c48dfab3723f29404e17ca84cec573ea183fb79455ee423238a2da0ca361f56d716a2a1805835b33a628ed458c9d61e21393f1a454d0916cd9c7f41cbb7d0c9cf00188e78f7c8c42210d74ce6332c");
                yield return new(Key: "01020304", Msg: "", Custom: "0102030405060708", Mac: "d1ee285714138a2c7a852777da9e26cd2b764f298a35579edab554e4fcf02e33d5043ecc883dd9504c5ebd2a46e991adc2df5b7a1a9897286f05dbefd7e98a87b5c0cbbf0d233dbc44f21f502f6cb103830f77cc67ffc90d1604a8694492a57a9b5d24804bbe4e372389db81ef9de862845631807fe2eee1b77ab885c5f7e3e05e4d7bbedbb7c826afdab20663d9c7e1cce8ba532173c5e4b34283ff3e62a61257d54f24583f073efe635022ac9608e5ab8ee25b550a1cbb0fbf53d3ac077ea0c2733920612d52a1");
                yield return new(Key: "01020304", Msg: "68656c6c6f", Custom: "0102030405060708", Mac: "b6abf2e9f743fc0b991860c66cd397205ffc345865e6641cafcef1b8a44df9cdacebc83765e3064976df2372cd5a6b71d863744d7b595a624f0ec6733e6ad1bc086ab5e1731501bb36697de81dd430d7c25968fd54a6255c191c50323707918c02217508dbd599fa892c8782cabb7127e069e101375b8078dde824debbf2c73b9e13930a8aebdaf4c6b2fd7a2e9a2deff121b87d805dbad2ae089312eff200f4cc881dc78082a8c93b7d137b988e254e346358eeefe52fae766270101dd02e6eaaaa8b930ef7fe01");
                yield return new(Key: "01020304", Msg: "676f6f64627965", Custom: "0102030405060708", Mac: "fb62ef385a7ba6b0c266eac9ccf53605fb951890b9110b36577fbecf53113b1d077e7ab32abf07ba6e131612a2ea25105e2e4ad9e751e5a593ebd517c8243301a128737e077a2feba56795b1ba84017af250aa26c4e91f7262a86b95c618157ea3b07d58a680a02c378205de5874eb5560830b499651b99601f11d7c4a73641708e4a0a9290d7d18b1f4bf9dd5544c06f77b912dfc480ba0f01429700bc8da286b17c56b6441630db59d841d4826d76263c004489e4ea185d8a5c78a378db816fb3da1641c7bf0b7");
                yield return new(Key: "01020304", Msg: "", Custom: "", Mac: "1edf5ff7778a07ca599f5b27f73ecc2bd1f0ef9bd15a791df1a7e847da574000a439724d132131bfab8b63fcd8165be710b3b51816b53d940b4ee6735c3fd40602d9206de8216cd7d8465911029a8a4a3c29bcd10e230d0f271dfb12f20ede83f8d80b6d23be9a7071bde1b4a0c1a297d35828d43a7a693a8a2da4fdd147c02cf9ff9a5416eda686a3c8919ea6f7f0a0f24b236336add66998ca36b5c174844a3532ee01afb9365e432f4f913c98298e28511e948b31d5da06fc2d343505439386eb643772574f9c4c31f3e9be3761e528d50342c0a516e8");
                yield return new(Key: "01020304", Msg: "68656c6c6f", Custom: "", Mac: "9cbce8cc6b7d8277fccb402159222bd6f492a3aa02aed56568505a59e40b2839c38bafbfe0deef1487ed6dba36acdd28d1df39262dc2dcddca7bb21a1574936a9f30738d01eede3dfd5825092b9c0366069ed975282d9d85c7355dbd56f0f0cdffea6221221032a1dcd779ad2a4a00a0cd69aba3d9581cec6cace1ed107dbbcd22d14c05eef6c971e63f40787bc155004b55b0690f9b8dea3c8412751c2e6b7df878b0f0d862e02f0a73b0ab1b4a47205e764e023d2ece3873c6087d53628a7ed2feef4d4bdcb522dc49c5f041bed65667a58392f33b97c0");
                yield return new(Key: "01020304", Msg: "676f6f64627965", Custom: "", Mac: "084eef7e8e172654bd90eba7c90adf965df0f11629011d3c8dd7fcbfc8d9976c7bccad09730053c89c1b8f9b8f2f670d96f7ffe6dbfe19e8e6931adcaa6845bd9752c7c1aa418ae8646d552a05044ef1de8c48f9f813a06c57336c9d5978d4a133a029b2e31ab7bbe38294e300f7c99f5a651fdc05d641d3583c48dfab3723f29404e17ca84cec573ea183fb79455ee423238a2da0ca361f56d716a2a1805835b33a628ed458c9d61e21393f1a454d0916cd9c7f41cbb7d0c9cf00188e78f7c8c42210d74ce6332c18b394919039fd9a47bee98f913a7170");
                yield return new(Key: "01020304", Msg: "", Custom: "0102030405060708", Mac: "d1ee285714138a2c7a852777da9e26cd2b764f298a35579edab554e4fcf02e33d5043ecc883dd9504c5ebd2a46e991adc2df5b7a1a9897286f05dbefd7e98a87b5c0cbbf0d233dbc44f21f502f6cb103830f77cc67ffc90d1604a8694492a57a9b5d24804bbe4e372389db81ef9de862845631807fe2eee1b77ab885c5f7e3e05e4d7bbedbb7c826afdab20663d9c7e1cce8ba532173c5e4b34283ff3e62a61257d54f24583f073efe635022ac9608e5ab8ee25b550a1cbb0fbf53d3ac077ea0c2733920612d52a199a87fd10e01f449b678958769112676");
                yield return new(Key: "01020304", Msg: "68656c6c6f", Custom: "0102030405060708", Mac: "b6abf2e9f743fc0b991860c66cd397205ffc345865e6641cafcef1b8a44df9cdacebc83765e3064976df2372cd5a6b71d863744d7b595a624f0ec6733e6ad1bc086ab5e1731501bb36697de81dd430d7c25968fd54a6255c191c50323707918c02217508dbd599fa892c8782cabb7127e069e101375b8078dde824debbf2c73b9e13930a8aebdaf4c6b2fd7a2e9a2deff121b87d805dbad2ae089312eff200f4cc881dc78082a8c93b7d137b988e254e346358eeefe52fae766270101dd02e6eaaaa8b930ef7fe017359177b2b2d3642e80b33cf1ce43883");
                yield return new(Key: "01020304", Msg: "676f6f64627965", Custom: "0102030405060708", Mac: "fb62ef385a7ba6b0c266eac9ccf53605fb951890b9110b36577fbecf53113b1d077e7ab32abf07ba6e131612a2ea25105e2e4ad9e751e5a593ebd517c8243301a128737e077a2feba56795b1ba84017af250aa26c4e91f7262a86b95c618157ea3b07d58a680a02c378205de5874eb5560830b499651b99601f11d7c4a73641708e4a0a9290d7d18b1f4bf9dd5544c06f77b912dfc480ba0f01429700bc8da286b17c56b6441630db59d841d4826d76263c004489e4ea185d8a5c78a378db816fb3da1641c7bf0b7d279c15c866e5fad9716c6fea7ec7a03");
                yield return new(Key: "01020304", Msg: "", Custom: "", Mac: "1edf5ff7778a07ca599f5b27f73ecc2bd1f0ef9bd15a791df1a7e847da574000a439724d132131bfab8b63fcd8165be710b3b51816b53d940b4ee6735c3fd40602d9206de8216cd7d8465911029a8a4a3c29bcd10e230d0f271dfb12f20ede83f8d80b6d23be9a7071bde1b4a0c1a297d35828d43a7a693a8a2da4fdd147c02cf9ff9a5416eda686a3c8919ea6f7f0a0f24b236336add66998ca36b5c174844a3532ee01afb9365e432f4f913c98298e28511e948b31d5da06fc2d343505439386eb643772574f9c4c31f3e9be3761e528d50342c0a516e8778ff9dfb004bf18286ebe347400d894");
                yield return new(Key: "01020304", Msg: "68656c6c6f", Custom: "", Mac: "9cbce8cc6b7d8277fccb402159222bd6f492a3aa02aed56568505a59e40b2839c38bafbfe0deef1487ed6dba36acdd28d1df39262dc2dcddca7bb21a1574936a9f30738d01eede3dfd5825092b9c0366069ed975282d9d85c7355dbd56f0f0cdffea6221221032a1dcd779ad2a4a00a0cd69aba3d9581cec6cace1ed107dbbcd22d14c05eef6c971e63f40787bc155004b55b0690f9b8dea3c8412751c2e6b7df878b0f0d862e02f0a73b0ab1b4a47205e764e023d2ece3873c6087d53628a7ed2feef4d4bdcb522dc49c5f041bed65667a58392f33b97c06120091aee5fbadce2b232183d61314d");
                yield return new(Key: "01020304", Msg: "676f6f64627965", Custom: "", Mac: "084eef7e8e172654bd90eba7c90adf965df0f11629011d3c8dd7fcbfc8d9976c7bccad09730053c89c1b8f9b8f2f670d96f7ffe6dbfe19e8e6931adcaa6845bd9752c7c1aa418ae8646d552a05044ef1de8c48f9f813a06c57336c9d5978d4a133a029b2e31ab7bbe38294e300f7c99f5a651fdc05d641d3583c48dfab3723f29404e17ca84cec573ea183fb79455ee423238a2da0ca361f56d716a2a1805835b33a628ed458c9d61e21393f1a454d0916cd9c7f41cbb7d0c9cf00188e78f7c8c42210d74ce6332c18b394919039fd9a47bee98f913a71703e77926aad3375e57e96d26ee4be6e5c");
                yield return new(Key: "01020304", Msg: "", Custom: "0102030405060708", Mac: "d1ee285714138a2c7a852777da9e26cd2b764f298a35579edab554e4fcf02e33d5043ecc883dd9504c5ebd2a46e991adc2df5b7a1a9897286f05dbefd7e98a87b5c0cbbf0d233dbc44f21f502f6cb103830f77cc67ffc90d1604a8694492a57a9b5d24804bbe4e372389db81ef9de862845631807fe2eee1b77ab885c5f7e3e05e4d7bbedbb7c826afdab20663d9c7e1cce8ba532173c5e4b34283ff3e62a61257d54f24583f073efe635022ac9608e5ab8ee25b550a1cbb0fbf53d3ac077ea0c2733920612d52a199a87fd10e01f449b678958769112676103544a74733c8f0e5c380131546fe10");
                yield return new(Key: "01020304", Msg: "68656c6c6f", Custom: "0102030405060708", Mac: "b6abf2e9f743fc0b991860c66cd397205ffc345865e6641cafcef1b8a44df9cdacebc83765e3064976df2372cd5a6b71d863744d7b595a624f0ec6733e6ad1bc086ab5e1731501bb36697de81dd430d7c25968fd54a6255c191c50323707918c02217508dbd599fa892c8782cabb7127e069e101375b8078dde824debbf2c73b9e13930a8aebdaf4c6b2fd7a2e9a2deff121b87d805dbad2ae089312eff200f4cc881dc78082a8c93b7d137b988e254e346358eeefe52fae766270101dd02e6eaaaa8b930ef7fe017359177b2b2d3642e80b33cf1ce43883a9f5048f672512dfe01f900519541347");
                yield return new(Key: "01020304", Msg: "676f6f64627965", Custom: "0102030405060708", Mac: "fb62ef385a7ba6b0c266eac9ccf53605fb951890b9110b36577fbecf53113b1d077e7ab32abf07ba6e131612a2ea25105e2e4ad9e751e5a593ebd517c8243301a128737e077a2feba56795b1ba84017af250aa26c4e91f7262a86b95c618157ea3b07d58a680a02c378205de5874eb5560830b499651b99601f11d7c4a73641708e4a0a9290d7d18b1f4bf9dd5544c06f77b912dfc480ba0f01429700bc8da286b17c56b6441630db59d841d4826d76263c004489e4ea185d8a5c78a378db816fb3da1641c7bf0b7d279c15c866e5fad9716c6fea7ec7a033bfe28df758a764c0be2e19c2883e7b9");
                yield return new(Key: "01020304", Msg: "", Custom: "", Mac: "1edf5ff7778a07ca599f5b27f73ecc2bd1f0ef9bd15a791df1a7e847da574000a439724d132131bfab8b63fcd8165be710b3b51816b53d940b4ee6735c3fd40602d9206de8216cd7d8465911029a8a4a3c29bcd10e230d0f271dfb12f20ede83f8d80b6d23be9a7071bde1b4a0c1a297d35828d43a7a693a8a2da4fdd147c02cf9ff9a5416eda686a3c8919ea6f7f0a0f24b236336add66998ca36b5c174844a3532ee01afb9365e432f4f913c98298e28511e948b31d5da06fc2d343505439386eb643772574f9c4c31f3e9be3761e528d50342c0a516e8778ff9dfb004bf18286ebe347400d894c63718d7fb64492ac7ddd628b7c47bab");
                yield return new(Key: "01020304", Msg: "68656c6c6f", Custom: "", Mac: "9cbce8cc6b7d8277fccb402159222bd6f492a3aa02aed56568505a59e40b2839c38bafbfe0deef1487ed6dba36acdd28d1df39262dc2dcddca7bb21a1574936a9f30738d01eede3dfd5825092b9c0366069ed975282d9d85c7355dbd56f0f0cdffea6221221032a1dcd779ad2a4a00a0cd69aba3d9581cec6cace1ed107dbbcd22d14c05eef6c971e63f40787bc155004b55b0690f9b8dea3c8412751c2e6b7df878b0f0d862e02f0a73b0ab1b4a47205e764e023d2ece3873c6087d53628a7ed2feef4d4bdcb522dc49c5f041bed65667a58392f33b97c06120091aee5fbadce2b232183d61314d63fd4dc0ea30d851b510a304fa314ac1");
                yield return new(Key: "01020304", Msg: "676f6f64627965", Custom: "", Mac: "084eef7e8e172654bd90eba7c90adf965df0f11629011d3c8dd7fcbfc8d9976c7bccad09730053c89c1b8f9b8f2f670d96f7ffe6dbfe19e8e6931adcaa6845bd9752c7c1aa418ae8646d552a05044ef1de8c48f9f813a06c57336c9d5978d4a133a029b2e31ab7bbe38294e300f7c99f5a651fdc05d641d3583c48dfab3723f29404e17ca84cec573ea183fb79455ee423238a2da0ca361f56d716a2a1805835b33a628ed458c9d61e21393f1a454d0916cd9c7f41cbb7d0c9cf00188e78f7c8c42210d74ce6332c18b394919039fd9a47bee98f913a71703e77926aad3375e57e96d26ee4be6e5c4fc69154eceab21f98c4baa13bd8ba38");
                yield return new(Key: "01020304", Msg: "", Custom: "0102030405060708", Mac: "d1ee285714138a2c7a852777da9e26cd2b764f298a35579edab554e4fcf02e33d5043ecc883dd9504c5ebd2a46e991adc2df5b7a1a9897286f05dbefd7e98a87b5c0cbbf0d233dbc44f21f502f6cb103830f77cc67ffc90d1604a8694492a57a9b5d24804bbe4e372389db81ef9de862845631807fe2eee1b77ab885c5f7e3e05e4d7bbedbb7c826afdab20663d9c7e1cce8ba532173c5e4b34283ff3e62a61257d54f24583f073efe635022ac9608e5ab8ee25b550a1cbb0fbf53d3ac077ea0c2733920612d52a199a87fd10e01f449b678958769112676103544a74733c8f0e5c380131546fe10cce9ed2a8efc2bb7baa95800758bd62d");
                yield return new(Key: "01020304", Msg: "68656c6c6f", Custom: "0102030405060708", Mac: "b6abf2e9f743fc0b991860c66cd397205ffc345865e6641cafcef1b8a44df9cdacebc83765e3064976df2372cd5a6b71d863744d7b595a624f0ec6733e6ad1bc086ab5e1731501bb36697de81dd430d7c25968fd54a6255c191c50323707918c02217508dbd599fa892c8782cabb7127e069e101375b8078dde824debbf2c73b9e13930a8aebdaf4c6b2fd7a2e9a2deff121b87d805dbad2ae089312eff200f4cc881dc78082a8c93b7d137b988e254e346358eeefe52fae766270101dd02e6eaaaa8b930ef7fe017359177b2b2d3642e80b33cf1ce43883a9f5048f672512dfe01f9005195413475a31349e315e543a908ce8ee57e7936c");
                yield return new(Key: "01020304", Msg: "676f6f64627965", Custom: "0102030405060708", Mac: "fb62ef385a7ba6b0c266eac9ccf53605fb951890b9110b36577fbecf53113b1d077e7ab32abf07ba6e131612a2ea25105e2e4ad9e751e5a593ebd517c8243301a128737e077a2feba56795b1ba84017af250aa26c4e91f7262a86b95c618157ea3b07d58a680a02c378205de5874eb5560830b499651b99601f11d7c4a73641708e4a0a9290d7d18b1f4bf9dd5544c06f77b912dfc480ba0f01429700bc8da286b17c56b6441630db59d841d4826d76263c004489e4ea185d8a5c78a378db816fb3da1641c7bf0b7d279c15c866e5fad9716c6fea7ec7a033bfe28df758a764c0be2e19c2883e7b9b8d46c2dabb2c86ac4b3d7b165757ab0");
                yield return new(Key: "01020304", Msg: "", Custom: "", Mac: "1edf5ff7778a07ca599f5b27f73ecc2bd1f0ef9bd15a791df1a7e847da574000a439724d132131bfab8b63fcd8165be710b3b51816b53d940b4ee6735c3fd40602d9206de8216cd7d8465911029a8a4a3c29bcd10e230d0f271dfb12f20ede83f8d80b6d23be9a7071bde1b4a0c1a297d35828d43a7a693a8a2da4fdd147c02cf9ff9a5416eda686a3c8919ea6f7f0a0f24b236336add66998ca36b5c174844a3532ee01afb9365e432f4f913c98298e28511e948b31d5da06fc2d343505439386eb643772574f9c4c31f3e9be3761e528d50342c0a516e8778ff9dfb004bf18286ebe347400d894c63718d7fb64492ac7ddd628b7c47baba0b461eb5d3104554c0d9441f8ad5fe8");
                yield return new(Key: "01020304", Msg: "68656c6c6f", Custom: "", Mac: "9cbce8cc6b7d8277fccb402159222bd6f492a3aa02aed56568505a59e40b2839c38bafbfe0deef1487ed6dba36acdd28d1df39262dc2dcddca7bb21a1574936a9f30738d01eede3dfd5825092b9c0366069ed975282d9d85c7355dbd56f0f0cdffea6221221032a1dcd779ad2a4a00a0cd69aba3d9581cec6cace1ed107dbbcd22d14c05eef6c971e63f40787bc155004b55b0690f9b8dea3c8412751c2e6b7df878b0f0d862e02f0a73b0ab1b4a47205e764e023d2ece3873c6087d53628a7ed2feef4d4bdcb522dc49c5f041bed65667a58392f33b97c06120091aee5fbadce2b232183d61314d63fd4dc0ea30d851b510a304fa314ac1fdd3f29ebf1afdf1c19c4c3ddd3b93c6");
                yield return new(Key: "01020304", Msg: "676f6f64627965", Custom: "", Mac: "084eef7e8e172654bd90eba7c90adf965df0f11629011d3c8dd7fcbfc8d9976c7bccad09730053c89c1b8f9b8f2f670d96f7ffe6dbfe19e8e6931adcaa6845bd9752c7c1aa418ae8646d552a05044ef1de8c48f9f813a06c57336c9d5978d4a133a029b2e31ab7bbe38294e300f7c99f5a651fdc05d641d3583c48dfab3723f29404e17ca84cec573ea183fb79455ee423238a2da0ca361f56d716a2a1805835b33a628ed458c9d61e21393f1a454d0916cd9c7f41cbb7d0c9cf00188e78f7c8c42210d74ce6332c18b394919039fd9a47bee98f913a71703e77926aad3375e57e96d26ee4be6e5c4fc69154eceab21f98c4baa13bd8ba387ce86ddad5559999a9f01a82606d9d5a");
                yield return new(Key: "01020304", Msg: "", Custom: "0102030405060708", Mac: "d1ee285714138a2c7a852777da9e26cd2b764f298a35579edab554e4fcf02e33d5043ecc883dd9504c5ebd2a46e991adc2df5b7a1a9897286f05dbefd7e98a87b5c0cbbf0d233dbc44f21f502f6cb103830f77cc67ffc90d1604a8694492a57a9b5d24804bbe4e372389db81ef9de862845631807fe2eee1b77ab885c5f7e3e05e4d7bbedbb7c826afdab20663d9c7e1cce8ba532173c5e4b34283ff3e62a61257d54f24583f073efe635022ac9608e5ab8ee25b550a1cbb0fbf53d3ac077ea0c2733920612d52a199a87fd10e01f449b678958769112676103544a74733c8f0e5c380131546fe10cce9ed2a8efc2bb7baa95800758bd62db4a91e9f6939e4204995b57ef5f786d4");
                yield return new(Key: "01020304", Msg: "68656c6c6f", Custom: "0102030405060708", Mac: "b6abf2e9f743fc0b991860c66cd397205ffc345865e6641cafcef1b8a44df9cdacebc83765e3064976df2372cd5a6b71d863744d7b595a624f0ec6733e6ad1bc086ab5e1731501bb36697de81dd430d7c25968fd54a6255c191c50323707918c02217508dbd599fa892c8782cabb7127e069e101375b8078dde824debbf2c73b9e13930a8aebdaf4c6b2fd7a2e9a2deff121b87d805dbad2ae089312eff200f4cc881dc78082a8c93b7d137b988e254e346358eeefe52fae766270101dd02e6eaaaa8b930ef7fe017359177b2b2d3642e80b33cf1ce43883a9f5048f672512dfe01f9005195413475a31349e315e543a908ce8ee57e7936c18e8dfd19de789f3cfc426c560b5faa5");
                yield return new(Key: "01020304", Msg: "676f6f64627965", Custom: "0102030405060708", Mac: "fb62ef385a7ba6b0c266eac9ccf53605fb951890b9110b36577fbecf53113b1d077e7ab32abf07ba6e131612a2ea25105e2e4ad9e751e5a593ebd517c8243301a128737e077a2feba56795b1ba84017af250aa26c4e91f7262a86b95c618157ea3b07d58a680a02c378205de5874eb5560830b499651b99601f11d7c4a73641708e4a0a9290d7d18b1f4bf9dd5544c06f77b912dfc480ba0f01429700bc8da286b17c56b6441630db59d841d4826d76263c004489e4ea185d8a5c78a378db816fb3da1641c7bf0b7d279c15c866e5fad9716c6fea7ec7a033bfe28df758a764c0be2e19c2883e7b9b8d46c2dabb2c86ac4b3d7b165757ab0b21c6bd11aef4cfcab003d8eea093dc6");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f10", Msg: "", Custom: "", Mac: "ee");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f10", Msg: "68656c6c6f", Custom: "", Mac: "07");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f10", Msg: "676f6f64627965", Custom: "", Mac: "69");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f10", Msg: "", Custom: "0102030405060708", Mac: "4f");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f10", Msg: "68656c6c6f", Custom: "0102030405060708", Mac: "9b");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f10", Msg: "676f6f64627965", Custom: "0102030405060708", Mac: "f0");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f10", Msg: "", Custom: "", Mac: "eed85f46eb311f90");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f10", Msg: "68656c6c6f", Custom: "", Mac: "07e748b261a2a6cb");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f10", Msg: "676f6f64627965", Custom: "", Mac: "69a7cd47c75385f1");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f10", Msg: "", Custom: "0102030405060708", Mac: "4f6cbb936643a4be");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f10", Msg: "68656c6c6f", Custom: "0102030405060708", Mac: "9ba7e3f7e9f8ab1f");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f10", Msg: "676f6f64627965", Custom: "0102030405060708", Mac: "f00ecc5c55d937ab");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f10", Msg: "", Custom: "", Mac: "eed85f46eb311f90e5bd22f2ba051b7912f01cc8a9188088");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f10", Msg: "68656c6c6f", Custom: "", Mac: "07e748b261a2a6cb0b1beb59434ded0dfb3e261e7feb128a");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f10", Msg: "676f6f64627965", Custom: "", Mac: "69a7cd47c75385f1a742ca3bfb5e22022f15e5f93e1f6616");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f10", Msg: "", Custom: "0102030405060708", Mac: "4f6cbb936643a4bebca4322adaae6bb9178180c052b6dd59");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f10", Msg: "68656c6c6f", Custom: "0102030405060708", Mac: "9ba7e3f7e9f8ab1f4d79ed4b375d26073335c51403cc3f2d");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f10", Msg: "676f6f64627965", Custom: "0102030405060708", Mac: "f00ecc5c55d937ab4b0e8acba15481979b9aee95c315d9cf");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f10", Msg: "", Custom: "", Mac: "eed85f46eb311f90e5bd22f2ba051b7912f01cc8a91880885b7e671bf9c37da5972e131eef9d0c23");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f10", Msg: "68656c6c6f", Custom: "", Mac: "07e748b261a2a6cb0b1beb59434ded0dfb3e261e7feb128a6d4ed7b2a0f328052f0608af596cd745");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f10", Msg: "676f6f64627965", Custom: "", Mac: "69a7cd47c75385f1a742ca3bfb5e22022f15e5f93e1f66160de3dcf648cc37c7e49bca6a099d1d38");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f10", Msg: "", Custom: "0102030405060708", Mac: "4f6cbb936643a4bebca4322adaae6bb9178180c052b6dd595ac08a7c98919990fee52a3edc87cd56");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f10", Msg: "68656c6c6f", Custom: "0102030405060708", Mac: "9ba7e3f7e9f8ab1f4d79ed4b375d26073335c51403cc3f2d9de7a915fba7134ff102d42be6a1586c");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f10", Msg: "676f6f64627965", Custom: "0102030405060708", Mac: "f00ecc5c55d937ab4b0e8acba15481979b9aee95c315d9cf580eb10ed05cc3cc1fc6202436ab6d06");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f10", Msg: "", Custom: "", Mac: "eed85f46eb311f90e5bd22f2ba051b7912f01cc8a91880885b7e671bf9c37da5972e131eef9d0c235df735b7dfbe995ef87020cc83078557");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f10", Msg: "68656c6c6f", Custom: "", Mac: "07e748b261a2a6cb0b1beb59434ded0dfb3e261e7feb128a6d4ed7b2a0f328052f0608af596cd745ee3483f375c713323155581c24a58dcd");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f10", Msg: "676f6f64627965", Custom: "", Mac: "69a7cd47c75385f1a742ca3bfb5e22022f15e5f93e1f66160de3dcf648cc37c7e49bca6a099d1d3884112e551e19cc9995aaf8bddc9be5c0");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f10", Msg: "", Custom: "0102030405060708", Mac: "4f6cbb936643a4bebca4322adaae6bb9178180c052b6dd595ac08a7c98919990fee52a3edc87cd564ce01ef414333d9c83cca84cce852d62");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f10", Msg: "68656c6c6f", Custom: "0102030405060708", Mac: "9ba7e3f7e9f8ab1f4d79ed4b375d26073335c51403cc3f2d9de7a915fba7134ff102d42be6a1586c70c447723050f96f8f8e8b140c82c14a");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f10", Msg: "676f6f64627965", Custom: "0102030405060708", Mac: "f00ecc5c55d937ab4b0e8acba15481979b9aee95c315d9cf580eb10ed05cc3cc1fc6202436ab6d06ade5aefb93b9d7ac604a6f9dfcf1eb45");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f10", Msg: "", Custom: "", Mac: "eed85f46eb311f90e5bd22f2ba051b7912f01cc8a91880885b7e671bf9c37da5972e131eef9d0c235df735b7dfbe995ef87020cc830785573b7faeea3ea2d7fcd1a153c45b23fef8");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f10", Msg: "68656c6c6f", Custom: "", Mac: "07e748b261a2a6cb0b1beb59434ded0dfb3e261e7feb128a6d4ed7b2a0f328052f0608af596cd745ee3483f375c713323155581c24a58dcd21c63cda31f7544067b116a3fd2de8c7");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f10", Msg: "676f6f64627965", Custom: "", Mac: "69a7cd47c75385f1a742ca3bfb5e22022f15e5f93e1f66160de3dcf648cc37c7e49bca6a099d1d3884112e551e19cc9995aaf8bddc9be5c062c6fc6355012c612e460817dd68d829");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f10", Msg: "", Custom: "0102030405060708", Mac: "4f6cbb936643a4bebca4322adaae6bb9178180c052b6dd595ac08a7c98919990fee52a3edc87cd564ce01ef414333d9c83cca84cce852d62b2dab74d073ad16cc2e1b0765e0c1825");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f10", Msg: "68656c6c6f", Custom: "0102030405060708", Mac: "9ba7e3f7e9f8ab1f4d79ed4b375d26073335c51403cc3f2d9de7a915fba7134ff102d42be6a1586c70c447723050f96f8f8e8b140c82c14a2d542c1359d79247824a9328e93f4153");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f10", Msg: "676f6f64627965", Custom: "0102030405060708", Mac: "f00ecc5c55d937ab4b0e8acba15481979b9aee95c315d9cf580eb10ed05cc3cc1fc6202436ab6d06ade5aefb93b9d7ac604a6f9dfcf1eb4596d81ef26f0f7e2adf9ffd720f505fc7");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f10", Msg: "", Custom: "", Mac: "eed85f46eb311f90e5bd22f2ba051b7912f01cc8a91880885b7e671bf9c37da5972e131eef9d0c235df735b7dfbe995ef87020cc830785573b7faeea3ea2d7fcd1a153c45b23fef80bfca0d4274167d59fd35970b7346201");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f10", Msg: "68656c6c6f", Custom: "", Mac: "07e748b261a2a6cb0b1beb59434ded0dfb3e261e7feb128a6d4ed7b2a0f328052f0608af596cd745ee3483f375c713323155581c24a58dcd21c63cda31f7544067b116a3fd2de8c719a1f8690dcbe275472af61a60a10234");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f10", Msg: "676f6f64627965", Custom: "", Mac: "69a7cd47c75385f1a742ca3bfb5e22022f15e5f93e1f66160de3dcf648cc37c7e49bca6a099d1d3884112e551e19cc9995aaf8bddc9be5c062c6fc6355012c612e460817dd68d82914ae94ee4ecc877c0327fd396ac180ea");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f10", Msg: "", Custom: "0102030405060708", Mac: "4f6cbb936643a4bebca4322adaae6bb9178180c052b6dd595ac08a7c98919990fee52a3edc87cd564ce01ef414333d9c83cca84cce852d62b2dab74d073ad16cc2e1b0765e0c1825edd11eba4ce9ddb230b2803ffd54b305");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f10", Msg: "68656c6c6f", Custom: "0102030405060708", Mac: "9ba7e3f7e9f8ab1f4d79ed4b375d26073335c51403cc3f2d9de7a915fba7134ff102d42be6a1586c70c447723050f96f8f8e8b140c82c14a2d542c1359d79247824a9328e93f41539019442b130a5fc387d7f38f35f99326");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f10", Msg: "676f6f64627965", Custom: "0102030405060708", Mac: "f00ecc5c55d937ab4b0e8acba15481979b9aee95c315d9cf580eb10ed05cc3cc1fc6202436ab6d06ade5aefb93b9d7ac604a6f9dfcf1eb4596d81ef26f0f7e2adf9ffd720f505fc7d5c358ce7649c4daa98af93e96bcb21c");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f10", Msg: "", Custom: "", Mac: "eed85f46eb311f90e5bd22f2ba051b7912f01cc8a91880885b7e671bf9c37da5972e131eef9d0c235df735b7dfbe995ef87020cc830785573b7faeea3ea2d7fcd1a153c45b23fef80bfca0d4274167d59fd35970b7346201ba69535011b6ba0493fb612de7671cc1");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f10", Msg: "68656c6c6f", Custom: "", Mac: "07e748b261a2a6cb0b1beb59434ded0dfb3e261e7feb128a6d4ed7b2a0f328052f0608af596cd745ee3483f375c713323155581c24a58dcd21c63cda31f7544067b116a3fd2de8c719a1f8690dcbe275472af61a60a1023453d824a18e0018e01660c75ab4434175");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f10", Msg: "676f6f64627965", Custom: "", Mac: "69a7cd47c75385f1a742ca3bfb5e22022f15e5f93e1f66160de3dcf648cc37c7e49bca6a099d1d3884112e551e19cc9995aaf8bddc9be5c062c6fc6355012c612e460817dd68d82914ae94ee4ecc877c0327fd396ac180ead919538bbad5e99ead85c9ac235ea9a2");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f10", Msg: "", Custom: "0102030405060708", Mac: "4f6cbb936643a4bebca4322adaae6bb9178180c052b6dd595ac08a7c98919990fee52a3edc87cd564ce01ef414333d9c83cca84cce852d62b2dab74d073ad16cc2e1b0765e0c1825edd11eba4ce9ddb230b2803ffd54b30564b2319b4450a45bfcc7c512fd9d5663");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f10", Msg: "68656c6c6f", Custom: "0102030405060708", Mac: "9ba7e3f7e9f8ab1f4d79ed4b375d26073335c51403cc3f2d9de7a915fba7134ff102d42be6a1586c70c447723050f96f8f8e8b140c82c14a2d542c1359d79247824a9328e93f41539019442b130a5fc387d7f38f35f993263d14a15e8043b77010e9894cafd7fa39");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f10", Msg: "676f6f64627965", Custom: "0102030405060708", Mac: "f00ecc5c55d937ab4b0e8acba15481979b9aee95c315d9cf580eb10ed05cc3cc1fc6202436ab6d06ade5aefb93b9d7ac604a6f9dfcf1eb4596d81ef26f0f7e2adf9ffd720f505fc7d5c358ce7649c4daa98af93e96bcb21c06e7a48f3a8a7da0daac5830c1fe0165");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f10", Msg: "", Custom: "", Mac: "eed85f46eb311f90e5bd22f2ba051b7912f01cc8a91880885b7e671bf9c37da5972e131eef9d0c235df735b7dfbe995ef87020cc830785573b7faeea3ea2d7fcd1a153c45b23fef80bfca0d4274167d59fd35970b7346201ba69535011b6ba0493fb612de7671cc1ea0b098f7fe2594bac9add3faa52af4d");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f10", Msg: "68656c6c6f", Custom: "", Mac: "07e748b261a2a6cb0b1beb59434ded0dfb3e261e7feb128a6d4ed7b2a0f328052f0608af596cd745ee3483f375c713323155581c24a58dcd21c63cda31f7544067b116a3fd2de8c719a1f8690dcbe275472af61a60a1023453d824a18e0018e01660c75ab4434175e35ffe6bee8a7d5e55cdad4729132ba7");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f10", Msg: "676f6f64627965", Custom: "", Mac: "69a7cd47c75385f1a742ca3bfb5e22022f15e5f93e1f66160de3dcf648cc37c7e49bca6a099d1d3884112e551e19cc9995aaf8bddc9be5c062c6fc6355012c612e460817dd68d82914ae94ee4ecc877c0327fd396ac180ead919538bbad5e99ead85c9ac235ea9a255fc0ad55969eac003f0b3f13d6a7705");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f10", Msg: "", Custom: "0102030405060708", Mac: "4f6cbb936643a4bebca4322adaae6bb9178180c052b6dd595ac08a7c98919990fee52a3edc87cd564ce01ef414333d9c83cca84cce852d62b2dab74d073ad16cc2e1b0765e0c1825edd11eba4ce9ddb230b2803ffd54b30564b2319b4450a45bfcc7c512fd9d566304e18dfee179b164cd2a7851f295189a");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f10", Msg: "68656c6c6f", Custom: "0102030405060708", Mac: "9ba7e3f7e9f8ab1f4d79ed4b375d26073335c51403cc3f2d9de7a915fba7134ff102d42be6a1586c70c447723050f96f8f8e8b140c82c14a2d542c1359d79247824a9328e93f41539019442b130a5fc387d7f38f35f993263d14a15e8043b77010e9894cafd7fa3901f179fdc6725cf53b6f8297effba386");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f10", Msg: "676f6f64627965", Custom: "0102030405060708", Mac: "f00ecc5c55d937ab4b0e8acba15481979b9aee95c315d9cf580eb10ed05cc3cc1fc6202436ab6d06ade5aefb93b9d7ac604a6f9dfcf1eb4596d81ef26f0f7e2adf9ffd720f505fc7d5c358ce7649c4daa98af93e96bcb21c06e7a48f3a8a7da0daac5830c1fe01652c5e95416525ac20623a0d83b11869ec");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f10", Msg: "", Custom: "", Mac: "eed85f46eb311f90e5bd22f2ba051b7912f01cc8a91880885b7e671bf9c37da5972e131eef9d0c235df735b7dfbe995ef87020cc830785573b7faeea3ea2d7fcd1a153c45b23fef80bfca0d4274167d59fd35970b7346201ba69535011b6ba0493fb612de7671cc1ea0b098f7fe2594bac9add3faa52af4d415ac1bd54fcb43f844805dbe4053cb1");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f10", Msg: "68656c6c6f", Custom: "", Mac: "07e748b261a2a6cb0b1beb59434ded0dfb3e261e7feb128a6d4ed7b2a0f328052f0608af596cd745ee3483f375c713323155581c24a58dcd21c63cda31f7544067b116a3fd2de8c719a1f8690dcbe275472af61a60a1023453d824a18e0018e01660c75ab4434175e35ffe6bee8a7d5e55cdad4729132ba79ddaa4293fbb81c7a41e7d289c07b4b6");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f10", Msg: "676f6f64627965", Custom: "", Mac: "69a7cd47c75385f1a742ca3bfb5e22022f15e5f93e1f66160de3dcf648cc37c7e49bca6a099d1d3884112e551e19cc9995aaf8bddc9be5c062c6fc6355012c612e460817dd68d82914ae94ee4ecc877c0327fd396ac180ead919538bbad5e99ead85c9ac235ea9a255fc0ad55969eac003f0b3f13d6a77052fab975ef6e5705b10a9b99dc2c00663");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f10", Msg: "", Custom: "0102030405060708", Mac: "4f6cbb936643a4bebca4322adaae6bb9178180c052b6dd595ac08a7c98919990fee52a3edc87cd564ce01ef414333d9c83cca84cce852d62b2dab74d073ad16cc2e1b0765e0c1825edd11eba4ce9ddb230b2803ffd54b30564b2319b4450a45bfcc7c512fd9d566304e18dfee179b164cd2a7851f295189a4fabbe619872e1578571731f977d8aac");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f10", Msg: "68656c6c6f", Custom: "0102030405060708", Mac: "9ba7e3f7e9f8ab1f4d79ed4b375d26073335c51403cc3f2d9de7a915fba7134ff102d42be6a1586c70c447723050f96f8f8e8b140c82c14a2d542c1359d79247824a9328e93f41539019442b130a5fc387d7f38f35f993263d14a15e8043b77010e9894cafd7fa3901f179fdc6725cf53b6f8297effba386414b94137b785ed96280dbf90546e609");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f10", Msg: "676f6f64627965", Custom: "0102030405060708", Mac: "f00ecc5c55d937ab4b0e8acba15481979b9aee95c315d9cf580eb10ed05cc3cc1fc6202436ab6d06ade5aefb93b9d7ac604a6f9dfcf1eb4596d81ef26f0f7e2adf9ffd720f505fc7d5c358ce7649c4daa98af93e96bcb21c06e7a48f3a8a7da0daac5830c1fe01652c5e95416525ac20623a0d83b11869ece5fab6c0fb29c12db31a09d80d3a01c6");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f10", Msg: "", Custom: "", Mac: "eed85f46eb311f90e5bd22f2ba051b7912f01cc8a91880885b7e671bf9c37da5972e131eef9d0c235df735b7dfbe995ef87020cc830785573b7faeea3ea2d7fcd1a153c45b23fef80bfca0d4274167d59fd35970b7346201ba69535011b6ba0493fb612de7671cc1ea0b098f7fe2594bac9add3faa52af4d415ac1bd54fcb43f844805dbe4053cb115f8fdd240a9294469bb2a4ed82220e8");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f10", Msg: "68656c6c6f", Custom: "", Mac: "07e748b261a2a6cb0b1beb59434ded0dfb3e261e7feb128a6d4ed7b2a0f328052f0608af596cd745ee3483f375c713323155581c24a58dcd21c63cda31f7544067b116a3fd2de8c719a1f8690dcbe275472af61a60a1023453d824a18e0018e01660c75ab4434175e35ffe6bee8a7d5e55cdad4729132ba79ddaa4293fbb81c7a41e7d289c07b4b69e755933812b4a7aef961b664ea76f2f");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f10", Msg: "676f6f64627965", Custom: "", Mac: "69a7cd47c75385f1a742ca3bfb5e22022f15e5f93e1f66160de3dcf648cc37c7e49bca6a099d1d3884112e551e19cc9995aaf8bddc9be5c062c6fc6355012c612e460817dd68d82914ae94ee4ecc877c0327fd396ac180ead919538bbad5e99ead85c9ac235ea9a255fc0ad55969eac003f0b3f13d6a77052fab975ef6e5705b10a9b99dc2c006637705943c812104603feed4a8535500c0");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f10", Msg: "", Custom: "0102030405060708", Mac: "4f6cbb936643a4bebca4322adaae6bb9178180c052b6dd595ac08a7c98919990fee52a3edc87cd564ce01ef414333d9c83cca84cce852d62b2dab74d073ad16cc2e1b0765e0c1825edd11eba4ce9ddb230b2803ffd54b30564b2319b4450a45bfcc7c512fd9d566304e18dfee179b164cd2a7851f295189a4fabbe619872e1578571731f977d8aac2e7df57def5f8939853c3098c5742d7a");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f10", Msg: "68656c6c6f", Custom: "0102030405060708", Mac: "9ba7e3f7e9f8ab1f4d79ed4b375d26073335c51403cc3f2d9de7a915fba7134ff102d42be6a1586c70c447723050f96f8f8e8b140c82c14a2d542c1359d79247824a9328e93f41539019442b130a5fc387d7f38f35f993263d14a15e8043b77010e9894cafd7fa3901f179fdc6725cf53b6f8297effba386414b94137b785ed96280dbf90546e609a186a6eb10818adf8325f75ca002161d");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f10", Msg: "676f6f64627965", Custom: "0102030405060708", Mac: "f00ecc5c55d937ab4b0e8acba15481979b9aee95c315d9cf580eb10ed05cc3cc1fc6202436ab6d06ade5aefb93b9d7ac604a6f9dfcf1eb4596d81ef26f0f7e2adf9ffd720f505fc7d5c358ce7649c4daa98af93e96bcb21c06e7a48f3a8a7da0daac5830c1fe01652c5e95416525ac20623a0d83b11869ece5fab6c0fb29c12db31a09d80d3a01c61ed62dafbc423bba12b3f88610968942");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f10", Msg: "", Custom: "", Mac: "eed85f46eb311f90e5bd22f2ba051b7912f01cc8a91880885b7e671bf9c37da5972e131eef9d0c235df735b7dfbe995ef87020cc830785573b7faeea3ea2d7fcd1a153c45b23fef80bfca0d4274167d59fd35970b7346201ba69535011b6ba0493fb612de7671cc1ea0b098f7fe2594bac9add3faa52af4d415ac1bd54fcb43f844805dbe4053cb115f8fdd240a9294469bb2a4ed82220e87a8f8964780bde37a63d338d9a36d10a");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f10", Msg: "68656c6c6f", Custom: "", Mac: "07e748b261a2a6cb0b1beb59434ded0dfb3e261e7feb128a6d4ed7b2a0f328052f0608af596cd745ee3483f375c713323155581c24a58dcd21c63cda31f7544067b116a3fd2de8c719a1f8690dcbe275472af61a60a1023453d824a18e0018e01660c75ab4434175e35ffe6bee8a7d5e55cdad4729132ba79ddaa4293fbb81c7a41e7d289c07b4b69e755933812b4a7aef961b664ea76f2ff75eaa3357f3cf41274bd8b97a489db1");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f10", Msg: "676f6f64627965", Custom: "", Mac: "69a7cd47c75385f1a742ca3bfb5e22022f15e5f93e1f66160de3dcf648cc37c7e49bca6a099d1d3884112e551e19cc9995aaf8bddc9be5c062c6fc6355012c612e460817dd68d82914ae94ee4ecc877c0327fd396ac180ead919538bbad5e99ead85c9ac235ea9a255fc0ad55969eac003f0b3f13d6a77052fab975ef6e5705b10a9b99dc2c006637705943c812104603feed4a8535500c05d1760ddc2b3fc08517aa524de4af8e8");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f10", Msg: "", Custom: "0102030405060708", Mac: "4f6cbb936643a4bebca4322adaae6bb9178180c052b6dd595ac08a7c98919990fee52a3edc87cd564ce01ef414333d9c83cca84cce852d62b2dab74d073ad16cc2e1b0765e0c1825edd11eba4ce9ddb230b2803ffd54b30564b2319b4450a45bfcc7c512fd9d566304e18dfee179b164cd2a7851f295189a4fabbe619872e1578571731f977d8aac2e7df57def5f8939853c3098c5742d7a12dc9eaa3750865d43025a365346ad47");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f10", Msg: "68656c6c6f", Custom: "0102030405060708", Mac: "9ba7e3f7e9f8ab1f4d79ed4b375d26073335c51403cc3f2d9de7a915fba7134ff102d42be6a1586c70c447723050f96f8f8e8b140c82c14a2d542c1359d79247824a9328e93f41539019442b130a5fc387d7f38f35f993263d14a15e8043b77010e9894cafd7fa3901f179fdc6725cf53b6f8297effba386414b94137b785ed96280dbf90546e609a186a6eb10818adf8325f75ca002161d218cf54f38a6331a118397775e341a28");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f10", Msg: "676f6f64627965", Custom: "0102030405060708", Mac: "f00ecc5c55d937ab4b0e8acba15481979b9aee95c315d9cf580eb10ed05cc3cc1fc6202436ab6d06ade5aefb93b9d7ac604a6f9dfcf1eb4596d81ef26f0f7e2adf9ffd720f505fc7d5c358ce7649c4daa98af93e96bcb21c06e7a48f3a8a7da0daac5830c1fe01652c5e95416525ac20623a0d83b11869ece5fab6c0fb29c12db31a09d80d3a01c61ed62dafbc423bba12b3f886109689423d175b9ca6169d54c6cbb6d26e0ef693");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f10", Msg: "", Custom: "", Mac: "eed85f46eb311f90e5bd22f2ba051b7912f01cc8a91880885b7e671bf9c37da5972e131eef9d0c235df735b7dfbe995ef87020cc830785573b7faeea3ea2d7fcd1a153c45b23fef80bfca0d4274167d59fd35970b7346201ba69535011b6ba0493fb612de7671cc1ea0b098f7fe2594bac9add3faa52af4d415ac1bd54fcb43f844805dbe4053cb115f8fdd240a9294469bb2a4ed82220e87a8f8964780bde37a63d338d9a36d10aa550b0fbf24f68075e854431c02c132a");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f10", Msg: "68656c6c6f", Custom: "", Mac: "07e748b261a2a6cb0b1beb59434ded0dfb3e261e7feb128a6d4ed7b2a0f328052f0608af596cd745ee3483f375c713323155581c24a58dcd21c63cda31f7544067b116a3fd2de8c719a1f8690dcbe275472af61a60a1023453d824a18e0018e01660c75ab4434175e35ffe6bee8a7d5e55cdad4729132ba79ddaa4293fbb81c7a41e7d289c07b4b69e755933812b4a7aef961b664ea76f2ff75eaa3357f3cf41274bd8b97a489db167cdf3b00f7482dda7db32e1587db541");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f10", Msg: "676f6f64627965", Custom: "", Mac: "69a7cd47c75385f1a742ca3bfb5e22022f15e5f93e1f66160de3dcf648cc37c7e49bca6a099d1d3884112e551e19cc9995aaf8bddc9be5c062c6fc6355012c612e460817dd68d82914ae94ee4ecc877c0327fd396ac180ead919538bbad5e99ead85c9ac235ea9a255fc0ad55969eac003f0b3f13d6a77052fab975ef6e5705b10a9b99dc2c006637705943c812104603feed4a8535500c05d1760ddc2b3fc08517aa524de4af8e8bba6a2c0662d94dca9da2d7fdbf88e70");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f10", Msg: "", Custom: "0102030405060708", Mac: "4f6cbb936643a4bebca4322adaae6bb9178180c052b6dd595ac08a7c98919990fee52a3edc87cd564ce01ef414333d9c83cca84cce852d62b2dab74d073ad16cc2e1b0765e0c1825edd11eba4ce9ddb230b2803ffd54b30564b2319b4450a45bfcc7c512fd9d566304e18dfee179b164cd2a7851f295189a4fabbe619872e1578571731f977d8aac2e7df57def5f8939853c3098c5742d7a12dc9eaa3750865d43025a365346ad477f899d9da6fd3751e76f0d241f885082");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f10", Msg: "68656c6c6f", Custom: "0102030405060708", Mac: "9ba7e3f7e9f8ab1f4d79ed4b375d26073335c51403cc3f2d9de7a915fba7134ff102d42be6a1586c70c447723050f96f8f8e8b140c82c14a2d542c1359d79247824a9328e93f41539019442b130a5fc387d7f38f35f993263d14a15e8043b77010e9894cafd7fa3901f179fdc6725cf53b6f8297effba386414b94137b785ed96280dbf90546e609a186a6eb10818adf8325f75ca002161d218cf54f38a6331a118397775e341a28af691667053e0f2f8583acba818c302d");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f10", Msg: "676f6f64627965", Custom: "0102030405060708", Mac: "f00ecc5c55d937ab4b0e8acba15481979b9aee95c315d9cf580eb10ed05cc3cc1fc6202436ab6d06ade5aefb93b9d7ac604a6f9dfcf1eb4596d81ef26f0f7e2adf9ffd720f505fc7d5c358ce7649c4daa98af93e96bcb21c06e7a48f3a8a7da0daac5830c1fe01652c5e95416525ac20623a0d83b11869ece5fab6c0fb29c12db31a09d80d3a01c61ed62dafbc423bba12b3f886109689423d175b9ca6169d54c6cbb6d26e0ef693cf226785d06cb6503a753513f2c2d972");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f10", Msg: "", Custom: "", Mac: "eed85f46eb311f90e5bd22f2ba051b7912f01cc8a91880885b7e671bf9c37da5972e131eef9d0c235df735b7dfbe995ef87020cc830785573b7faeea3ea2d7fcd1a153c45b23fef80bfca0d4274167d59fd35970b7346201ba69535011b6ba0493fb612de7671cc1ea0b098f7fe2594bac9add3faa52af4d415ac1bd54fcb43f844805dbe4053cb115f8fdd240a9294469bb2a4ed82220e87a8f8964780bde37a63d338d9a36d10aa550b0fbf24f68075e854431c02c132a3ed245f07f3b78be39abbb144a62fcb7");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f10", Msg: "68656c6c6f", Custom: "", Mac: "07e748b261a2a6cb0b1beb59434ded0dfb3e261e7feb128a6d4ed7b2a0f328052f0608af596cd745ee3483f375c713323155581c24a58dcd21c63cda31f7544067b116a3fd2de8c719a1f8690dcbe275472af61a60a1023453d824a18e0018e01660c75ab4434175e35ffe6bee8a7d5e55cdad4729132ba79ddaa4293fbb81c7a41e7d289c07b4b69e755933812b4a7aef961b664ea76f2ff75eaa3357f3cf41274bd8b97a489db167cdf3b00f7482dda7db32e1587db5417046484b145ec75fcb26f2df310d4014");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f10", Msg: "676f6f64627965", Custom: "", Mac: "69a7cd47c75385f1a742ca3bfb5e22022f15e5f93e1f66160de3dcf648cc37c7e49bca6a099d1d3884112e551e19cc9995aaf8bddc9be5c062c6fc6355012c612e460817dd68d82914ae94ee4ecc877c0327fd396ac180ead919538bbad5e99ead85c9ac235ea9a255fc0ad55969eac003f0b3f13d6a77052fab975ef6e5705b10a9b99dc2c006637705943c812104603feed4a8535500c05d1760ddc2b3fc08517aa524de4af8e8bba6a2c0662d94dca9da2d7fdbf88e70ad5f5005b9bda7226a62a1e5f6e06b57");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f10", Msg: "", Custom: "0102030405060708", Mac: "4f6cbb936643a4bebca4322adaae6bb9178180c052b6dd595ac08a7c98919990fee52a3edc87cd564ce01ef414333d9c83cca84cce852d62b2dab74d073ad16cc2e1b0765e0c1825edd11eba4ce9ddb230b2803ffd54b30564b2319b4450a45bfcc7c512fd9d566304e18dfee179b164cd2a7851f295189a4fabbe619872e1578571731f977d8aac2e7df57def5f8939853c3098c5742d7a12dc9eaa3750865d43025a365346ad477f899d9da6fd3751e76f0d241f885082ac5d35b109270c389e9e0f744e5d4c73");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f10", Msg: "68656c6c6f", Custom: "0102030405060708", Mac: "9ba7e3f7e9f8ab1f4d79ed4b375d26073335c51403cc3f2d9de7a915fba7134ff102d42be6a1586c70c447723050f96f8f8e8b140c82c14a2d542c1359d79247824a9328e93f41539019442b130a5fc387d7f38f35f993263d14a15e8043b77010e9894cafd7fa3901f179fdc6725cf53b6f8297effba386414b94137b785ed96280dbf90546e609a186a6eb10818adf8325f75ca002161d218cf54f38a6331a118397775e341a28af691667053e0f2f8583acba818c302d1ec2e34e67f377d79da87b0f3ba675d4");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f10", Msg: "676f6f64627965", Custom: "0102030405060708", Mac: "f00ecc5c55d937ab4b0e8acba15481979b9aee95c315d9cf580eb10ed05cc3cc1fc6202436ab6d06ade5aefb93b9d7ac604a6f9dfcf1eb4596d81ef26f0f7e2adf9ffd720f505fc7d5c358ce7649c4daa98af93e96bcb21c06e7a48f3a8a7da0daac5830c1fe01652c5e95416525ac20623a0d83b11869ece5fab6c0fb29c12db31a09d80d3a01c61ed62dafbc423bba12b3f886109689423d175b9ca6169d54c6cbb6d26e0ef693cf226785d06cb6503a753513f2c2d9723e0cf1049f229e018719c8c3804692ed");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f10", Msg: "", Custom: "", Mac: "eed85f46eb311f90e5bd22f2ba051b7912f01cc8a91880885b7e671bf9c37da5972e131eef9d0c235df735b7dfbe995ef87020cc830785573b7faeea3ea2d7fcd1a153c45b23fef80bfca0d4274167d59fd35970b7346201ba69535011b6ba0493fb612de7671cc1ea0b098f7fe2594bac9add3faa52af4d415ac1bd54fcb43f844805dbe4053cb115f8fdd240a9294469bb2a4ed82220e87a8f8964780bde37a63d338d9a36d10aa550b0fbf24f68075e854431c02c132a3ed245f07f3b78be39abbb144a62fcb7c4cdcaad3c6f4fa48d8451a2d1795a91");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f10", Msg: "68656c6c6f", Custom: "", Mac: "07e748b261a2a6cb0b1beb59434ded0dfb3e261e7feb128a6d4ed7b2a0f328052f0608af596cd745ee3483f375c713323155581c24a58dcd21c63cda31f7544067b116a3fd2de8c719a1f8690dcbe275472af61a60a1023453d824a18e0018e01660c75ab4434175e35ffe6bee8a7d5e55cdad4729132ba79ddaa4293fbb81c7a41e7d289c07b4b69e755933812b4a7aef961b664ea76f2ff75eaa3357f3cf41274bd8b97a489db167cdf3b00f7482dda7db32e1587db5417046484b145ec75fcb26f2df310d401403f9212c55d7ad85ebe5affc123ed2d3");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f10", Msg: "676f6f64627965", Custom: "", Mac: "69a7cd47c75385f1a742ca3bfb5e22022f15e5f93e1f66160de3dcf648cc37c7e49bca6a099d1d3884112e551e19cc9995aaf8bddc9be5c062c6fc6355012c612e460817dd68d82914ae94ee4ecc877c0327fd396ac180ead919538bbad5e99ead85c9ac235ea9a255fc0ad55969eac003f0b3f13d6a77052fab975ef6e5705b10a9b99dc2c006637705943c812104603feed4a8535500c05d1760ddc2b3fc08517aa524de4af8e8bba6a2c0662d94dca9da2d7fdbf88e70ad5f5005b9bda7226a62a1e5f6e06b57bd47490b0cc6c526d4295c487344498d");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f10", Msg: "", Custom: "0102030405060708", Mac: "4f6cbb936643a4bebca4322adaae6bb9178180c052b6dd595ac08a7c98919990fee52a3edc87cd564ce01ef414333d9c83cca84cce852d62b2dab74d073ad16cc2e1b0765e0c1825edd11eba4ce9ddb230b2803ffd54b30564b2319b4450a45bfcc7c512fd9d566304e18dfee179b164cd2a7851f295189a4fabbe619872e1578571731f977d8aac2e7df57def5f8939853c3098c5742d7a12dc9eaa3750865d43025a365346ad477f899d9da6fd3751e76f0d241f885082ac5d35b109270c389e9e0f744e5d4c73f7ab7f760c1b04474ea3ec90cf4633e4");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f10", Msg: "68656c6c6f", Custom: "0102030405060708", Mac: "9ba7e3f7e9f8ab1f4d79ed4b375d26073335c51403cc3f2d9de7a915fba7134ff102d42be6a1586c70c447723050f96f8f8e8b140c82c14a2d542c1359d79247824a9328e93f41539019442b130a5fc387d7f38f35f993263d14a15e8043b77010e9894cafd7fa3901f179fdc6725cf53b6f8297effba386414b94137b785ed96280dbf90546e609a186a6eb10818adf8325f75ca002161d218cf54f38a6331a118397775e341a28af691667053e0f2f8583acba818c302d1ec2e34e67f377d79da87b0f3ba675d48ae69a07083a6d8a77d41a2bc055ee99");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f10", Msg: "676f6f64627965", Custom: "0102030405060708", Mac: "f00ecc5c55d937ab4b0e8acba15481979b9aee95c315d9cf580eb10ed05cc3cc1fc6202436ab6d06ade5aefb93b9d7ac604a6f9dfcf1eb4596d81ef26f0f7e2adf9ffd720f505fc7d5c358ce7649c4daa98af93e96bcb21c06e7a48f3a8a7da0daac5830c1fe01652c5e95416525ac20623a0d83b11869ece5fab6c0fb29c12db31a09d80d3a01c61ed62dafbc423bba12b3f886109689423d175b9ca6169d54c6cbb6d26e0ef693cf226785d06cb6503a753513f2c2d9723e0cf1049f229e018719c8c3804692eddf505a5de901cbea56ead36390dae6a5");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f10", Msg: "", Custom: "", Mac: "eed85f46eb311f90e5bd22f2ba051b7912f01cc8a91880885b7e671bf9c37da5972e131eef9d0c235df735b7dfbe995ef87020cc830785573b7faeea3ea2d7fcd1a153c45b23fef80bfca0d4274167d59fd35970b7346201ba69535011b6ba0493fb612de7671cc1ea0b098f7fe2594bac9add3faa52af4d415ac1bd54fcb43f844805dbe4053cb115f8fdd240a9294469bb2a4ed82220e87a8f8964780bde37a63d338d9a36d10aa550b0fbf24f68075e854431c02c132a3ed245f07f3b78be39abbb144a62fcb7c4cdcaad3c6f4fa48d8451a2d1795a9196967864b12d44e426544a9c0ea4f146");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f10", Msg: "68656c6c6f", Custom: "", Mac: "07e748b261a2a6cb0b1beb59434ded0dfb3e261e7feb128a6d4ed7b2a0f328052f0608af596cd745ee3483f375c713323155581c24a58dcd21c63cda31f7544067b116a3fd2de8c719a1f8690dcbe275472af61a60a1023453d824a18e0018e01660c75ab4434175e35ffe6bee8a7d5e55cdad4729132ba79ddaa4293fbb81c7a41e7d289c07b4b69e755933812b4a7aef961b664ea76f2ff75eaa3357f3cf41274bd8b97a489db167cdf3b00f7482dda7db32e1587db5417046484b145ec75fcb26f2df310d401403f9212c55d7ad85ebe5affc123ed2d342fa61026f8254b773f890458dc48188");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f10", Msg: "676f6f64627965", Custom: "", Mac: "69a7cd47c75385f1a742ca3bfb5e22022f15e5f93e1f66160de3dcf648cc37c7e49bca6a099d1d3884112e551e19cc9995aaf8bddc9be5c062c6fc6355012c612e460817dd68d82914ae94ee4ecc877c0327fd396ac180ead919538bbad5e99ead85c9ac235ea9a255fc0ad55969eac003f0b3f13d6a77052fab975ef6e5705b10a9b99dc2c006637705943c812104603feed4a8535500c05d1760ddc2b3fc08517aa524de4af8e8bba6a2c0662d94dca9da2d7fdbf88e70ad5f5005b9bda7226a62a1e5f6e06b57bd47490b0cc6c526d4295c487344498d7bd49c823c5f3ce6f795ef749ef909bf");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f10", Msg: "", Custom: "0102030405060708", Mac: "4f6cbb936643a4bebca4322adaae6bb9178180c052b6dd595ac08a7c98919990fee52a3edc87cd564ce01ef414333d9c83cca84cce852d62b2dab74d073ad16cc2e1b0765e0c1825edd11eba4ce9ddb230b2803ffd54b30564b2319b4450a45bfcc7c512fd9d566304e18dfee179b164cd2a7851f295189a4fabbe619872e1578571731f977d8aac2e7df57def5f8939853c3098c5742d7a12dc9eaa3750865d43025a365346ad477f899d9da6fd3751e76f0d241f885082ac5d35b109270c389e9e0f744e5d4c73f7ab7f760c1b04474ea3ec90cf4633e4d86cf0ec40bb4ed189702a339ec99c4e");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f10", Msg: "68656c6c6f", Custom: "0102030405060708", Mac: "9ba7e3f7e9f8ab1f4d79ed4b375d26073335c51403cc3f2d9de7a915fba7134ff102d42be6a1586c70c447723050f96f8f8e8b140c82c14a2d542c1359d79247824a9328e93f41539019442b130a5fc387d7f38f35f993263d14a15e8043b77010e9894cafd7fa3901f179fdc6725cf53b6f8297effba386414b94137b785ed96280dbf90546e609a186a6eb10818adf8325f75ca002161d218cf54f38a6331a118397775e341a28af691667053e0f2f8583acba818c302d1ec2e34e67f377d79da87b0f3ba675d48ae69a07083a6d8a77d41a2bc055ee9931e258faf6a5b43ba4dafa27e7985580");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f10", Msg: "676f6f64627965", Custom: "0102030405060708", Mac: "f00ecc5c55d937ab4b0e8acba15481979b9aee95c315d9cf580eb10ed05cc3cc1fc6202436ab6d06ade5aefb93b9d7ac604a6f9dfcf1eb4596d81ef26f0f7e2adf9ffd720f505fc7d5c358ce7649c4daa98af93e96bcb21c06e7a48f3a8a7da0daac5830c1fe01652c5e95416525ac20623a0d83b11869ece5fab6c0fb29c12db31a09d80d3a01c61ed62dafbc423bba12b3f886109689423d175b9ca6169d54c6cbb6d26e0ef693cf226785d06cb6503a753513f2c2d9723e0cf1049f229e018719c8c3804692eddf505a5de901cbea56ead36390dae6a5ed00cb2526d925f6f085d94be299e977");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f10", Msg: "", Custom: "", Mac: "eed85f46eb311f90e5bd22f2ba051b7912f01cc8a91880885b7e671bf9c37da5972e131eef9d0c235df735b7dfbe995ef87020cc830785573b7faeea3ea2d7fcd1a153c45b23fef80bfca0d4274167d59fd35970b7346201ba69535011b6ba0493fb612de7671cc1ea0b098f7fe2594bac9add3faa52af4d415ac1bd54fcb43f844805dbe4053cb115f8fdd240a9294469bb2a4ed82220e87a8f8964780bde37a63d338d9a36d10aa550b0fbf24f68075e854431c02c132a3ed245f07f3b78be39abbb144a62fcb7c4cdcaad3c6f4fa48d8451a2d1795a9196967864b12d44e426544a9c0ea4f14630ae23b9dc696ce507696ae1ddfcc00e");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f10", Msg: "68656c6c6f", Custom: "", Mac: "07e748b261a2a6cb0b1beb59434ded0dfb3e261e7feb128a6d4ed7b2a0f328052f0608af596cd745ee3483f375c713323155581c24a58dcd21c63cda31f7544067b116a3fd2de8c719a1f8690dcbe275472af61a60a1023453d824a18e0018e01660c75ab4434175e35ffe6bee8a7d5e55cdad4729132ba79ddaa4293fbb81c7a41e7d289c07b4b69e755933812b4a7aef961b664ea76f2ff75eaa3357f3cf41274bd8b97a489db167cdf3b00f7482dda7db32e1587db5417046484b145ec75fcb26f2df310d401403f9212c55d7ad85ebe5affc123ed2d342fa61026f8254b773f890458dc48188fd71af715378ca97a03fd5267a584af3");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f10", Msg: "676f6f64627965", Custom: "", Mac: "69a7cd47c75385f1a742ca3bfb5e22022f15e5f93e1f66160de3dcf648cc37c7e49bca6a099d1d3884112e551e19cc9995aaf8bddc9be5c062c6fc6355012c612e460817dd68d82914ae94ee4ecc877c0327fd396ac180ead919538bbad5e99ead85c9ac235ea9a255fc0ad55969eac003f0b3f13d6a77052fab975ef6e5705b10a9b99dc2c006637705943c812104603feed4a8535500c05d1760ddc2b3fc08517aa524de4af8e8bba6a2c0662d94dca9da2d7fdbf88e70ad5f5005b9bda7226a62a1e5f6e06b57bd47490b0cc6c526d4295c487344498d7bd49c823c5f3ce6f795ef749ef909bfd16a72945f55c4c36ce48ccaeb612437");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f10", Msg: "", Custom: "0102030405060708", Mac: "4f6cbb936643a4bebca4322adaae6bb9178180c052b6dd595ac08a7c98919990fee52a3edc87cd564ce01ef414333d9c83cca84cce852d62b2dab74d073ad16cc2e1b0765e0c1825edd11eba4ce9ddb230b2803ffd54b30564b2319b4450a45bfcc7c512fd9d566304e18dfee179b164cd2a7851f295189a4fabbe619872e1578571731f977d8aac2e7df57def5f8939853c3098c5742d7a12dc9eaa3750865d43025a365346ad477f899d9da6fd3751e76f0d241f885082ac5d35b109270c389e9e0f744e5d4c73f7ab7f760c1b04474ea3ec90cf4633e4d86cf0ec40bb4ed189702a339ec99c4ece8e83a4a284237cf68417e14af939f8");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f10", Msg: "68656c6c6f", Custom: "0102030405060708", Mac: "9ba7e3f7e9f8ab1f4d79ed4b375d26073335c51403cc3f2d9de7a915fba7134ff102d42be6a1586c70c447723050f96f8f8e8b140c82c14a2d542c1359d79247824a9328e93f41539019442b130a5fc387d7f38f35f993263d14a15e8043b77010e9894cafd7fa3901f179fdc6725cf53b6f8297effba386414b94137b785ed96280dbf90546e609a186a6eb10818adf8325f75ca002161d218cf54f38a6331a118397775e341a28af691667053e0f2f8583acba818c302d1ec2e34e67f377d79da87b0f3ba675d48ae69a07083a6d8a77d41a2bc055ee9931e258faf6a5b43ba4dafa27e79855802bb35b3493f596d0831ac7ca9e88c621");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f10", Msg: "676f6f64627965", Custom: "0102030405060708", Mac: "f00ecc5c55d937ab4b0e8acba15481979b9aee95c315d9cf580eb10ed05cc3cc1fc6202436ab6d06ade5aefb93b9d7ac604a6f9dfcf1eb4596d81ef26f0f7e2adf9ffd720f505fc7d5c358ce7649c4daa98af93e96bcb21c06e7a48f3a8a7da0daac5830c1fe01652c5e95416525ac20623a0d83b11869ece5fab6c0fb29c12db31a09d80d3a01c61ed62dafbc423bba12b3f886109689423d175b9ca6169d54c6cbb6d26e0ef693cf226785d06cb6503a753513f2c2d9723e0cf1049f229e018719c8c3804692eddf505a5de901cbea56ead36390dae6a5ed00cb2526d925f6f085d94be299e977e9deb5e4228c60aca37ece3f36285d2a");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f10", Msg: "", Custom: "", Mac: "eed85f46eb311f90e5bd22f2ba051b7912f01cc8a91880885b7e671bf9c37da5972e131eef9d0c235df735b7dfbe995ef87020cc830785573b7faeea3ea2d7fcd1a153c45b23fef80bfca0d4274167d59fd35970b7346201ba69535011b6ba0493fb612de7671cc1ea0b098f7fe2594bac9add3faa52af4d415ac1bd54fcb43f844805dbe4053cb115f8fdd240a9294469bb2a4ed82220e87a8f8964780bde37a63d338d9a36d10aa550b0fbf24f68075e854431c02c132a3ed245f07f3b78be39abbb144a62fcb7c4cdcaad3c6f4fa48d8451a2d1795a9196967864b12d44e426544a9c0ea4f14630ae23b9dc696ce507696ae1ddfcc00e9f32758c0505c7f507679a4ee42ad779");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f10", Msg: "68656c6c6f", Custom: "", Mac: "07e748b261a2a6cb0b1beb59434ded0dfb3e261e7feb128a6d4ed7b2a0f328052f0608af596cd745ee3483f375c713323155581c24a58dcd21c63cda31f7544067b116a3fd2de8c719a1f8690dcbe275472af61a60a1023453d824a18e0018e01660c75ab4434175e35ffe6bee8a7d5e55cdad4729132ba79ddaa4293fbb81c7a41e7d289c07b4b69e755933812b4a7aef961b664ea76f2ff75eaa3357f3cf41274bd8b97a489db167cdf3b00f7482dda7db32e1587db5417046484b145ec75fcb26f2df310d401403f9212c55d7ad85ebe5affc123ed2d342fa61026f8254b773f890458dc48188fd71af715378ca97a03fd5267a584af372796d087451c51325efca650716537b");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f10", Msg: "676f6f64627965", Custom: "", Mac: "69a7cd47c75385f1a742ca3bfb5e22022f15e5f93e1f66160de3dcf648cc37c7e49bca6a099d1d3884112e551e19cc9995aaf8bddc9be5c062c6fc6355012c612e460817dd68d82914ae94ee4ecc877c0327fd396ac180ead919538bbad5e99ead85c9ac235ea9a255fc0ad55969eac003f0b3f13d6a77052fab975ef6e5705b10a9b99dc2c006637705943c812104603feed4a8535500c05d1760ddc2b3fc08517aa524de4af8e8bba6a2c0662d94dca9da2d7fdbf88e70ad5f5005b9bda7226a62a1e5f6e06b57bd47490b0cc6c526d4295c487344498d7bd49c823c5f3ce6f795ef749ef909bfd16a72945f55c4c36ce48ccaeb61243712034460276255b5cb9b443c3e3a7843");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f10", Msg: "", Custom: "0102030405060708", Mac: "4f6cbb936643a4bebca4322adaae6bb9178180c052b6dd595ac08a7c98919990fee52a3edc87cd564ce01ef414333d9c83cca84cce852d62b2dab74d073ad16cc2e1b0765e0c1825edd11eba4ce9ddb230b2803ffd54b30564b2319b4450a45bfcc7c512fd9d566304e18dfee179b164cd2a7851f295189a4fabbe619872e1578571731f977d8aac2e7df57def5f8939853c3098c5742d7a12dc9eaa3750865d43025a365346ad477f899d9da6fd3751e76f0d241f885082ac5d35b109270c389e9e0f744e5d4c73f7ab7f760c1b04474ea3ec90cf4633e4d86cf0ec40bb4ed189702a339ec99c4ece8e83a4a284237cf68417e14af939f851fcd2697e9aa425fe5e54f6f0442353");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f10", Msg: "68656c6c6f", Custom: "0102030405060708", Mac: "9ba7e3f7e9f8ab1f4d79ed4b375d26073335c51403cc3f2d9de7a915fba7134ff102d42be6a1586c70c447723050f96f8f8e8b140c82c14a2d542c1359d79247824a9328e93f41539019442b130a5fc387d7f38f35f993263d14a15e8043b77010e9894cafd7fa3901f179fdc6725cf53b6f8297effba386414b94137b785ed96280dbf90546e609a186a6eb10818adf8325f75ca002161d218cf54f38a6331a118397775e341a28af691667053e0f2f8583acba818c302d1ec2e34e67f377d79da87b0f3ba675d48ae69a07083a6d8a77d41a2bc055ee9931e258faf6a5b43ba4dafa27e79855802bb35b3493f596d0831ac7ca9e88c621df4be242dc8d4a0f1441544ea25214e3");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f10", Msg: "676f6f64627965", Custom: "0102030405060708", Mac: "f00ecc5c55d937ab4b0e8acba15481979b9aee95c315d9cf580eb10ed05cc3cc1fc6202436ab6d06ade5aefb93b9d7ac604a6f9dfcf1eb4596d81ef26f0f7e2adf9ffd720f505fc7d5c358ce7649c4daa98af93e96bcb21c06e7a48f3a8a7da0daac5830c1fe01652c5e95416525ac20623a0d83b11869ece5fab6c0fb29c12db31a09d80d3a01c61ed62dafbc423bba12b3f886109689423d175b9ca6169d54c6cbb6d26e0ef693cf226785d06cb6503a753513f2c2d9723e0cf1049f229e018719c8c3804692eddf505a5de901cbea56ead36390dae6a5ed00cb2526d925f6f085d94be299e977e9deb5e4228c60aca37ece3f36285d2a83b30ee97d93a3183eabe5f3024c3fd1");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f20", Msg: "", Custom: "", Mac: "55");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f20", Msg: "68656c6c6f", Custom: "", Mac: "47");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f20", Msg: "676f6f64627965", Custom: "", Mac: "1b");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f20", Msg: "", Custom: "0102030405060708", Mac: "62");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f20", Msg: "68656c6c6f", Custom: "0102030405060708", Mac: "b7");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f20", Msg: "676f6f64627965", Custom: "0102030405060708", Mac: "b0");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f20", Msg: "", Custom: "", Mac: "55d862d2b9eefff8");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f20", Msg: "68656c6c6f", Custom: "", Mac: "47dea12022735718");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f20", Msg: "676f6f64627965", Custom: "", Mac: "1be3884fd1e9b4e3");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f20", Msg: "", Custom: "0102030405060708", Mac: "62c61528bcf4f66d");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f20", Msg: "68656c6c6f", Custom: "0102030405060708", Mac: "b72ee447b0912ece");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f20", Msg: "676f6f64627965", Custom: "0102030405060708", Mac: "b03e21098a8e314a");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f20", Msg: "", Custom: "", Mac: "55d862d2b9eefff8ded9ab36874432ca4aad02f692bbb0a1");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f20", Msg: "68656c6c6f", Custom: "", Mac: "47dea12022735718146c3aea6d7e1629fd4d7167b862a374");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f20", Msg: "676f6f64627965", Custom: "", Mac: "1be3884fd1e9b4e3002b1cc9c85982181c2b055ffda653bb");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f20", Msg: "", Custom: "0102030405060708", Mac: "62c61528bcf4f66de1fa628cba7d9a40b950e4b4ef312248");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f20", Msg: "68656c6c6f", Custom: "0102030405060708", Mac: "b72ee447b0912ecea797596e603eeaa87bf52e6e9eaaeb05");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f20", Msg: "676f6f64627965", Custom: "0102030405060708", Mac: "b03e21098a8e314a647074bdcdec1670fb4502526f88399c");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f20", Msg: "", Custom: "", Mac: "55d862d2b9eefff8ded9ab36874432ca4aad02f692bbb0a16fee9fad9ed8e5af440de955cf529d81");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f20", Msg: "68656c6c6f", Custom: "", Mac: "47dea12022735718146c3aea6d7e1629fd4d7167b862a374d09e6145eed159c6d79667e6738d6af4");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f20", Msg: "676f6f64627965", Custom: "", Mac: "1be3884fd1e9b4e3002b1cc9c85982181c2b055ffda653bb7402a8ebcce472eb59212abbed023267");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f20", Msg: "", Custom: "0102030405060708", Mac: "62c61528bcf4f66de1fa628cba7d9a40b950e4b4ef31224817a3542f22e4c7cb0070f68f4e3dc1ae");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f20", Msg: "68656c6c6f", Custom: "0102030405060708", Mac: "b72ee447b0912ecea797596e603eeaa87bf52e6e9eaaeb05af600bfc7fee8da948c5dbb0672c2a63");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f20", Msg: "676f6f64627965", Custom: "0102030405060708", Mac: "b03e21098a8e314a647074bdcdec1670fb4502526f88399ca2e529aea42ffadc27b9a9b65302a4b0");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f20", Msg: "", Custom: "", Mac: "55d862d2b9eefff8ded9ab36874432ca4aad02f692bbb0a16fee9fad9ed8e5af440de955cf529d81a43a4e60b17f50b5f70cc0668acb60df");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f20", Msg: "68656c6c6f", Custom: "", Mac: "47dea12022735718146c3aea6d7e1629fd4d7167b862a374d09e6145eed159c6d79667e6738d6af446330c2b19d70b0ca9593060f7fb8329");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f20", Msg: "676f6f64627965", Custom: "", Mac: "1be3884fd1e9b4e3002b1cc9c85982181c2b055ffda653bb7402a8ebcce472eb59212abbed023267819788a7561e719b7f9c111e376b7dfe");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f20", Msg: "", Custom: "0102030405060708", Mac: "62c61528bcf4f66de1fa628cba7d9a40b950e4b4ef31224817a3542f22e4c7cb0070f68f4e3dc1aee4d5c40d01cf096f66695d68c6e6fc32");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f20", Msg: "68656c6c6f", Custom: "0102030405060708", Mac: "b72ee447b0912ecea797596e603eeaa87bf52e6e9eaaeb05af600bfc7fee8da948c5dbb0672c2a632cafb659c619fb91daa6a56d2a8f48fa");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f20", Msg: "676f6f64627965", Custom: "0102030405060708", Mac: "b03e21098a8e314a647074bdcdec1670fb4502526f88399ca2e529aea42ffadc27b9a9b65302a4b0c5f564fc9b2c6f6d04f49c7d2a78d5ff");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f20", Msg: "", Custom: "", Mac: "55d862d2b9eefff8ded9ab36874432ca4aad02f692bbb0a16fee9fad9ed8e5af440de955cf529d81a43a4e60b17f50b5f70cc0668acb60dfff5f5ab82f39613a35df187219b9351c");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f20", Msg: "68656c6c6f", Custom: "", Mac: "47dea12022735718146c3aea6d7e1629fd4d7167b862a374d09e6145eed159c6d79667e6738d6af446330c2b19d70b0ca9593060f7fb83290ccc80ed3f6c65fa1443071228e467a8");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f20", Msg: "676f6f64627965", Custom: "", Mac: "1be3884fd1e9b4e3002b1cc9c85982181c2b055ffda653bb7402a8ebcce472eb59212abbed023267819788a7561e719b7f9c111e376b7dfe844a4da9536c63e0a203eb5a05a08c78");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f20", Msg: "", Custom: "0102030405060708", Mac: "62c61528bcf4f66de1fa628cba7d9a40b950e4b4ef31224817a3542f22e4c7cb0070f68f4e3dc1aee4d5c40d01cf096f66695d68c6e6fc329da4b6351179d546015074c74e4d1bca");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f20", Msg: "68656c6c6f", Custom: "0102030405060708", Mac: "b72ee447b0912ecea797596e603eeaa87bf52e6e9eaaeb05af600bfc7fee8da948c5dbb0672c2a632cafb659c619fb91daa6a56d2a8f48faf609852c1bd37653bf43161d91e8f1ae");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f20", Msg: "676f6f64627965", Custom: "0102030405060708", Mac: "b03e21098a8e314a647074bdcdec1670fb4502526f88399ca2e529aea42ffadc27b9a9b65302a4b0c5f564fc9b2c6f6d04f49c7d2a78d5ffc9485df9560d0c796d29660432468167");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f20", Msg: "", Custom: "", Mac: "55d862d2b9eefff8ded9ab36874432ca4aad02f692bbb0a16fee9fad9ed8e5af440de955cf529d81a43a4e60b17f50b5f70cc0668acb60dfff5f5ab82f39613a35df187219b9351cd26c62281e234841fe25db3e38292e0b");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f20", Msg: "68656c6c6f", Custom: "", Mac: "47dea12022735718146c3aea6d7e1629fd4d7167b862a374d09e6145eed159c6d79667e6738d6af446330c2b19d70b0ca9593060f7fb83290ccc80ed3f6c65fa1443071228e467a88fb1ea5ec3c41a960b83c25a52921ba3");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f20", Msg: "676f6f64627965", Custom: "", Mac: "1be3884fd1e9b4e3002b1cc9c85982181c2b055ffda653bb7402a8ebcce472eb59212abbed023267819788a7561e719b7f9c111e376b7dfe844a4da9536c63e0a203eb5a05a08c784fc290d1610ecaeebb5de6c001c93be4");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f20", Msg: "", Custom: "0102030405060708", Mac: "62c61528bcf4f66de1fa628cba7d9a40b950e4b4ef31224817a3542f22e4c7cb0070f68f4e3dc1aee4d5c40d01cf096f66695d68c6e6fc329da4b6351179d546015074c74e4d1bca9bfae4c040bd7b4b0f4e7ffc1a3fa3a5");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f20", Msg: "68656c6c6f", Custom: "0102030405060708", Mac: "b72ee447b0912ecea797596e603eeaa87bf52e6e9eaaeb05af600bfc7fee8da948c5dbb0672c2a632cafb659c619fb91daa6a56d2a8f48faf609852c1bd37653bf43161d91e8f1ae6c10c32a89f9d03ff1f40fbab0648c09");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f20", Msg: "676f6f64627965", Custom: "0102030405060708", Mac: "b03e21098a8e314a647074bdcdec1670fb4502526f88399ca2e529aea42ffadc27b9a9b65302a4b0c5f564fc9b2c6f6d04f49c7d2a78d5ffc9485df9560d0c796d296604324681673266dec4aa24d6f23af90ec3c8d5538c");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f20", Msg: "", Custom: "", Mac: "55d862d2b9eefff8ded9ab36874432ca4aad02f692bbb0a16fee9fad9ed8e5af440de955cf529d81a43a4e60b17f50b5f70cc0668acb60dfff5f5ab82f39613a35df187219b9351cd26c62281e234841fe25db3e38292e0ba97aa6803a6a41bd2b7e1dffa39eec27");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f20", Msg: "68656c6c6f", Custom: "", Mac: "47dea12022735718146c3aea6d7e1629fd4d7167b862a374d09e6145eed159c6d79667e6738d6af446330c2b19d70b0ca9593060f7fb83290ccc80ed3f6c65fa1443071228e467a88fb1ea5ec3c41a960b83c25a52921ba396c6c168e1b3a3efd02bfed8301798ee");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f20", Msg: "676f6f64627965", Custom: "", Mac: "1be3884fd1e9b4e3002b1cc9c85982181c2b055ffda653bb7402a8ebcce472eb59212abbed023267819788a7561e719b7f9c111e376b7dfe844a4da9536c63e0a203eb5a05a08c784fc290d1610ecaeebb5de6c001c93be4fbc7c09544721bcab18723a25c498635");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f20", Msg: "", Custom: "0102030405060708", Mac: "62c61528bcf4f66de1fa628cba7d9a40b950e4b4ef31224817a3542f22e4c7cb0070f68f4e3dc1aee4d5c40d01cf096f66695d68c6e6fc329da4b6351179d546015074c74e4d1bca9bfae4c040bd7b4b0f4e7ffc1a3fa3a52784d51fe3d36fce0e734c2038d02b74");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f20", Msg: "68656c6c6f", Custom: "0102030405060708", Mac: "b72ee447b0912ecea797596e603eeaa87bf52e6e9eaaeb05af600bfc7fee8da948c5dbb0672c2a632cafb659c619fb91daa6a56d2a8f48faf609852c1bd37653bf43161d91e8f1ae6c10c32a89f9d03ff1f40fbab0648c09d543129bdf78c670cef2d1f4c22c70db");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f20", Msg: "676f6f64627965", Custom: "0102030405060708", Mac: "b03e21098a8e314a647074bdcdec1670fb4502526f88399ca2e529aea42ffadc27b9a9b65302a4b0c5f564fc9b2c6f6d04f49c7d2a78d5ffc9485df9560d0c796d296604324681673266dec4aa24d6f23af90ec3c8d5538c5bcc42998c1f0f201a31b928671d7797");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f20", Msg: "", Custom: "", Mac: "55d862d2b9eefff8ded9ab36874432ca4aad02f692bbb0a16fee9fad9ed8e5af440de955cf529d81a43a4e60b17f50b5f70cc0668acb60dfff5f5ab82f39613a35df187219b9351cd26c62281e234841fe25db3e38292e0ba97aa6803a6a41bd2b7e1dffa39eec27ba85114bdfea41fdb42c7b59413397fd");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f20", Msg: "68656c6c6f", Custom: "", Mac: "47dea12022735718146c3aea6d7e1629fd4d7167b862a374d09e6145eed159c6d79667e6738d6af446330c2b19d70b0ca9593060f7fb83290ccc80ed3f6c65fa1443071228e467a88fb1ea5ec3c41a960b83c25a52921ba396c6c168e1b3a3efd02bfed8301798ee55d4ee41c7f94c365c3035792de7def3");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f20", Msg: "676f6f64627965", Custom: "", Mac: "1be3884fd1e9b4e3002b1cc9c85982181c2b055ffda653bb7402a8ebcce472eb59212abbed023267819788a7561e719b7f9c111e376b7dfe844a4da9536c63e0a203eb5a05a08c784fc290d1610ecaeebb5de6c001c93be4fbc7c09544721bcab18723a25c498635d62b24ed3fbc7d9116c26ba2bc77f3cc");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f20", Msg: "", Custom: "0102030405060708", Mac: "62c61528bcf4f66de1fa628cba7d9a40b950e4b4ef31224817a3542f22e4c7cb0070f68f4e3dc1aee4d5c40d01cf096f66695d68c6e6fc329da4b6351179d546015074c74e4d1bca9bfae4c040bd7b4b0f4e7ffc1a3fa3a52784d51fe3d36fce0e734c2038d02b746b01db8f618efd430de1e3b9a820255b");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f20", Msg: "68656c6c6f", Custom: "0102030405060708", Mac: "b72ee447b0912ecea797596e603eeaa87bf52e6e9eaaeb05af600bfc7fee8da948c5dbb0672c2a632cafb659c619fb91daa6a56d2a8f48faf609852c1bd37653bf43161d91e8f1ae6c10c32a89f9d03ff1f40fbab0648c09d543129bdf78c670cef2d1f4c22c70db041f1e097f9b3c586cb2ff71534f09cc");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f20", Msg: "676f6f64627965", Custom: "0102030405060708", Mac: "b03e21098a8e314a647074bdcdec1670fb4502526f88399ca2e529aea42ffadc27b9a9b65302a4b0c5f564fc9b2c6f6d04f49c7d2a78d5ffc9485df9560d0c796d296604324681673266dec4aa24d6f23af90ec3c8d5538c5bcc42998c1f0f201a31b928671d7797696a6e398cb9bf1c8fff282a0333a8df");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f20", Msg: "", Custom: "", Mac: "55d862d2b9eefff8ded9ab36874432ca4aad02f692bbb0a16fee9fad9ed8e5af440de955cf529d81a43a4e60b17f50b5f70cc0668acb60dfff5f5ab82f39613a35df187219b9351cd26c62281e234841fe25db3e38292e0ba97aa6803a6a41bd2b7e1dffa39eec27ba85114bdfea41fdb42c7b59413397fd2678a54f1f99f15a2da133f50884c938");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f20", Msg: "68656c6c6f", Custom: "", Mac: "47dea12022735718146c3aea6d7e1629fd4d7167b862a374d09e6145eed159c6d79667e6738d6af446330c2b19d70b0ca9593060f7fb83290ccc80ed3f6c65fa1443071228e467a88fb1ea5ec3c41a960b83c25a52921ba396c6c168e1b3a3efd02bfed8301798ee55d4ee41c7f94c365c3035792de7def395f7e6902cd1364e231c93d33d69ba29");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f20", Msg: "676f6f64627965", Custom: "", Mac: "1be3884fd1e9b4e3002b1cc9c85982181c2b055ffda653bb7402a8ebcce472eb59212abbed023267819788a7561e719b7f9c111e376b7dfe844a4da9536c63e0a203eb5a05a08c784fc290d1610ecaeebb5de6c001c93be4fbc7c09544721bcab18723a25c498635d62b24ed3fbc7d9116c26ba2bc77f3cc93431ccc18a8442c51c702f29ccd1e32");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f20", Msg: "", Custom: "0102030405060708", Mac: "62c61528bcf4f66de1fa628cba7d9a40b950e4b4ef31224817a3542f22e4c7cb0070f68f4e3dc1aee4d5c40d01cf096f66695d68c6e6fc329da4b6351179d546015074c74e4d1bca9bfae4c040bd7b4b0f4e7ffc1a3fa3a52784d51fe3d36fce0e734c2038d02b746b01db8f618efd430de1e3b9a820255be214be1095b3e9f8f66873055b4daed0");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f20", Msg: "68656c6c6f", Custom: "0102030405060708", Mac: "b72ee447b0912ecea797596e603eeaa87bf52e6e9eaaeb05af600bfc7fee8da948c5dbb0672c2a632cafb659c619fb91daa6a56d2a8f48faf609852c1bd37653bf43161d91e8f1ae6c10c32a89f9d03ff1f40fbab0648c09d543129bdf78c670cef2d1f4c22c70db041f1e097f9b3c586cb2ff71534f09cc2a4417a84d09d5a7098833bbf04a9e81");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f20", Msg: "676f6f64627965", Custom: "0102030405060708", Mac: "b03e21098a8e314a647074bdcdec1670fb4502526f88399ca2e529aea42ffadc27b9a9b65302a4b0c5f564fc9b2c6f6d04f49c7d2a78d5ffc9485df9560d0c796d296604324681673266dec4aa24d6f23af90ec3c8d5538c5bcc42998c1f0f201a31b928671d7797696a6e398cb9bf1c8fff282a0333a8dfedbe0015b14e5137fa4bb6f614b03779");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f20", Msg: "", Custom: "", Mac: "55d862d2b9eefff8ded9ab36874432ca4aad02f692bbb0a16fee9fad9ed8e5af440de955cf529d81a43a4e60b17f50b5f70cc0668acb60dfff5f5ab82f39613a35df187219b9351cd26c62281e234841fe25db3e38292e0ba97aa6803a6a41bd2b7e1dffa39eec27ba85114bdfea41fdb42c7b59413397fd2678a54f1f99f15a2da133f50884c938854868d86eda93b9d53a862851af42b5");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f20", Msg: "68656c6c6f", Custom: "", Mac: "47dea12022735718146c3aea6d7e1629fd4d7167b862a374d09e6145eed159c6d79667e6738d6af446330c2b19d70b0ca9593060f7fb83290ccc80ed3f6c65fa1443071228e467a88fb1ea5ec3c41a960b83c25a52921ba396c6c168e1b3a3efd02bfed8301798ee55d4ee41c7f94c365c3035792de7def395f7e6902cd1364e231c93d33d69ba29f092c653ef7b9ef5d6b5081c47a0e088");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f20", Msg: "676f6f64627965", Custom: "", Mac: "1be3884fd1e9b4e3002b1cc9c85982181c2b055ffda653bb7402a8ebcce472eb59212abbed023267819788a7561e719b7f9c111e376b7dfe844a4da9536c63e0a203eb5a05a08c784fc290d1610ecaeebb5de6c001c93be4fbc7c09544721bcab18723a25c498635d62b24ed3fbc7d9116c26ba2bc77f3cc93431ccc18a8442c51c702f29ccd1e32221a84182e7f2c8897bb106d41c0d160");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f20", Msg: "", Custom: "0102030405060708", Mac: "62c61528bcf4f66de1fa628cba7d9a40b950e4b4ef31224817a3542f22e4c7cb0070f68f4e3dc1aee4d5c40d01cf096f66695d68c6e6fc329da4b6351179d546015074c74e4d1bca9bfae4c040bd7b4b0f4e7ffc1a3fa3a52784d51fe3d36fce0e734c2038d02b746b01db8f618efd430de1e3b9a820255be214be1095b3e9f8f66873055b4daed095bb6465567b8e36d0cbf99a487c54f6");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f20", Msg: "68656c6c6f", Custom: "0102030405060708", Mac: "b72ee447b0912ecea797596e603eeaa87bf52e6e9eaaeb05af600bfc7fee8da948c5dbb0672c2a632cafb659c619fb91daa6a56d2a8f48faf609852c1bd37653bf43161d91e8f1ae6c10c32a89f9d03ff1f40fbab0648c09d543129bdf78c670cef2d1f4c22c70db041f1e097f9b3c586cb2ff71534f09cc2a4417a84d09d5a7098833bbf04a9e81fb80c425b87bb56b7dfd179c488d5c55");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f20", Msg: "676f6f64627965", Custom: "0102030405060708", Mac: "b03e21098a8e314a647074bdcdec1670fb4502526f88399ca2e529aea42ffadc27b9a9b65302a4b0c5f564fc9b2c6f6d04f49c7d2a78d5ffc9485df9560d0c796d296604324681673266dec4aa24d6f23af90ec3c8d5538c5bcc42998c1f0f201a31b928671d7797696a6e398cb9bf1c8fff282a0333a8dfedbe0015b14e5137fa4bb6f614b03779bd008cd19dfca2248dbb81fa531c24f7");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f20", Msg: "", Custom: "", Mac: "55d862d2b9eefff8ded9ab36874432ca4aad02f692bbb0a16fee9fad9ed8e5af440de955cf529d81a43a4e60b17f50b5f70cc0668acb60dfff5f5ab82f39613a35df187219b9351cd26c62281e234841fe25db3e38292e0ba97aa6803a6a41bd2b7e1dffa39eec27ba85114bdfea41fdb42c7b59413397fd2678a54f1f99f15a2da133f50884c938854868d86eda93b9d53a862851af42b53e7c77760d905c0a26d4500bbd243c16");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f20", Msg: "68656c6c6f", Custom: "", Mac: "47dea12022735718146c3aea6d7e1629fd4d7167b862a374d09e6145eed159c6d79667e6738d6af446330c2b19d70b0ca9593060f7fb83290ccc80ed3f6c65fa1443071228e467a88fb1ea5ec3c41a960b83c25a52921ba396c6c168e1b3a3efd02bfed8301798ee55d4ee41c7f94c365c3035792de7def395f7e6902cd1364e231c93d33d69ba29f092c653ef7b9ef5d6b5081c47a0e08845b1f2b4378896838f3f3de99a84d3d9");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f20", Msg: "676f6f64627965", Custom: "", Mac: "1be3884fd1e9b4e3002b1cc9c85982181c2b055ffda653bb7402a8ebcce472eb59212abbed023267819788a7561e719b7f9c111e376b7dfe844a4da9536c63e0a203eb5a05a08c784fc290d1610ecaeebb5de6c001c93be4fbc7c09544721bcab18723a25c498635d62b24ed3fbc7d9116c26ba2bc77f3cc93431ccc18a8442c51c702f29ccd1e32221a84182e7f2c8897bb106d41c0d160cc60d0346924a5b29843b1b264c9b1a4");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f20", Msg: "", Custom: "0102030405060708", Mac: "62c61528bcf4f66de1fa628cba7d9a40b950e4b4ef31224817a3542f22e4c7cb0070f68f4e3dc1aee4d5c40d01cf096f66695d68c6e6fc329da4b6351179d546015074c74e4d1bca9bfae4c040bd7b4b0f4e7ffc1a3fa3a52784d51fe3d36fce0e734c2038d02b746b01db8f618efd430de1e3b9a820255be214be1095b3e9f8f66873055b4daed095bb6465567b8e36d0cbf99a487c54f6793cd41ccc88ae2c0f63f25fd9121a9d");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f20", Msg: "68656c6c6f", Custom: "0102030405060708", Mac: "b72ee447b0912ecea797596e603eeaa87bf52e6e9eaaeb05af600bfc7fee8da948c5dbb0672c2a632cafb659c619fb91daa6a56d2a8f48faf609852c1bd37653bf43161d91e8f1ae6c10c32a89f9d03ff1f40fbab0648c09d543129bdf78c670cef2d1f4c22c70db041f1e097f9b3c586cb2ff71534f09cc2a4417a84d09d5a7098833bbf04a9e81fb80c425b87bb56b7dfd179c488d5c556568f618eb1e199f81d862e7d0ba4982");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f20", Msg: "676f6f64627965", Custom: "0102030405060708", Mac: "b03e21098a8e314a647074bdcdec1670fb4502526f88399ca2e529aea42ffadc27b9a9b65302a4b0c5f564fc9b2c6f6d04f49c7d2a78d5ffc9485df9560d0c796d296604324681673266dec4aa24d6f23af90ec3c8d5538c5bcc42998c1f0f201a31b928671d7797696a6e398cb9bf1c8fff282a0333a8dfedbe0015b14e5137fa4bb6f614b03779bd008cd19dfca2248dbb81fa531c24f74a6414be14e7d207717358ec8a4e3657");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f20", Msg: "", Custom: "", Mac: "55d862d2b9eefff8ded9ab36874432ca4aad02f692bbb0a16fee9fad9ed8e5af440de955cf529d81a43a4e60b17f50b5f70cc0668acb60dfff5f5ab82f39613a35df187219b9351cd26c62281e234841fe25db3e38292e0ba97aa6803a6a41bd2b7e1dffa39eec27ba85114bdfea41fdb42c7b59413397fd2678a54f1f99f15a2da133f50884c938854868d86eda93b9d53a862851af42b53e7c77760d905c0a26d4500bbd243c16d3b4ca3149ad1420d238c1442f4c1977");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f20", Msg: "68656c6c6f", Custom: "", Mac: "47dea12022735718146c3aea6d7e1629fd4d7167b862a374d09e6145eed159c6d79667e6738d6af446330c2b19d70b0ca9593060f7fb83290ccc80ed3f6c65fa1443071228e467a88fb1ea5ec3c41a960b83c25a52921ba396c6c168e1b3a3efd02bfed8301798ee55d4ee41c7f94c365c3035792de7def395f7e6902cd1364e231c93d33d69ba29f092c653ef7b9ef5d6b5081c47a0e08845b1f2b4378896838f3f3de99a84d3d991890c05aa6c86946ebe8b22aeda85cd");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f20", Msg: "676f6f64627965", Custom: "", Mac: "1be3884fd1e9b4e3002b1cc9c85982181c2b055ffda653bb7402a8ebcce472eb59212abbed023267819788a7561e719b7f9c111e376b7dfe844a4da9536c63e0a203eb5a05a08c784fc290d1610ecaeebb5de6c001c93be4fbc7c09544721bcab18723a25c498635d62b24ed3fbc7d9116c26ba2bc77f3cc93431ccc18a8442c51c702f29ccd1e32221a84182e7f2c8897bb106d41c0d160cc60d0346924a5b29843b1b264c9b1a482e7af953679866376c24261d394528d");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f20", Msg: "", Custom: "0102030405060708", Mac: "62c61528bcf4f66de1fa628cba7d9a40b950e4b4ef31224817a3542f22e4c7cb0070f68f4e3dc1aee4d5c40d01cf096f66695d68c6e6fc329da4b6351179d546015074c74e4d1bca9bfae4c040bd7b4b0f4e7ffc1a3fa3a52784d51fe3d36fce0e734c2038d02b746b01db8f618efd430de1e3b9a820255be214be1095b3e9f8f66873055b4daed095bb6465567b8e36d0cbf99a487c54f6793cd41ccc88ae2c0f63f25fd9121a9d6f1d84dbdab38ee24ade86fae5d34a79");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f20", Msg: "68656c6c6f", Custom: "0102030405060708", Mac: "b72ee447b0912ecea797596e603eeaa87bf52e6e9eaaeb05af600bfc7fee8da948c5dbb0672c2a632cafb659c619fb91daa6a56d2a8f48faf609852c1bd37653bf43161d91e8f1ae6c10c32a89f9d03ff1f40fbab0648c09d543129bdf78c670cef2d1f4c22c70db041f1e097f9b3c586cb2ff71534f09cc2a4417a84d09d5a7098833bbf04a9e81fb80c425b87bb56b7dfd179c488d5c556568f618eb1e199f81d862e7d0ba498252757f158111b78b610c250e56219148");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f20", Msg: "676f6f64627965", Custom: "0102030405060708", Mac: "b03e21098a8e314a647074bdcdec1670fb4502526f88399ca2e529aea42ffadc27b9a9b65302a4b0c5f564fc9b2c6f6d04f49c7d2a78d5ffc9485df9560d0c796d296604324681673266dec4aa24d6f23af90ec3c8d5538c5bcc42998c1f0f201a31b928671d7797696a6e398cb9bf1c8fff282a0333a8dfedbe0015b14e5137fa4bb6f614b03779bd008cd19dfca2248dbb81fa531c24f74a6414be14e7d207717358ec8a4e365750d4665516d344c057ef6912673a7fb2");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f20", Msg: "", Custom: "", Mac: "55d862d2b9eefff8ded9ab36874432ca4aad02f692bbb0a16fee9fad9ed8e5af440de955cf529d81a43a4e60b17f50b5f70cc0668acb60dfff5f5ab82f39613a35df187219b9351cd26c62281e234841fe25db3e38292e0ba97aa6803a6a41bd2b7e1dffa39eec27ba85114bdfea41fdb42c7b59413397fd2678a54f1f99f15a2da133f50884c938854868d86eda93b9d53a862851af42b53e7c77760d905c0a26d4500bbd243c16d3b4ca3149ad1420d238c1442f4c19778ef484774052e8f8b126c6e8e478a1d3");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f20", Msg: "68656c6c6f", Custom: "", Mac: "47dea12022735718146c3aea6d7e1629fd4d7167b862a374d09e6145eed159c6d79667e6738d6af446330c2b19d70b0ca9593060f7fb83290ccc80ed3f6c65fa1443071228e467a88fb1ea5ec3c41a960b83c25a52921ba396c6c168e1b3a3efd02bfed8301798ee55d4ee41c7f94c365c3035792de7def395f7e6902cd1364e231c93d33d69ba29f092c653ef7b9ef5d6b5081c47a0e08845b1f2b4378896838f3f3de99a84d3d991890c05aa6c86946ebe8b22aeda85cdb8e5e18a721e118c6947f4f3513683bd");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f20", Msg: "676f6f64627965", Custom: "", Mac: "1be3884fd1e9b4e3002b1cc9c85982181c2b055ffda653bb7402a8ebcce472eb59212abbed023267819788a7561e719b7f9c111e376b7dfe844a4da9536c63e0a203eb5a05a08c784fc290d1610ecaeebb5de6c001c93be4fbc7c09544721bcab18723a25c498635d62b24ed3fbc7d9116c26ba2bc77f3cc93431ccc18a8442c51c702f29ccd1e32221a84182e7f2c8897bb106d41c0d160cc60d0346924a5b29843b1b264c9b1a482e7af953679866376c24261d394528db8c99389cc84244c47f7f9c22a484757");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f20", Msg: "", Custom: "0102030405060708", Mac: "62c61528bcf4f66de1fa628cba7d9a40b950e4b4ef31224817a3542f22e4c7cb0070f68f4e3dc1aee4d5c40d01cf096f66695d68c6e6fc329da4b6351179d546015074c74e4d1bca9bfae4c040bd7b4b0f4e7ffc1a3fa3a52784d51fe3d36fce0e734c2038d02b746b01db8f618efd430de1e3b9a820255be214be1095b3e9f8f66873055b4daed095bb6465567b8e36d0cbf99a487c54f6793cd41ccc88ae2c0f63f25fd9121a9d6f1d84dbdab38ee24ade86fae5d34a795a21a600e31a045a34e00e9d7672d0aa");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f20", Msg: "68656c6c6f", Custom: "0102030405060708", Mac: "b72ee447b0912ecea797596e603eeaa87bf52e6e9eaaeb05af600bfc7fee8da948c5dbb0672c2a632cafb659c619fb91daa6a56d2a8f48faf609852c1bd37653bf43161d91e8f1ae6c10c32a89f9d03ff1f40fbab0648c09d543129bdf78c670cef2d1f4c22c70db041f1e097f9b3c586cb2ff71534f09cc2a4417a84d09d5a7098833bbf04a9e81fb80c425b87bb56b7dfd179c488d5c556568f618eb1e199f81d862e7d0ba498252757f158111b78b610c250e56219148530c52237e431563ccd13e5ffe8dd95d");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f20", Msg: "676f6f64627965", Custom: "0102030405060708", Mac: "b03e21098a8e314a647074bdcdec1670fb4502526f88399ca2e529aea42ffadc27b9a9b65302a4b0c5f564fc9b2c6f6d04f49c7d2a78d5ffc9485df9560d0c796d296604324681673266dec4aa24d6f23af90ec3c8d5538c5bcc42998c1f0f201a31b928671d7797696a6e398cb9bf1c8fff282a0333a8dfedbe0015b14e5137fa4bb6f614b03779bd008cd19dfca2248dbb81fa531c24f74a6414be14e7d207717358ec8a4e365750d4665516d344c057ef6912673a7fb2deb1fc165c1d44733b605a5650034c3d");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f20", Msg: "", Custom: "", Mac: "55d862d2b9eefff8ded9ab36874432ca4aad02f692bbb0a16fee9fad9ed8e5af440de955cf529d81a43a4e60b17f50b5f70cc0668acb60dfff5f5ab82f39613a35df187219b9351cd26c62281e234841fe25db3e38292e0ba97aa6803a6a41bd2b7e1dffa39eec27ba85114bdfea41fdb42c7b59413397fd2678a54f1f99f15a2da133f50884c938854868d86eda93b9d53a862851af42b53e7c77760d905c0a26d4500bbd243c16d3b4ca3149ad1420d238c1442f4c19778ef484774052e8f8b126c6e8e478a1d34d90e5fbf467713c32e37ee6f5ff6319");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f20", Msg: "68656c6c6f", Custom: "", Mac: "47dea12022735718146c3aea6d7e1629fd4d7167b862a374d09e6145eed159c6d79667e6738d6af446330c2b19d70b0ca9593060f7fb83290ccc80ed3f6c65fa1443071228e467a88fb1ea5ec3c41a960b83c25a52921ba396c6c168e1b3a3efd02bfed8301798ee55d4ee41c7f94c365c3035792de7def395f7e6902cd1364e231c93d33d69ba29f092c653ef7b9ef5d6b5081c47a0e08845b1f2b4378896838f3f3de99a84d3d991890c05aa6c86946ebe8b22aeda85cdb8e5e18a721e118c6947f4f3513683bd279df13d977fccd1d2c37cba14f6fc70");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f20", Msg: "676f6f64627965", Custom: "", Mac: "1be3884fd1e9b4e3002b1cc9c85982181c2b055ffda653bb7402a8ebcce472eb59212abbed023267819788a7561e719b7f9c111e376b7dfe844a4da9536c63e0a203eb5a05a08c784fc290d1610ecaeebb5de6c001c93be4fbc7c09544721bcab18723a25c498635d62b24ed3fbc7d9116c26ba2bc77f3cc93431ccc18a8442c51c702f29ccd1e32221a84182e7f2c8897bb106d41c0d160cc60d0346924a5b29843b1b264c9b1a482e7af953679866376c24261d394528db8c99389cc84244c47f7f9c22a484757c858955f016156ea095d8bdf25afe1b1");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f20", Msg: "", Custom: "0102030405060708", Mac: "62c61528bcf4f66de1fa628cba7d9a40b950e4b4ef31224817a3542f22e4c7cb0070f68f4e3dc1aee4d5c40d01cf096f66695d68c6e6fc329da4b6351179d546015074c74e4d1bca9bfae4c040bd7b4b0f4e7ffc1a3fa3a52784d51fe3d36fce0e734c2038d02b746b01db8f618efd430de1e3b9a820255be214be1095b3e9f8f66873055b4daed095bb6465567b8e36d0cbf99a487c54f6793cd41ccc88ae2c0f63f25fd9121a9d6f1d84dbdab38ee24ade86fae5d34a795a21a600e31a045a34e00e9d7672d0aa234e7d9f751839d4f7275183ed3df700");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f20", Msg: "68656c6c6f", Custom: "0102030405060708", Mac: "b72ee447b0912ecea797596e603eeaa87bf52e6e9eaaeb05af600bfc7fee8da948c5dbb0672c2a632cafb659c619fb91daa6a56d2a8f48faf609852c1bd37653bf43161d91e8f1ae6c10c32a89f9d03ff1f40fbab0648c09d543129bdf78c670cef2d1f4c22c70db041f1e097f9b3c586cb2ff71534f09cc2a4417a84d09d5a7098833bbf04a9e81fb80c425b87bb56b7dfd179c488d5c556568f618eb1e199f81d862e7d0ba498252757f158111b78b610c250e56219148530c52237e431563ccd13e5ffe8dd95dee0705a678f86961b9f10eb7616430f2");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f20", Msg: "676f6f64627965", Custom: "0102030405060708", Mac: "b03e21098a8e314a647074bdcdec1670fb4502526f88399ca2e529aea42ffadc27b9a9b65302a4b0c5f564fc9b2c6f6d04f49c7d2a78d5ffc9485df9560d0c796d296604324681673266dec4aa24d6f23af90ec3c8d5538c5bcc42998c1f0f201a31b928671d7797696a6e398cb9bf1c8fff282a0333a8dfedbe0015b14e5137fa4bb6f614b03779bd008cd19dfca2248dbb81fa531c24f74a6414be14e7d207717358ec8a4e365750d4665516d344c057ef6912673a7fb2deb1fc165c1d44733b605a5650034c3dd562784ee0e68976e88aca784078529c");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f20", Msg: "", Custom: "", Mac: "55d862d2b9eefff8ded9ab36874432ca4aad02f692bbb0a16fee9fad9ed8e5af440de955cf529d81a43a4e60b17f50b5f70cc0668acb60dfff5f5ab82f39613a35df187219b9351cd26c62281e234841fe25db3e38292e0ba97aa6803a6a41bd2b7e1dffa39eec27ba85114bdfea41fdb42c7b59413397fd2678a54f1f99f15a2da133f50884c938854868d86eda93b9d53a862851af42b53e7c77760d905c0a26d4500bbd243c16d3b4ca3149ad1420d238c1442f4c19778ef484774052e8f8b126c6e8e478a1d34d90e5fbf467713c32e37ee6f5ff6319747bfdeb89e05d6a855a6ac734d06e8b");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f20", Msg: "68656c6c6f", Custom: "", Mac: "47dea12022735718146c3aea6d7e1629fd4d7167b862a374d09e6145eed159c6d79667e6738d6af446330c2b19d70b0ca9593060f7fb83290ccc80ed3f6c65fa1443071228e467a88fb1ea5ec3c41a960b83c25a52921ba396c6c168e1b3a3efd02bfed8301798ee55d4ee41c7f94c365c3035792de7def395f7e6902cd1364e231c93d33d69ba29f092c653ef7b9ef5d6b5081c47a0e08845b1f2b4378896838f3f3de99a84d3d991890c05aa6c86946ebe8b22aeda85cdb8e5e18a721e118c6947f4f3513683bd279df13d977fccd1d2c37cba14f6fc70861ef558428cbe31b8f0ce3e6391d1ab");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f20", Msg: "676f6f64627965", Custom: "", Mac: "1be3884fd1e9b4e3002b1cc9c85982181c2b055ffda653bb7402a8ebcce472eb59212abbed023267819788a7561e719b7f9c111e376b7dfe844a4da9536c63e0a203eb5a05a08c784fc290d1610ecaeebb5de6c001c93be4fbc7c09544721bcab18723a25c498635d62b24ed3fbc7d9116c26ba2bc77f3cc93431ccc18a8442c51c702f29ccd1e32221a84182e7f2c8897bb106d41c0d160cc60d0346924a5b29843b1b264c9b1a482e7af953679866376c24261d394528db8c99389cc84244c47f7f9c22a484757c858955f016156ea095d8bdf25afe1b1bbe54f31099c422e1907e042370ffe45");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f20", Msg: "", Custom: "0102030405060708", Mac: "62c61528bcf4f66de1fa628cba7d9a40b950e4b4ef31224817a3542f22e4c7cb0070f68f4e3dc1aee4d5c40d01cf096f66695d68c6e6fc329da4b6351179d546015074c74e4d1bca9bfae4c040bd7b4b0f4e7ffc1a3fa3a52784d51fe3d36fce0e734c2038d02b746b01db8f618efd430de1e3b9a820255be214be1095b3e9f8f66873055b4daed095bb6465567b8e36d0cbf99a487c54f6793cd41ccc88ae2c0f63f25fd9121a9d6f1d84dbdab38ee24ade86fae5d34a795a21a600e31a045a34e00e9d7672d0aa234e7d9f751839d4f7275183ed3df700223182117adc60e3f91ac8dba1655ccf");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f20", Msg: "68656c6c6f", Custom: "0102030405060708", Mac: "b72ee447b0912ecea797596e603eeaa87bf52e6e9eaaeb05af600bfc7fee8da948c5dbb0672c2a632cafb659c619fb91daa6a56d2a8f48faf609852c1bd37653bf43161d91e8f1ae6c10c32a89f9d03ff1f40fbab0648c09d543129bdf78c670cef2d1f4c22c70db041f1e097f9b3c586cb2ff71534f09cc2a4417a84d09d5a7098833bbf04a9e81fb80c425b87bb56b7dfd179c488d5c556568f618eb1e199f81d862e7d0ba498252757f158111b78b610c250e56219148530c52237e431563ccd13e5ffe8dd95dee0705a678f86961b9f10eb7616430f2200f2fca7cb8c83794f771ab1f1d202a");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f20", Msg: "676f6f64627965", Custom: "0102030405060708", Mac: "b03e21098a8e314a647074bdcdec1670fb4502526f88399ca2e529aea42ffadc27b9a9b65302a4b0c5f564fc9b2c6f6d04f49c7d2a78d5ffc9485df9560d0c796d296604324681673266dec4aa24d6f23af90ec3c8d5538c5bcc42998c1f0f201a31b928671d7797696a6e398cb9bf1c8fff282a0333a8dfedbe0015b14e5137fa4bb6f614b03779bd008cd19dfca2248dbb81fa531c24f74a6414be14e7d207717358ec8a4e365750d4665516d344c057ef6912673a7fb2deb1fc165c1d44733b605a5650034c3dd562784ee0e68976e88aca784078529cc73156757f2f818b3c14a68974f6f273");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f20", Msg: "", Custom: "", Mac: "55d862d2b9eefff8ded9ab36874432ca4aad02f692bbb0a16fee9fad9ed8e5af440de955cf529d81a43a4e60b17f50b5f70cc0668acb60dfff5f5ab82f39613a35df187219b9351cd26c62281e234841fe25db3e38292e0ba97aa6803a6a41bd2b7e1dffa39eec27ba85114bdfea41fdb42c7b59413397fd2678a54f1f99f15a2da133f50884c938854868d86eda93b9d53a862851af42b53e7c77760d905c0a26d4500bbd243c16d3b4ca3149ad1420d238c1442f4c19778ef484774052e8f8b126c6e8e478a1d34d90e5fbf467713c32e37ee6f5ff6319747bfdeb89e05d6a855a6ac734d06e8b4208679935c3951ce359b88c3672d029");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f20", Msg: "68656c6c6f", Custom: "", Mac: "47dea12022735718146c3aea6d7e1629fd4d7167b862a374d09e6145eed159c6d79667e6738d6af446330c2b19d70b0ca9593060f7fb83290ccc80ed3f6c65fa1443071228e467a88fb1ea5ec3c41a960b83c25a52921ba396c6c168e1b3a3efd02bfed8301798ee55d4ee41c7f94c365c3035792de7def395f7e6902cd1364e231c93d33d69ba29f092c653ef7b9ef5d6b5081c47a0e08845b1f2b4378896838f3f3de99a84d3d991890c05aa6c86946ebe8b22aeda85cdb8e5e18a721e118c6947f4f3513683bd279df13d977fccd1d2c37cba14f6fc70861ef558428cbe31b8f0ce3e6391d1ab3b1c999d92a13cec237371c038ec7cd0");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f20", Msg: "676f6f64627965", Custom: "", Mac: "1be3884fd1e9b4e3002b1cc9c85982181c2b055ffda653bb7402a8ebcce472eb59212abbed023267819788a7561e719b7f9c111e376b7dfe844a4da9536c63e0a203eb5a05a08c784fc290d1610ecaeebb5de6c001c93be4fbc7c09544721bcab18723a25c498635d62b24ed3fbc7d9116c26ba2bc77f3cc93431ccc18a8442c51c702f29ccd1e32221a84182e7f2c8897bb106d41c0d160cc60d0346924a5b29843b1b264c9b1a482e7af953679866376c24261d394528db8c99389cc84244c47f7f9c22a484757c858955f016156ea095d8bdf25afe1b1bbe54f31099c422e1907e042370ffe45f4efe05b229ee36c59b16c446840200f");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f20", Msg: "", Custom: "0102030405060708", Mac: "62c61528bcf4f66de1fa628cba7d9a40b950e4b4ef31224817a3542f22e4c7cb0070f68f4e3dc1aee4d5c40d01cf096f66695d68c6e6fc329da4b6351179d546015074c74e4d1bca9bfae4c040bd7b4b0f4e7ffc1a3fa3a52784d51fe3d36fce0e734c2038d02b746b01db8f618efd430de1e3b9a820255be214be1095b3e9f8f66873055b4daed095bb6465567b8e36d0cbf99a487c54f6793cd41ccc88ae2c0f63f25fd9121a9d6f1d84dbdab38ee24ade86fae5d34a795a21a600e31a045a34e00e9d7672d0aa234e7d9f751839d4f7275183ed3df700223182117adc60e3f91ac8dba1655ccfbec73bccd092c8857bd0eb54227944b5");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f20", Msg: "68656c6c6f", Custom: "0102030405060708", Mac: "b72ee447b0912ecea797596e603eeaa87bf52e6e9eaaeb05af600bfc7fee8da948c5dbb0672c2a632cafb659c619fb91daa6a56d2a8f48faf609852c1bd37653bf43161d91e8f1ae6c10c32a89f9d03ff1f40fbab0648c09d543129bdf78c670cef2d1f4c22c70db041f1e097f9b3c586cb2ff71534f09cc2a4417a84d09d5a7098833bbf04a9e81fb80c425b87bb56b7dfd179c488d5c556568f618eb1e199f81d862e7d0ba498252757f158111b78b610c250e56219148530c52237e431563ccd13e5ffe8dd95dee0705a678f86961b9f10eb7616430f2200f2fca7cb8c83794f771ab1f1d202a965b57c76d3f233155ae10dc32261c6d");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f20", Msg: "676f6f64627965", Custom: "0102030405060708", Mac: "b03e21098a8e314a647074bdcdec1670fb4502526f88399ca2e529aea42ffadc27b9a9b65302a4b0c5f564fc9b2c6f6d04f49c7d2a78d5ffc9485df9560d0c796d296604324681673266dec4aa24d6f23af90ec3c8d5538c5bcc42998c1f0f201a31b928671d7797696a6e398cb9bf1c8fff282a0333a8dfedbe0015b14e5137fa4bb6f614b03779bd008cd19dfca2248dbb81fa531c24f74a6414be14e7d207717358ec8a4e365750d4665516d344c057ef6912673a7fb2deb1fc165c1d44733b605a5650034c3dd562784ee0e68976e88aca784078529cc73156757f2f818b3c14a68974f6f273200fa950a6d99f868a2c0da86d5d86df");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f20", Msg: "", Custom: "", Mac: "55d862d2b9eefff8ded9ab36874432ca4aad02f692bbb0a16fee9fad9ed8e5af440de955cf529d81a43a4e60b17f50b5f70cc0668acb60dfff5f5ab82f39613a35df187219b9351cd26c62281e234841fe25db3e38292e0ba97aa6803a6a41bd2b7e1dffa39eec27ba85114bdfea41fdb42c7b59413397fd2678a54f1f99f15a2da133f50884c938854868d86eda93b9d53a862851af42b53e7c77760d905c0a26d4500bbd243c16d3b4ca3149ad1420d238c1442f4c19778ef484774052e8f8b126c6e8e478a1d34d90e5fbf467713c32e37ee6f5ff6319747bfdeb89e05d6a855a6ac734d06e8b4208679935c3951ce359b88c3672d029b87a5b650d1e8af0d8cbdc83abf00e8e");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f20", Msg: "68656c6c6f", Custom: "", Mac: "47dea12022735718146c3aea6d7e1629fd4d7167b862a374d09e6145eed159c6d79667e6738d6af446330c2b19d70b0ca9593060f7fb83290ccc80ed3f6c65fa1443071228e467a88fb1ea5ec3c41a960b83c25a52921ba396c6c168e1b3a3efd02bfed8301798ee55d4ee41c7f94c365c3035792de7def395f7e6902cd1364e231c93d33d69ba29f092c653ef7b9ef5d6b5081c47a0e08845b1f2b4378896838f3f3de99a84d3d991890c05aa6c86946ebe8b22aeda85cdb8e5e18a721e118c6947f4f3513683bd279df13d977fccd1d2c37cba14f6fc70861ef558428cbe31b8f0ce3e6391d1ab3b1c999d92a13cec237371c038ec7cd08322016a1fa7dd1aff871712bddaab53");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f20", Msg: "676f6f64627965", Custom: "", Mac: "1be3884fd1e9b4e3002b1cc9c85982181c2b055ffda653bb7402a8ebcce472eb59212abbed023267819788a7561e719b7f9c111e376b7dfe844a4da9536c63e0a203eb5a05a08c784fc290d1610ecaeebb5de6c001c93be4fbc7c09544721bcab18723a25c498635d62b24ed3fbc7d9116c26ba2bc77f3cc93431ccc18a8442c51c702f29ccd1e32221a84182e7f2c8897bb106d41c0d160cc60d0346924a5b29843b1b264c9b1a482e7af953679866376c24261d394528db8c99389cc84244c47f7f9c22a484757c858955f016156ea095d8bdf25afe1b1bbe54f31099c422e1907e042370ffe45f4efe05b229ee36c59b16c446840200f93baf38a781eace937e8520e670ae181");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f20", Msg: "", Custom: "0102030405060708", Mac: "62c61528bcf4f66de1fa628cba7d9a40b950e4b4ef31224817a3542f22e4c7cb0070f68f4e3dc1aee4d5c40d01cf096f66695d68c6e6fc329da4b6351179d546015074c74e4d1bca9bfae4c040bd7b4b0f4e7ffc1a3fa3a52784d51fe3d36fce0e734c2038d02b746b01db8f618efd430de1e3b9a820255be214be1095b3e9f8f66873055b4daed095bb6465567b8e36d0cbf99a487c54f6793cd41ccc88ae2c0f63f25fd9121a9d6f1d84dbdab38ee24ade86fae5d34a795a21a600e31a045a34e00e9d7672d0aa234e7d9f751839d4f7275183ed3df700223182117adc60e3f91ac8dba1655ccfbec73bccd092c8857bd0eb54227944b53df2ba44d040744ad70961dbdb3323b7");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f20", Msg: "68656c6c6f", Custom: "0102030405060708", Mac: "b72ee447b0912ecea797596e603eeaa87bf52e6e9eaaeb05af600bfc7fee8da948c5dbb0672c2a632cafb659c619fb91daa6a56d2a8f48faf609852c1bd37653bf43161d91e8f1ae6c10c32a89f9d03ff1f40fbab0648c09d543129bdf78c670cef2d1f4c22c70db041f1e097f9b3c586cb2ff71534f09cc2a4417a84d09d5a7098833bbf04a9e81fb80c425b87bb56b7dfd179c488d5c556568f618eb1e199f81d862e7d0ba498252757f158111b78b610c250e56219148530c52237e431563ccd13e5ffe8dd95dee0705a678f86961b9f10eb7616430f2200f2fca7cb8c83794f771ab1f1d202a965b57c76d3f233155ae10dc32261c6d1451af550d48b3a3b4239a94f0689f07");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f20", Msg: "676f6f64627965", Custom: "0102030405060708", Mac: "b03e21098a8e314a647074bdcdec1670fb4502526f88399ca2e529aea42ffadc27b9a9b65302a4b0c5f564fc9b2c6f6d04f49c7d2a78d5ffc9485df9560d0c796d296604324681673266dec4aa24d6f23af90ec3c8d5538c5bcc42998c1f0f201a31b928671d7797696a6e398cb9bf1c8fff282a0333a8dfedbe0015b14e5137fa4bb6f614b03779bd008cd19dfca2248dbb81fa531c24f74a6414be14e7d207717358ec8a4e365750d4665516d344c057ef6912673a7fb2deb1fc165c1d44733b605a5650034c3dd562784ee0e68976e88aca784078529cc73156757f2f818b3c14a68974f6f273200fa950a6d99f868a2c0da86d5d86df0e246a9ec04faf2f7a14fc56ba3884f0");

                // NIST test vectors
                // From https://csrc.nist.gov/CSRC/media/Projects/Cryptographic-Standards-and-Guidelines/documents/examples/KMACXOF_samples.pdf
                yield return new NistKmacTestVector(Name: "Sample #1", Key:"404142434445464748494A4B4C4D4E4F505152535455565758595A5B5C5D5E5F", Msg: "00010203", CustomText: "", Mac: "CD83740BBD92CCC8CF032B1481A0F4460E7CA9DD12B08A0C4031178BACD6EC35");
                yield return new NistKmacTestVector(Name: "Sample #2", Key:"404142434445464748494A4B4C4D4E4F505152535455565758595A5B5C5D5E5F", Msg: "00010203", CustomText: "My Tagged Application", Mac: "31A44527B4ED9F5C6101D11DE6D26F0620AA5C341DEF41299657FE9DF1A3B16C");
                yield return new NistKmacTestVector(Name: "Sample #3", Key:"404142434445464748494A4B4C4D4E4F505152535455565758595A5B5C5D5E5F", Msg: "000102030405060708090A0B0C0D0E0F101112131415161718191A1B1C1D1E1F202122232425262728292A2B2C2D2E2F303132333435363738393A3B3C3D3E3F404142434445464748494A4B4C4D4E4F505152535455565758595A5B5C5D5E5F606162636465666768696A6B6C6D6E6F707172737475767778797A7B7C7D7E7F808182838485868788898A8B8C8D8E8F909192939495969798999A9B9C9D9E9FA0A1A2A3A4A5A6A7A8A9AAABACADAEAFB0B1B2B3B4B5B6B7B8B9BABBBCBDBEBFC0C1C2C3C4C5C6C7", CustomText: "My Tagged Application", Mac: "47026C7CD793084AA0283C253EF658490C0DB61438B8326FE9BDDF281B83AE0F");
            }
        }
    }
}
