/* 
 * IRiver ifp tuner manipulation functions.
 * $Id: tuner.c,v 1.5.2.1 2005/02/19 17:18:03 oakhamg Exp $
 *
 * Copyright (C) Geoff Oakham, 2004; <oakhamg@users.sourceforge.net>
 */

static const char rcsid[] = "$Id: ";

#include <stdio.h>
#include <string.h>

#include "ifp.h"
#include "ifp_os.h"

#define FREQ_WIDTH 6
#define CALL_WIDTH IFP_TUNER_LABEL
#define ROW_WIDTH 12

static int get_station_helper(uint8_t * b, char * callsign, int * freq) {
	uint8_t * bm = b + CALL_WIDTH;
	uint8_t * bf = b;

	if (bf[3] != '.') {
		ifp_err("data consistancy problem %d should be %d (internal error)",
			(int)bf[3], (int)'.');
		return -1;
	}
	if (	bf[0] > 9 ||
		bf[1] > 9 ||
		bf[2] > 9 ||
		bf[4] > 9 ||
		bf[5] > 9 )
	{
		ifp_err("data consistancy problem (internal error)");
		return -1;
	}
	*freq = 
		bf[0]*10000 +
		bf[1]*1000 +
		bf[2]*100 +
		bf[4]*10 +
		bf[5]*1;
	if (*freq > IFP_FREQ_MAX || *freq < IFP_FREQ_MIN) {
		//ifp_wrn("frequency is out of range: %d.%02dMHz",
		//	*freq/100, *freq%100);
		printf("warning: frequency is out of range: %d.%02dMHz\n",
			*freq/100, *freq%100);
	}

	callsign[CALL_WIDTH] = '\0';
	memcpy(callsign, bm, CALL_WIDTH);

	return 0;
}
/** returns IFP_ERR_BAD_FREQUENCY */
static int set_station_helper(uint8_t * b, const char * callsign, int freq) {
	char tmp[FREQ_WIDTH+1];
	uint8_t * bm = b + CALL_WIDTH;
	uint8_t * bf = b;

	int j,i;

	if (freq > IFP_FREQ_MAX || freq < IFP_FREQ_MIN) {
		//ifp_dbg("frequency out of range");
		return IFP_ERR_BAD_FREQUENCY;
	}
	i = snprintf(tmp, FREQ_WIDTH+1, "%03d.%02d", freq/100, freq%100);
	if (i != FREQ_WIDTH || tmp[3] != '.') {
		ifp_err("(internal) formatting error for freq=%d. i=%d, tmp[3]=%d",
			freq, i, (int)tmp[3]);
	}
	for (j=0; j!=FREQ_WIDTH; j++) {
		if (tmp[j] != '.') {
			tmp[j] -= '0';
		}
	}
	memcpy(bf, tmp, FREQ_WIDTH);

	memset(bm, '\0', CALL_WIDTH);
	strncpy(bm, callsign, CALL_WIDTH);

	return 0;
}

/** \brief Reads station #n from the binary datablock into more
 * useful forms.
 *
 * b is the block of data loaded using ::ifp_get_tuner_presets,
 * n is a number between 0 and ::IFP_PRESET_TOTAL-1,
 * callsign is a pointer to a buffer of at least ::IFP_TUNER_LABEL+1 bytes,
 * freq is a pointer to an integer, where the station's frequency will be
 * saved.  The frequency units are 10*kHz (0.01MHz) and should range from
 * ::IFP_FREQ_MIN to ::IFP_FREQ_MAX.
 *
 * Apon successful return, 'callsign' will be loaded the station's label
 * as a zero-terminated string, and 'freq' will be the station's
 * frequency in units of 10*kHz (0.01MHz).
 */
int ifp_get_station(int n, void * b, char * callsign, int * freq) {
	if (n >= IFP_PRESET_TOTAL || n < 0) {
		ifp_err("n=%d is out of range",n);
		return -EINVAL;
	}
	return get_station_helper((uint8_t *)b + n*12, callsign, freq);
}
IFP_EXPORT(ifp_get_station);

/** \brief Sets station #n to 'freq' and 'callsign'.
 *
 * b is the block of data loaded using ::ifp_get_tuner_presets,
 * n is a number between 0 and ::IFP_PRESET_TOTAL-1,
 * callsign is a zero-terminated string (but only the first 6 characters will be
 * used),
 * freq is the FM frequency in units of Hz*10^4 (or 0.01 MHz); freq should
 * range from ::IFP_FREQ_MIN to ::IFP_FREQ_MAX.
 *
 * eg:
 * 	//sets station #4 to 94.9MHz and labeled 'bbc1'
 * 	i = ::ifp_set_station(3, p, "bbc1", 9490);
 *
 * 
 *
 * Notes:
 * 	-::ifp_set_tuner_presets must be called before any changes to take affect.
 *
 *	-Although this interface suggests frequency accuracy of 0.01MHz is
 *	possible, I've found some players only support increments 0.05MHz.
 *	Setting a frequency to a more accurate value is *not* an error: the
 *	hardware will silently truncate to a nearby acceptable value.
 *
 *	-User interfaces can provide feedback to the user of the above "effect"
 *	by saving and reloading the preset data after every change.  Any
 *	trunction by the hardware will be immediately obvious.
 *
 * Returns ::IFP_ERR_BAD_FREQUENCY if the frequency is out of range.
 */
int ifp_set_station(int n, void * b, const char * callsign, int freq) {
	if (n >= IFP_PRESET_TOTAL || n < 0) {
		ifp_err("n=%d is out of range",n);
		return -EINVAL;
	}
	if (callsign == NULL) {
		callsign = "";
	}
	return set_station_helper((uint8_t *)b + n*12, callsign, freq);
}
IFP_EXPORT(ifp_set_station);


